Option Explicit

 ' Description:
 '  This module contains constants and wrappers related
 '  to the AS/400 APPC service APIs. See ZZPAPW2.BAS for
 '  action APIs.

 ' Global Constants:

  ' APPC: return codes
  Global Const gnCA_OK = &H0                                 ' router verb completed OK
  Global Const gnCA_DEALLOC_NORMAL = &H1                     ' deallocation normal
  Global Const gnCA_PROGRAM_ERR_NO_TRUNCATION = &H2          ' program error: no truncation
  Global Const gnCA_PROGRAM_ERR_TRUNCATION = &H3             ' program error: truncation
  Global Const gnCA_PROGRAM_ERR_PURGING = &H4                ' program error: purging
  Global Const gnCA_RESOURCE_FAILURE_RETRY = &H5             ' resource failure: retry
  Global Const gnCA_RESOURCE_FAILURE_NORETRY = &H6           ' resource failure: no retry
  Global Const gnCA_UNSUCCESSFUL = &H7                       ' unsuccessful
  Global Const gnCA_APPC_BUSY = &H8                          ' APPC busy
  Global Const gnCA_PARMCHK_INVALID_VERB = &H14              ' parameter check: invalid verb
  Global Const gnCA_PARMCHK_INVALID_CONVERID = &H15          ' parameter check: invalid converssation ID
  Global Const gnCA_PARMCHK_BUFFER_CROSS_SEG = &H16          ' parameter check: buffer crossed segment
  Global Const gnCA_PARMCHK_TP_NAME_LENGTH = &H17            ' parameter check: trangsACTION0 program name length
  Global Const gnCA_PARMCHK_INV_CONVER_TYPE = &H18           ' parameter check: invalid conversation type
  Global Const gnCA_PARMCHK_BAD_SYNCLVL_ALLOC = &H19         ' parameter check: bad synchronization level allocation
  Global Const gnCA_PARMCHK_BAD_RETURN_CTRL = &H1A           ' parameter check: bad return control
  Global Const gnCA_PARMCHK_PIP_TOO_LONG = &H1B              ' parameter check: PIP data too long
  Global Const gnCA_PARMCHK_BAD_PARTNER_NAME = &H1C          ' parameter check: bad partner name
  Global Const gnCA_PARMCHK_CONF_NOT_ALLOWED = &H1D          ' parameter check: confirm not allowed
  Global Const gnCA_PARMCHK_BAD_DEALLOC_TYPE = &H1E          ' parameter check: bad deallocation type
  Global Const gnCA_PARMCHK_PREP_TO_RCV_TYPE = &H1F          ' parameter check: prepare to receive type
  Global Const gnCA_PARMCHK_BAD_FILL_TYPE = &H20             ' parameter check: bad fill type
  Global Const gnCA_PARMCHK_REC_MAX_LEN = &H21               ' parameter check: receive maximum length
  Global Const gnCA_PARMCHK_UNKNOWN_SEC_TYPE = &H22          ' parameter check: unknown security type
  Global Const gnCA_PARMCHK_RES_FLD_NOT_ZERO = &H23          ' parameter check: reserved field not zero
  Global Const gnCA_STATECHK_NOT_IN_CONF_STAT = &H28         ' state check: not in confirmed state
  Global Const gnCA_STATECHK_NOT_IN_RECEIVE = &H29           ' state check: not in receive state
  Global Const gnCA_STATECHK_REQ_SND_BAD_STAT = &H2A         ' state check: request to send bad state
  Global Const gnCA_STATECHK_SND_IN_BAD_STATE = &H2B         ' state check: send in bad state
  Global Const gnCA_STATECHK_SND_ERR_BAD_STAT = &H2C         ' state check: send error bad state
  Global Const gnCA_ALLOCERR_NO_RETRY = &H32                 ' allocation error: no retry
  Global Const gnCA_ALLOCERR_RETRY = &H33                    ' allocation error: retry
  Global Const gnCA_ALLOCERR_PGM_NOT_AVAIL_NR = &H34         ' allocation error: program not available, no retry
  Global Const gnCA_ALLOCERR_TPN_NOT_RECOG = &H35            ' allocation error: transaction program name not recognized
  Global Const gnCA_ALLOCERR_PGM_NOT_AVAIL_R = &H36          ' allocation error: program not available, retry
  Global Const gnCA_ALLOCERR_SEC_NOT_VALID = &H37            ' allocation error: security not valid
  Global Const gnCA_ALLOCERR_CONV_TYP = &H38                 ' allocation error: conversation type mismatch
  Global Const gnCA_ALLOCERR_PIP_NOT_ALLOWED = &H39          ' allocation error: PIP data not allowed
  Global Const gnCA_ALLOCERR_PIP_NOT_CORRECT = &H3A          ' allocation error: PIP data not correct
  Global Const gnCA_ALLOCERR_SYNCH_LEVEL = &H3B              ' allocation error: synchronization level not supported
  Global Const gnCA_DEALLOC_ABEND_PROGRAM = &H46             ' dellocation abend program
  Global Const gnCA_INSUFFICIENT_MEMORY = &H47               ' unsufficient memory
  Global Const gnCA_MEMORY_ALLOC_ERROR = &H48                ' memory allocation error
  Global Const gnCA_TOO_MANY_CONVERSATIONS = &H49            ' too many conversations
  Global Const gnCA_CONV_TABLE_FULL = &H4A                   ' conversion table full
  Global Const gnCA_ROUTER_NOT_INSTALLED = &H4B              ' router not installed
  Global Const gnCA_ROUTER_WRONG_LEVEL = &H4C                ' router at wrong level
  Global Const gnCA_PCSWIN_NOT_LOADED = &H4D                 ' PCSWIN not loaded
  Global Const gnCA_PCSWIN_OUT_OF_MEMORY = &H4E              ' PCSWIN out of memory
  Global Const gnCA_INVALID_USER_ID_LEN = &H4F               ' invalid user ID length
  Global Const gnCA_INVALID_PASSWORD_LEN = &H50              ' invalid password length
  Global Const gnCA_INVALID_U_NAME = &H51                    ' invalid LU length
  Global Const gnCA_UNDEFINED = &H63                         ' undefined error

  ' APPC: conversation types (nCAConvType)
  Global Const gnCA_BASIC = 0                                ' basic conversation
  Global Const gnCA_MAPPED = 1                               ' mapped conversation

  ' APPC: what received from router (nCAWhatRcvd)
  Global Const gnCA_RCVD_DATA = 0                            ' received maximum amount or end of data
  Global Const gnCA_RCVD_DATA_COMPLETE = 1                   ' received a complete logical record or last part of record
  Global Const gnCA_RCVD_DATA_INCOMPLETE = 2                 ' received less than a complete logical record, use more receives
  Global Const gnCA_RCVD_CONFIRM = 3                         ' received confirmation request from partner
  Global Const gnCA_RCVD_CONFIRM_SEND = 4                    ' received prepare to receive from partner
  Global Const gnCA_RCVD_CONFIRM_DEALLOCATE = 5              ' received deallocation request from partner
  Global Const gnCA_RCVD_SEND = 6                            ' received partner is in receive state so program can now send

  ' APPC: fill type (nCAFill)
  Global Const gnCA_FILL_BUFFER = 0                          ' receive the maximum amount or completion of data
  Global Const gnCA_FILL_LL = 1                              ' receive a complete or trunicated logical record

  ' APPC: synchronization level (nCASynchLvl)
  Global Const gnCA_SYNCLVL_NONE = 0                         ' no synchronization allowed
  Global Const gnCA_SYNCLVL_CONFIRM = 1                      ' synchronization allowed

  ' APPC: session types (nCAConvTyp)
  Global Const gnCA_SESSION_NORMAL = 0                       ' normal
  Global Const gnCA_SESSION_EXTENDED = 1                     ' extended

  ' APPC: type of deallocate requests (nCADeAllocTyp)
  Global Const gnCA_DEALLOCATE_SYNCLVL = 0                   ' deallocate based on synchronization level
  Global Const gnCA_DEALLOCATE_FLUSH = 1                     ' flush buffer then deallocate
  Global Const gnCA_DEALLOCATE_ABEND = 2                     ' dellocate abnormally

  ' APPC: security types (nCASecLvl)
  Global Const gnCA_SECURITY_NONE = 0                        ' none
  Global Const gnCA_SECURITY_SAME = 1                        ' same as router
  Global Const gnCA_SECURITY_PGM = 2                         ' program based
  
  ' APPC: possible conversation states
  Global Const gnCA_RESET_STATE = 0                          ' reset state
  Global Const gnCA_SEND_STATE = 1                           ' send state
  Global Const gnCA_RECEIVE_STATE = 2                        ' receive state
  Global Const gnCA_RCVD_CONF_STATE = 3                      ' received confirmation request state
  Global Const gnCA_RCVD_CONF_SEND_STATE = 4                 ' sent confirmation request state
  Global Const gnCA_RCVD_CONF_DEALL_STATE = 5                ' sent deallocate request state
  Global Const gnCA_PEND_DEALLOCATE_STATE = 6                ' pending deallocation state
  Global Const gnCA_INVALID_STATE = 7                        ' invalid state

  ' APPC: other constants
  Global Const gnCA_BASIC_HEADER_LEN = 2                     ' length of APPC basic conversation header
  Global Const gnCA_MAPPED_HEADER_LEN = 4                    ' length of APPC mapped header
  
 ' Constants:

  ' others
  Const nMAX_SYSTEM_COUNT = 32                          ' max number of systems in list
  Const nSYSTEM_LIST_LENGTH = 10                        ' length of each AS400 name in list

 ' Types:

  ' is used to retrieve the router's capabilities three fields
  ' are returned: nCAMaxFrameSize, sCAQueryConvSup, and
  ' sCAExtAllocSup. See "zzGetCACapabilities".
  Type CACapabilitiesType
    sHeaderReturnCode           As String * 1
    sHeaderRequest              As String * 1
    nVerbLength                 As Integer
    nCap1Length                 As Integer
    sCap1Request                As String * 1
    sCap1ReturnCode             As String * 1
    nCAMaxFrameSize             As Integer
    nCap2Length                 As Integer
    sCap2Request                As String * 1
    sCap2ReturnCode             As String * 1
    sCAQueryConvSup             As String * 1
    nCap3Length                 As Integer
    sCap3Request                As String * 1
    sCap3ReturnCode             As String * 1
    sCAExtAllocSup              As String * 1
  End Type

 ' Variables:
  Dim tCACapabilities           As CACapabilitiesType        ' router capabilities structure
  Dim nCArc                     As Integer                   ' API return code

 ' APIs:
  Declare Function EHNAPPC_GetCapabilities% Lib "EHNAPPC.DLL" (ByVal hWnd%, tCACapabilities As CACapabilitiesType)
  Declare Function EHNAPPC_GetDefaultSystem% Lib "EHNAPPC.DLL" (ByVal hWnd%, ByVal sCASystem$)
  Declare Function EHNAPPC_IsRouterLoaded% Lib "EHNAPPC.DLL" (ByVal hWnd%)
  Declare Function EHNAPPC_QuerySystems% Lib "EHNAPPC.DLL" (ByVal hWnd%, nCASystemCount%, ByVal sCASystemList$)
  Declare Function EHNAPPC_QueryUserid% Lib "EHNAPPC.DLL" (ByVal hWnd%, ByVal sCASystem$, ByVal sCAUserID$)
  Declare Function EHNAPPC_RemoteProgramStart% Lib "EHNAPPC.DLL" (ByVal hWnd%, ByVal sCASystem$, ByVal sCAProgram$, ByVal sCALibrary$, ByVal sCAPipData$, ByVal nCAPipLen%)

Sub zzCAGetCapabilities (ByVal hWnd%, nCAMaxFrameSize%, bCAQueryConvSup%, bCAExtAllocSup%)
  
 ' Description:
 '  Return router capabilities

 ' Parameters:
 ' Input:
 '  hWnd                  windows handle
 ' Output:
 '  nCAMaxFrameSize       maximum frame Size
 '  bCAQueryConvSup       query of conversation state supported
 '  bCaExtAllocSup        extended allocate supported
 
 ' Constants:

  ' get capabilities constants
  Const nGET_BUFFERSIZE_VERB_SIZE = 6
  Const nGET_BUFFERSIZE_REQUEST = 2
  Const nGET_CAPABILITIES_VERB_SIZE = 20
  Const nGET_CAPABILITIES_REQUEST = &H17
  Const nGET_EXTENDEDALLOWED_VERB_SIZE = 5
  Const nGET_EXTENDEDALLOWED_REQUEST = 4
  Const nGET_QUERYALLOWED_VERB_SIZE = 5
  Const nGET_QUERYALLOWED_REQUEST = 3

  ' others
  Const nEXTENDED_ALLOCATE_SUPPORTED = 1     ' extended allocate supported
  Const nQUERY_CONVERSATION_SUPPORTED = 1    ' query conversation supported

  ' handle errors
  On Error Resume Next
  
  ' setup call to API
  tCACapabilities.sHeaderRequest = Chr$(nGET_CAPABILITIES_REQUEST)
  tCACapabilities.nVerbLength = nGET_CAPABILITIES_VERB_SIZE

  ' buffer size inquiry
  tCACapabilities.nCap1Length = nGET_BUFFERSIZE_VERB_SIZE
  tCACapabilities.sCap1Request = Chr$(nGET_BUFFERSIZE_REQUEST)
  
  ' query conversation state allowed inquiry
  tCACapabilities.nCap2Length = nGET_QUERYALLOWED_VERB_SIZE
  tCACapabilities.sCap2Request = Chr$(nGET_QUERYALLOWED_REQUEST)

  ' extended allocate allowed inquiry
  tCACapabilities.nCap3Length = nGET_EXTENDEDALLOWED_VERB_SIZE
  tCACapabilities.sCap3Request = Chr$(nGET_EXTENDEDALLOWED_REQUEST)

  ' get capabilities of router
  nCArc = EHNAPPC_GetCapabilities(hWnd, tCACapabilities)

  ' if no error occurred
  If Err = 0 Then
  
    ' return value to caller
    nCAMaxFrameSize = tCACapabilities.nCAMaxFrameSize
  
    ' return value to caller as "real" True or False
    bCAQueryConvSup = (tCACapabilities.sCAQueryConvSup = Chr$(nQUERY_CONVERSATION_SUPPORTED))
    
    ' return value to caller as "real" True or False
    bCAExtAllocSup = (tCACapabilities.sCAExtAllocSup = Chr$(nEXTENDED_ALLOCATE_SUPPORTED))

  End If
  
End Sub

Function zzCAGetDefaultSystem (ByVal hWnd%) As String

 ' Description:
 '  Returns default AS/400. If any error
 '  occurs then nothing returned.

 ' Parameters:
 '  hWnd                          windows handle

 ' Constants:
  Const nSYSTEM_NAME_LENGTH = 8   ' length of each AS400 name

 ' Variables:
  Dim nPos             As Integer ' position of null terminator
  Dim sDefaultCASystem As String  ' default system name

  ' handle missing DLL and
  ' other possible DOS errors
  On Error Resume Next

  ' pad to hold maximum name length
  sDefaultCASystem = Space$(nSYSTEM_NAME_LENGTH)

  ' get default system name
  nCArc = EHNAPPC_GetDefaultSystem(hWnd, sDefaultCASystem)

  ' if API error then return nothing
  If nCArc <> False Or Err <> 0 Then sDefaultCASystem = gsEMPTY

  ' search for Null terminator
  nPos = InStr(sDefaultCASystem, gsCHR_NUL)
  If nPos > 0 Then sDefaultCASystem = Left$(sDefaultCASystem, nPos - 1)
  
  ' return value to caller
  zzCAGetDefaultSystem = RTrim$(sDefaultCASystem)

End Function

Function zzCAGetMaxFrameSize (ByVal hWnd%) As Integer

 ' Description:
 '  Returns the maximum frame size as a function value

 ' Parameters:
 '  hWnd              windows handle

 ' Variables:
  Dim nCAMaxFrameSize As Integer ' maximum frame size
  Dim bCAQueryConvSup As Integer ' query of conversation state supported
  Dim bCAExtAllocSup  As Integer ' extended allocate supported

  ' call routine that gets router capabilities
  Call zzCAGetCapabilities(hWnd, nCAMaxFrameSize, bCAQueryConvSup, bCAExtAllocSup)
  
  ' return value to caller
  zzCAGetMaxFrameSize = nCAMaxFrameSize

End Function

Function zzCAGetSystemCount (ByVal hWnd%) As Integer

 ' Description:
 '   Returns the number of AS/400s available.
 '   If any error occurs returns 0.

 ' Parameters:
 '  hWnd                          windows handle

 ' Variables:
  Dim nCASystemCount As Integer   ' number of systems available
  Dim sCASystemList  As String    ' list of systems available

  ' handle missing DLL
  On Error Resume Next
  
  ' setup for maximum length
  sCASystemList = Space$(nMAX_SYSTEM_COUNT * nSYSTEM_LIST_LENGTH)
  
  ' make call to router
  nCArc = EHNAPPC_QuerySystems(hWnd, nCASystemCount, sCASystemList)
  If nCArc <> False Or Err <> 0 Then nCASystemCount = 0

  ' return value to caller
  zzCAGetSystemCount = nCASystemCount

End Function

Function zzCAGetSystemList (ByVal hWnd%) As String

 ' Description:
 '   Returns a list of AS/400s available.
 '   If any error returns nothing.

 ' Parameters:
 '  hWnd                          windows handle

 ' Variables:
  Dim nCASystemCount As Integer   ' number of systems available
  Dim sCASystemList  As String    ' list of systems available
  
  ' handle missing DLL
  On Error Resume Next
  
  ' setup for maximum length
  sCASystemList = Space$(nMAX_SYSTEM_COUNT * nSYSTEM_LIST_LENGTH)

  ' make call to router
  nCArc = EHNAPPC_QuerySystems(hWnd, nCASystemCount, sCASystemList)

  ' if API error then return nothing
  If nCArc <> False Or Err <> 0 Then sCASystemList = gsEMPTY

  ' return value to caller, free string space
  zzCAGetSystemList = RTrim$(sCASystemList)

End Function

Function zzCAGetUserID (ByVal hWnd%, ByVal sCASystem$) As String

 ' Description:
 '  Get User ID. If any error return nothing

 ' Parameters:
 '  hWnd                       windows handle
 '  sCASystem                  system name

 ' Constants:
  Const nUSER_NAME_LENGTH = 10 ' length of user name

 ' Variables:
  Dim nPos      As Integer     ' location of Null terminator
  Dim sCAUserID As String      ' current user ID

  ' handle missing DLL and
  ' other possible DOS errors
  On Error Resume Next

  ' set for maximum length
  sCAUserID = Space$(nUSER_NAME_LENGTH)
  
  ' make the call
  nCArc = EHNAPPC_QueryUserid(hWnd, sCASystem, sCAUserID)

  ' if API error then return nothing
  If nCArc <> False Or Err <> 0 Then sCAUserID = gsEMPTY

  ' replace Null terminator
  nPos = InStr(sCAUserID, gsCHR_NUL)
  If nPos > 0 Then sCAUserID = Left$(sCAUserID, nPos - 1)
  
  ' return value to caller
  zzCAGetUserID = Trim$(sCAUserID)

End Function

Sub zzCAPutSystemListIntoArray (ByVal hWnd%, asCASystem$())

 ' Description:
 '  Places list of AS/400s into an array.

 ' Parameters:
 '  hWnd                       windows handle
 '  asCASystem                 system name array
  
 ' Variables:
  Dim nItem         As Integer ' loop counter
  Dim nLb           As Integer ' lower boundary
  Dim nUb           As Integer ' upper boundary
  Dim nPos          As Integer ' 1st occurrence in a INSTR Search
  Dim sList         As String  ' list to be placed into array
  Dim sCASystem     As String  ' system name

  ' get boundaries
  nLb = LBound(asCASystem)
  nUb = UBound(asCASystem)

  ' get list of systems
  sList = zzCAGetSystemList$(hWnd) & Space$(nMAX_SYSTEM_COUNT * nSYSTEM_LIST_LENGTH)
  sList = Left$(sList, nMAX_SYSTEM_COUNT * nSYSTEM_LIST_LENGTH)

  ' loop to get each system
  For nItem = nLb To nUb

    ' get specific system
    sCASystem = RTrim$(Mid$(sList, 1 + ((nItem - 1) * nSYSTEM_LIST_LENGTH), nSYSTEM_LIST_LENGTH))

    ' search for Null terminator
    nPos = InStr(sCASystem, gsCHR_NUL)
    If nPos > 0 Then sCASystem = Left$(sCASystem, nPos - 1)

    ' if nothing then exit loop
    If sCASystem = gsEMPTY Then Exit For
    
    ' place into array
    asCASystem(nItem) = Trim$(sCASystem)
    
  Next nItem

End Sub

Sub zzCAPutSystemListIntoCtrl (ByVal hWnd%, ctl As Control)

 ' Description:
 '  Places list of AS/400s into a list or combo box.
  
 ' Parameters:
 '  hWnd              windows handle
 '  ctl               control to populate
  
 ' Variables:
  Dim nItem           As Integer   ' loop counter
  Dim nPos            As Integer   ' 1st occurrence in a INSTR Search
  Dim sList           As String    ' list to be placed into control
  Dim sCASystem       As String    ' system name
  
  ' clear the control
  ctl.Clear

  ' get list of systems
  sList = zzCAGetSystemList(hWnd) & Space$(nMAX_SYSTEM_COUNT * nSYSTEM_LIST_LENGTH)
  sList = Left$(sList, nMAX_SYSTEM_COUNT * nSYSTEM_LIST_LENGTH)

  ' loop to get each system
  For nItem = 1 To nMAX_SYSTEM_COUNT
    
    ' get specific system
    sCASystem = RTrim$(Mid$(sList, 1 + ((nItem - 1) * nSYSTEM_LIST_LENGTH), nSYSTEM_LIST_LENGTH))

    ' search for Null terminator
    nPos = InStr(sCASystem, gsCHR_NUL)
    If nPos > 0 Then sCASystem = Left$(sCASystem, nPos - 1)

    ' if nothing then exit loop
    If sCASystem = gsEMPTY Then Exit For
    
    ' add to control
    ctl.AddItem Trim$(sCASystem)

  Next nItem

  ' if systems put in then set to first one
  If ctl.ListCount > 0 Then ctl.ListIndex = 0

End Sub

Function zzCARouterLoaded (ByVal hWnd%) As Integer

 ' Description:
 '  Returns True (-1) if router is loaded and DLL
 '  was found; returns False (0) otherwise

 ' Parameters:
 '  hWnd              windows handle
  
  ' turn on error handling
  On Error Resume Next
  
  ' determine if the router is loaded
  zzCARouterLoaded = (EHNAPPC_IsRouterLoaded(hWnd) <> False)
  
  ' if any DOS error then must assume router will not work
  If Err <> False Then zzCARouterLoaded = False

End Function

Function zzCAStartProgram (ByVal hWnd%, ByVal sCASystem$, ByVal sCALibrary$, ByVal sCAProgram$, ByVal sCAPipData$) As Integer

 ' Description:
 '  Start AS400 program.
 
 ' Parameters:
 '  hWnd              windows handle
 '  sCASystem         System
 '  sCALibrary        Library
 '  sCAProgram        Program
 '  sCAPipData        Program initialization parameters

  ' start AS400 program
  zzCAStartProgram = EHNAPPC_RemoteProgramStart(hWnd, RTrim$(sCASystem) & gsCHR_NUL, RTrim$(sCAProgram) & gsCHR_NUL, RTrim$(sCALibrary) & gsCHR_NUL, sCAPipData, Len(sCAPipData))
  
End Function

