/***************************************************************************\

   File Name:  addmenu.c

   1.  Lotus grants you a non-exclusive royalty free right to use and modify
       the source code version and to reproduce and distribute the object
       code version of the Example Programs provided in the Toolkit, provided
       that you:

       a) distribute the Example Programs only in conjunction with and as a
          part of your software application product which is designed as an
          add-in to Lotus 1-2-3, and

       b) do not represent your product to be a product of Lotus Development
          Corporation.

   2.  The Example Programs are provided as is, without warranty of any kind,
       either express or implied, including, without limitation, the implied
       warranties of merchantability of fitness for a particular purpose.

\***************************************************************************/

/***************************************************************************\

   Description: Sample application that demonstrates adding to the
                1-2-3 sheet menu. Adds a new popup menuitem to the
                Main menu that demonstrates dynamic checking,
                graying, and label changes.

\***************************************************************************/

/*************************************************************************/
/*
**             FILE INCLUDES
*/

#include "windows.h"
#include "string.h"
#include "lcicomn.h"
#include "lcigui.h"
#include "lciUtil.h"

#include "addmenu.h"

/*************************************************************************/
/*
**             DEFINES AND TYPEDEFS
*/

/* a series of unique, contiguous menu id's are allocated dynamically; the
 * first id in the series is stored in a global variable, wIdStart. this
 * macro allows us to refer to menu items symbolically in terms of an
 * offset from wIdStart.
 */
#define ID_MENU(x)   (wIdStart + x)

#define NEW_ITEM_COUNT        5  /* number of new menu items. */
#define NEW_MENU_POS          8

/*************************************************************************/
/*
**             DATA STRUCTURES
*/

/*************************************************************************/
/*
**             STATIC FUNCTION PROTOTYPES
*/

LCT_STATUS static NEAR CreateMenus(VOID);
VOID static NEAR DestroyMenus(VOID);
VOID static NEAR RestoreMainMenu(VOID);

/*************************************************************************/
/*
**             PUBLIC FUNCTION PROTOTYPES
*/

BOOL LibMain (HANDLE hMod, WORD wDataSeg, WORD cbHeap, LPSTR lpszCmdLine);
LCT_STATUS LCI_CALL AdnPreinit(LCH_CONTEXT Context);
LCT_STATUS LCI_CALL AdnInitialize(LCH_CONTEXT Context);
LCT_STATUS LCI_CALL AdnTerminate(LCH_CONTEXT Context);
lushort    LCI_CALL AdnVersion(LCH_CONTEXT Context);
DWORD FAR PASCAL AddinToolProc(LCH_TOOLINST hInst,WORD msg,DWORD p1,DWORD p2);

/*************************************************************************/
/*
**             EXTERNAL DATA
*/

/*************************************************************************/
/*
**             STATIC DATA
*/

static LCH_CONTEXT   hGlobalContext;
static LCH_TOOLCLASS hGlobalTool;
static LCH_MENU      hNewPopup;
static LCH_MENU      hMainMenu;

static HANDLE        hModule;
static lushort       wIdStart;

/*************************************************************************/
/*
**             PUBLIC DATA
*/

/*************************************************************************/
/*
**             EXTERNAL CODE
*/

/*************************************************************************/
/*
**             STATIC CODE
*/

LCT_STATUS static NEAR CreateMenus()
{
   LCT_STATUS  err;
   BYTE        pszShort[ID_MAX+1][MAX_SHORT_LEN];
   BYTE        pszLong[ID_MAX+1][MAX_LONG_LEN];
   lushort     i;
   lushort     menuItemNum = 5; /* tools menu */


   /* GET A HANDLE TO THE TOP LEVEL MENU. */
   if (err = LciTcGetMenu(hGlobalTool, &hMainMenu))
      goto xit;

   /* NEED 5 MENU IDS. ONE FOR THE ITEM WE'LL ADD TO THE TOOLS MENU AND
      FOUR FOR THE ITEMS IN THE NEW POPUP.  */
   if (err = LciTcGetMenuIds(hGlobalTool, NEW_ITEM_COUNT, &wIdStart))
      goto xit;


   /* load menu labels (short prompts) from resource file. */

   for (i = 0; i <= ID_MAX; i++)
      LoadString(hModule, i, (LPSTR)&pszShort[i], MAX_SHORT_LEN);

   /* load menu long prompts from resource file. */

   for (i = 0; i <= ID_MAX; i++)
      LoadString(hModule, i+16, (LPSTR)&pszLong[i], MAX_LONG_LEN);

   /* GET A HANDLE TO A NEW POPUP THAT WILL THEN HAVE 4 ITEMS INSERTED. */
   if (err = LciMnConstructNew(
         hGlobalContext,         /* add-in context */
         LCI_TOOLCLASS_SHEET,    /* new popup will be added to sheet menu */
         LNULL,                  /* class string is null for sheet class */
         ID_MENU (IDP_NEWPOPUP), /* id of new popup based on wIdStart */
         LCI_MENUTYPE_POPUP,     /* menu type */
         &hNewPopup))            /* address in which to return menu handle */
      goto xit;

   /* INSERT THE 4 ITEMS INTO THE NEW POPUP. */
   if (err = LciMnInsertItem(
         hNewPopup,                    /* handle of menu in which to insert */
         LCI_MENUITEM_APPEND,          /* append new item to menu */
         LCI_MENUITEM_BYPOSITION,      /* use BYPOSITION for append */
         ID_MENU (IDP_CHECKITEM),      /* id of new item based on wIdStart */
         LCI_MENUITEM_PROC,            /* item type */
         0L,                           /* item data (ignored for this item) */
         (LPSTR)pszShort[IDP_CHECKITEM],  /* short prompt (from res file) */
         (LPSTR)pszLong[IDP_CHECKITEM],   /* long prompt (from res file) */
         LCF_MENUITEM_NOREDRAW))          /* option flags */
      goto xit;

   if (err = LciMnInsertItem(
         hNewPopup,                    /* handle of menu in which to insert */
         LCI_MENUITEM_APPEND,          /* append new item to menu */
         LCI_MENUITEM_BYPOSITION,      /* use BYPOSITION for append */
         ID_MENU (IDP_GRAYITEM),       /* id of new item based on wIdStart */
         LCI_MENUITEM_PROC,            /* item type */
         0L,                           /* item data (ignored for this item) */
         (LPSTR)pszShort[IDP_GRAYITEM],   /* short prompt (from res file) */
         (LPSTR)pszLong[IDP_GRAYITEM],    /* long prompt (from res file) */
         LCF_MENUITEM_NOREDRAW))          /* option flags */
      goto xit;

   if (err = LciMnInsertItem(
         hNewPopup,                    /* handle of menu in which to insert */
         LCI_MENUITEM_APPEND,          /* append new item to menu */
         LCI_MENUITEM_BYPOSITION,      /* use BYPOSITION for append */
         ID_MENU (IDP_CHANGELABEL),    /* id of new item based on wIdStart */
         LCI_MENUITEM_PROC,            /* item type */
         0L,                           /* item data (ignored for this item) */
         (LPSTR)pszShort[IDP_CHANGELABEL],/* short prompt (from res file) */
         (LPSTR)pszLong[IDP_CHANGELABEL], /* long prompt (from res file) */
         LCF_MENUITEM_NOREDRAW))          /* option flags */
      goto xit;

   if (err = LciMnInsertItem(
         hNewPopup,                    /* handle of menu in which to insert */
         LCI_MENUITEM_APPEND,          /* append new item to menu */
         LCI_MENUITEM_BYPOSITION,      /* use BYPOSITION for append */
         ID_MENU (IDP_RESTORE),        /* id of new item based on wIdStart */
         LCI_MENUITEM_PROC,            /* item type */
         0L,                           /* item data (ignored for this item) */
         (LPSTR)pszShort[IDP_RESTORE], /* short prompt (from res file) */
         (LPSTR)pszLong[IDP_RESTORE],  /* long prompt (from res file) */
         LCF_MENUITEM_GRAYED | LCF_MENUITEM_NOREDRAW))   /* option flags */
      goto xit;

   if (err = LciMnInsertItem(
         hMainMenu,                    /* insert into Main menu */
         (lushort)NEW_MENU_POS,        /* append new item to menu */
         LCI_MENUITEM_BYPOSITION,      /* use BYPOSITION for append */
         ID_MENU (IDP_NEWPOPUP),       /* id of new item based on wIdStart */
         LCI_MENUITEM_POPUP,           /* item type popup */
         (lulong)hNewPopup,            /* item data is menu handle of popup */
         (LPSTR)pszShort[IDP_NEWPOPUP],/* short prompt (from res file) */
         (LPSTR)pszLong[IDP_NEWPOPUP], /* long prompt (from res file) */
         0L))                          /* no option flags */
      goto xit;
xit:
   if (err)
      DestroyMenus();

   return (err);
}

VOID static NEAR DestroyMenus()
{
   if (LNULL != hNewPopup)
      LciMnDestroy(&hNewPopup);

   if (LNULL != hMainMenu)
      LciMnDestroy(&hMainMenu);

   LciTcDeleteToolProc(hGlobalTool, AddinToolProc);
   LciMnRedraw(hGlobalContext) ;
   
}

VOID static NEAR RestoreMainMenu()
{

   if (LNULL != hMainMenu) {

      /* DELETING THE POPUP ITEM FROM THE TOOLS MENU CAUSES
       * THE POPUP AND ALL OF ITS ITEMS TO BE DELETED ALSO. */
      LciMnDeleteItem(hMainMenu,
                     ID_MENU (IDP_NEWPOPUP),
                     LCI_MENUITEM_BYCOMMAND);

      DestroyMenus();
   }
}

/*************************************************************************/
/*
**             PUBLIC CODE
*/
BOOL LibMain (HANDLE hMod, WORD wDataSeg, WORD cbHeap, LPSTR lpszCmdLine)
{
   wDataSeg;
   cbHeap;
   lpszCmdLine;

   hModule = hMod;

   return TRUE;
}



lushort LCI_CALL AdnVersion(LCH_CONTEXT Context)
{
   return (LCI_VERSION);
}

LCT_STATUS LCI_CALL AdnPreinit(LCH_CONTEXT Context)
{
   Context;

   return(LCS_SUCCESS);
}

LCT_STATUS LCI_CALL AdnInitialize(LCH_CONTEXT Context)
{
   LCT_STATUS  err = LCS_SUCCESS;

   hGlobalContext = Context;

   /* CREATE A TOOLCLASS OBJECT REFERRING TO THE SHEET TOOL.   */
   if (err = LciTcConstructName(Context,
                      LCI_TOOLCLASS_SHEET,
                      LNULL,
                      &hGlobalTool))
      goto xit;

   /* need to add a tool proc to the list of tool procs for the sheet
    * tool so that we can process messages for the new menu items.
    */

   if (err = LciTcAddToolProc(hGlobalTool, AddinToolProc))
      goto xit;

   if (err = CreateMenus())
      goto xit;

xit:
   if (err) {
      DestroyMenus();
      LciTcDestroy(&hGlobalTool);
   }
   return(err);
}


LCT_STATUS LCI_CALL AdnTerminate(LCH_CONTEXT Context)
{
   RestoreMainMenu();
   LciTcDestroy(&hGlobalTool);
   return(LCS_SUCCESS);
}

DWORD FAR PASCAL AddinToolProc(LCH_TOOLINST hInst,WORD msg,DWORD p1,DWORD p2)
{
   DWORD dwResult = 0L;
   BOOL  bDefault = LFALSE;
   LCT_STATUS  err = LCS_SUCCESS;
   BYTE pszLabel[MAX_SHORT_LEN];
   BYTE pszAltLabel[MAX_SHORT_LEN];

   switch (msg)
   {
      case TOOLMSG_MENURESET:
               if (err = CreateMenus()) {
                  DestroyMenus();
                  LciTcDestroy(&hGlobalTool);
                  }
               return(err);
               break;

      case TOOLMSG_MENUPROC:

         /* is it one of our menu items? */
         if (p2 >= wIdStart && p2 <= wIdStart + NEW_ITEM_COUNT)
         {
            /* since we don't know at compile time what our id base is,
             * we subtract it here to get a value that can be compared
             * to known values and used in a switch statement.
             */
            WORD wIdOffset = (WORD)p2 - wIdStart;

            switch (wIdOffset)
            {
               BOOL  bItemState;
               BYTE  pszLabelBuf[MAX_SHORT_LEN];

               /* Toggle the check mark */
               case IDP_CHECKITEM:
                  LciMnGetCheckFlag(hNewPopup,
                                    ID_MENU (IDP_CHECKITEM),
                                    LCI_MENUITEM_BYCOMMAND,
                                    &bItemState);
                  LciMnSetCheckFlag(hNewPopup,
                                    ID_MENU (IDP_CHECKITEM),
                                    LCI_MENUITEM_BYCOMMAND,
                                    !bItemState);
                  break;

               /* Toggle Gray */
               case IDP_GRAYITEM:
                  LciMnGetEnableFlag(hNewPopup,
                                     ID_MENU (IDP_RESTORE),
                                     LCI_MENUITEM_BYCOMMAND,
                                     &bItemState);
                  LciMnSetEnableFlag(hNewPopup,
                                     ID_MENU (IDP_RESTORE),
                                     LCI_MENUITEM_BYCOMMAND,
                                     !bItemState);
                  break;

               /* Change the main label in the menu */
               case IDP_CHANGELABEL:
                  LoadString(hModule,IDS_CHANGELABEL,pszLabel,MAX_SHORT_LEN);
                  LoadString(hModule,IDS_ALTLABEL,pszAltLabel,MAX_SHORT_LEN);
                  LciMnGetLabel(hNewPopup,
                                ID_MENU (IDP_CHANGELABEL),
                                LCI_MENUITEM_BYCOMMAND,
                                MAX_SHORT_LEN,
                                pszLabelBuf);
                  LciMnSetLabel(hNewPopup,
                                ID_MENU (IDP_CHANGELABEL),
                                LCI_MENUITEM_BYCOMMAND,
                                _fstrcmp((LPSTR)pszLabelBuf, (LPSTR)pszLabel)
                                          == 0 ? pszAltLabel : pszLabel);
                  break;

               /* Put back to original state */
               case IDP_RESTORE:

                  RestoreMainMenu();
                  break;
            }
         }
         else  /* if it's not one of our menu items, make sure the
                * message is passed on to the next tool proc.
                */
            bDefault = LTRUE;

         break;

      default:

         bDefault = LTRUE;
         break;

   }
   if (bDefault)
      dwResult = LciTcDefToolProc(hGlobalTool, hInst, msg, p1, p2);

   return (dwResult);
}
