unit SimpleThread;

interface
uses
 Windows, SysUtils, Classes;

type

  EThreadError = class( Exception );

  TSimpleThread = class( TComponent )
  private
    FTerminated :Boolean;
    FHandle :THandle;
    FThreadID :THandle;
    FPriority :TThreadPriority;
    FActive :Boolean;
    FSuspended :Boolean;
    FOnExecute :TNotifyEvent;
    FOnActivate :TNotifyEvent;
    FOnTerminate :TNotifyEvent;
    FOnSuspend :TNotifyEvent;
    FOnResume :TNotifyEvent;
    procedure UpdateThreadPriority;
    procedure SetSuspended( NewState :Boolean );
    procedure SetActive( NewState :Boolean );
    procedure SetPriority( NewPriority :TThreadPriority );
  protected
    procedure Loaded; override;
    procedure Activate; virtual;
    procedure Terminate; virtual;
    procedure Suspend; virtual;
    procedure Resume; virtual;
    procedure Execute; virtual;
    procedure DoActivate;
    procedure DoTerminate;
    procedure DoSuspend;
    procedure DoResume;
  public
    constructor Create( anOwner :TComponent ); override;
    destructor Destroy; override;
    procedure Kill;
    property Handle :THandle
      read FHandle;
    property ThreadID :THandle
      read FThreadID;
    property Terminated :Boolean read FTerminated;
  published
    property Priority :TThreadPriority
      read FPriority write SetPriority;
    property Active :Boolean
      read FActive write SetActive;
    property Suspended :Boolean
      read FSuspended write SetSuspended;
    property OnExecute :TnotifyEvent
      read FOnExecute write FOnExecute;
    property OnActivate :TNotifyEvent
      read FOnActivate write FOnActivate;
    property OnTerminate :TNotifyEvent
      read FOnTerminate write FOnTerminate;
    property OnSuspend :TNotifyEVent
      read FOnSuspend write FOnSuspend;
    property OnResume :TNotifyEvent
      read FOnResume write FOnResume;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents( 'more...', [ TSimpleThread ] );
end;


procedure ThreadProc( Self :TSimpleThread ); stdcall;
begin
 with Self do
  begin
    Activate;
    Execute;
    Terminate;
  end;
end;


{ TSimpleThread }

procedure TSimpleThread.Activate;
begin
 if assigned( FOnActivate ) then FOnActivate( Self );
end;

procedure TSimpleThread.Terminate;
begin
 if assigned( FOnTerminate ) then FOnTerminate( Self );
end;

procedure TSimpleThread.Suspend;
begin
 if assigned( FOnSuspend ) then FOnSuspend( Self );
end;

procedure TSimpleThread.Resume;
begin
 if assigned( FOnResume ) then FOnResume( Self );
end;

procedure TSimpleThread.Execute;
begin
 if Assigned( FOnExecute ) then FOnExecute( Self );
end;

constructor TSimpleThread.Create( anOwner :TComponent );
begin
 inherited Create( anOwner );
 FPriority := tpNormal;
 FHandle := 0;
 FTerminated := false;
 FActive := false;
 FSuspended := false;
end;

destructor TSimpleThread.Destroy;
begin
 Active := false;
 inherited;
end;

procedure TSimpleThread.DoActivate;
var F :Longint;
begin
 F := 0;
 if ( FSuspended ) then F := F OR CREATE_SUSPENDED;
 FHandle := CreateThread( nil, 0, @ThreadProc,
  Pointer( Self ), F, FThreadID );
 if ( FHandle = 0 ) then raise EThreadError.Create( 'Error creating thread' );
 FActive := true;
end;

procedure TSimpleThread.DoTerminate;
begin
 if ( FHandle = 0 ) then exit;
 if Active and ( not Suspended ) then
    begin
     FTerminated := true;
     WaitForSingleObject( FHandle, INFINITE );
     FActive := false;
     FTerminated := false;
    end;
 if ( FHandle <> 0 ) then CloseHandle( FHandle );
 FHandle := 0;
 FThreadID := 0;
end;

procedure TSimpleThread.DoSuspend;
begin
  FSuspended := true;
  SuspendThread( FHandle );
  Suspend;
end;

procedure TSimpleThread.DoResume;
begin
  if ResumeThread(FHandle) = 1 then
    begin
     FSuspended := False;
     Resume;
    end;
end;


procedure TSimpleThread.Kill;
begin
 if ( FHandle = 0 ) then exit;
 TerminateThread( FHandle, 0 );
 FHandle := 0;
 FThreadID := 0;
 FActive := false;
 Terminate;
end;

const
  Priorities: array [TThreadPriority] of Integer =
   (THREAD_PRIORITY_IDLE, THREAD_PRIORITY_LOWEST, THREAD_PRIORITY_BELOW_NORMAL,
    THREAD_PRIORITY_NORMAL, THREAD_PRIORITY_ABOVE_NORMAL,
    THREAD_PRIORITY_HIGHEST, THREAD_PRIORITY_TIME_CRITICAL);

procedure TSimpleThread.UpdateThreadPriority;
begin
  SetThreadPriority(FHandle, Priorities[ FPriority ] );
end;

procedure TSimpleThread.SetSuspended( NewState :Boolean );
begin
  if ( Suspended xor NewState ) then
   if FHandle = 0
     then FSuspended := NewState
     else if NewState then DOSuspend else DoResume;
end;

procedure TSimpleThread.SetActive( NewState :Boolean );
begin
  if ( csLoading in COmponentState ) or ( csDesigning in COmponentState )
   then begin
     FActive := NewState;
     exit;
   end;
  if ( Active xor NewState ) then
   if NewState
     then DoActivate
     else DoTerminate;
end;

procedure TSimpleThread.SetPriority( NewPriority :TThreadPriority );
begin
  FPriority := NewPriority;
  if FHandle <> 0 then UpdateThreadPriority;
end;

procedure TSimpleThread.Loaded;
begin
 inherited;
 if not ( csDesigning in COmponentState ) then
  if Active then DoActivate;
end;


end.
