{
 Designer: Craig Ward, 100554.2072@compuserve.com
 Date:     17/1/96
 Version:  1.0


 Function: Print Utility DLL.


 Calls:   There are several calls available. These are as follows:

          ShowPrintDlg(pAlias, pTable, pQuery, pHeader, pSub, pHelp: pChar);
          - shows the print dialog

          HidePrintDlg(pAlias, pTable, pQuery, pHeader, pSub, pHelp: pChar);
          - does not show the dilalog, and initiates the print routine


 Notes:   The edit boxes are not visible at run-time. These are used to set the
          fonts for the header and sub-header.

          The size of the array to store field names is 50, so only tables with
          50 or less fields will work in conjunction with this DLL, unless you
          increase these values.


 Extra:   The printing routines are taken from Xavier Pacheco and Steve Teixeira's
          "Delphi Developer's Guide", which is available from SAMS Publishing.
          Great book, with some really useful examples.

          All criticisms, help and general advice is greatly welcomed.
*********************************************************************************}
unit Print;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, Buttons, DBTables, DB, Printers, StdCtrls, ExtCtrls,
  Grids;

type
  TPrintDlg = class(TForm)
    editHead: TEdit;
    BitBtn1: TBitBtn;
    BitBtn2: TBitBtn;
    BitBtn3: TBitBtn;
    BitBtn4: TBitBtn;
    PrinterSetupDialog1: TPrinterSetupDialog;
    Bevel1: TBevel;
    Label1: TLabel;
    lblTable: TLabel;
    Label3: TLabel;
    lblRecords: TLabel;
    editSub: TEdit;
    editData: TEdit;
    editCol: TEdit;
    queryPrint: TQuery;
    BitBtn5: TBitBtn;
    procedure BitBtn2Click(Sender: TObject);
    procedure BitBtn1Click(Sender: TObject);
    procedure BitBtn3Click(Sender: TObject);
    procedure BitBtn4Click(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure BitBtn5Click(Sender: TObject);
  private
    { Private declarations }
    FAlias: string;                                  {stores alias name}
    FHeader: string;                                 {stores heading}
    FSubHeader: string;                              {stores sub-heading}
    FColumns: array[0..49] of string;                {array to store field names}
    FOrientation: TPrinterOrientation;               {store printer orientation on load}
    FStrings: TStrings;
    PixelsInInchx: integer;                          {stores Pixels per inch }
    LineHeight: Integer;                             {stores the line height }
    AmountPrinted: integer;                          {stores amount, in pixels, printed on a page }
    TenthsOfInchPixelsY: integer;                    {pixels in 1/10 of an inch used for line spacing }
    procedure printTab;
    procedure printHeader;
    procedure printSubHeader;
    procedure printColumnNames;
    procedure printLine(items: TStringList);
    procedure SetQuery;
    procedure SetColumns;
    procedure CustInitialise;
    procedure printFooter;
  public
    { Public declarations }
    FSQL: string;                                    {stores initial query for TQuery}
    FTable: string;                                  {stores table name}
  end;

var
  PrintDlg: TPrintDlg;
  iPage: integer;

const
 iArraySize = 49;     {size of field array}
 iWidth = 15;         {width used for columns}

{exported procedures}
procedure ShowPrintDlg(pAlias, pTable, pQuery, pHeader, pSub, pHelp: pChar); export;
procedure HidePrintDlg(pAlias, pTable, pQuery, pHeader, pSub, pHelp: pChar); export;


{*******************************************************************************}

implementation

{$R *.DFM}

uses
 AbortBx, Query;

{***exported routines***********************************************************}

procedure ShowPrintDlg(pAlias, pTable, pQuery, pHeader, pSub, pHelp: pChar);
begin

 {create dialog}
 try
  PrintDlg := TPrintDlg.create(application);

  {set fields}
  if Assigned(pAlias) then PrintDlg.FAlias := StrPAS(pAlias);
  if Assigned(pTable) then PrintDlg.FTable := StrPAS(pTable);
  if Assigned(pQuery) then PrintDlg.FSQL := StrPAS(pQuery);
  if Assigned(pHeader) then PrintDlg.FHeader := StrPAS(pHeader);
  if Assigned(pSub) then PrintDlg.FSubHeader := StrPAS(pSub);
  if Assigned(pHelp) then application.HelpFile := StrPAS(pHelp);

  PrintDlg.CustInitialise;

  QueryDlg := TQueryDlg.create(application);

  PrintDlg.showModal;

 finally
  PrintDlg.free;
  QueryDlg.free;
 end;

end;

procedure HidePrintDlg(pAlias, pTable, pQuery, pHeader, pSub, pHelp: pChar);
begin

 {create dialog}
 try
  PrintDlg := TPrintDlg.create(application);

  {set fields}
  if Assigned(pAlias) then PrintDlg.FAlias := StrPAS(pAlias);
  if Assigned(pTable) then PrintDlg.FTable := StrPAS(pTable);
  if Assigned(pQuery) then PrintDlg.FSQL := StrPAS(pQuery);
  if Assigned(pHeader) then PrintDlg.FHeader := StrPAS(pHeader);
  if Assigned(pSub) then PrintDlg.FSubHeader := StrPAS(pSub);
  if Assigned(pHelp) then application.HelpFile := StrPAS(pHelp);

  {initiate}
  PrintDlg.CustInitialise;
  PrintDlg.PrintTab;

 finally
  PrintDlg.free;
 end;

end;


{***custom routines*************************************************************}


{initialise}
procedure TPrintDlg.CustInitialise;
begin

 {set data}
 setQuery;
 setColumns;
 iPage := 1;

 {set lables}
 lblTable.caption := FTable;

 {save printer orientation}
 FOrientation := printer.orientation;

 {set landscape as default printer orientation}
 printer.orientation := poLandscape;

end;


{set table details}
procedure TPrintDlg.SetQuery;
begin
 with queryPrint do
  begin
   active := false;
    databaseName := FAlias;
    SQL.clear;
    SQL.add(FSQL);
   active := true;
   {update label}
   lblRecords.caption := IntToStr(queryPrint.RecordCount);
  end;
end;

{set field array}
procedure TPrintDlg.SetColumns;
var
 iInc: integer;
begin

 {clear columns}
 for iInc := 0 to iArraySize do
  begin
   FColumns[iInc] := '';
  end;

 {read columns}
  begin
   for iInc := 0 to (queryPrint.FieldCount -1) do
    begin
     FColumns[iInc] := queryPrint.Fields[iInc].FieldName;
    end;
  end;
end;


{print table}
procedure TPrintDlg.printTab;
var
  Items: TStringList;
  iInc: integer;
begin

  {initialise}
  Items := TStringList.Create;
  PixelsInInchx := GetDeviceCaps(Printer.Handle, LOGPIXELSX);              {get Pixels per inch horizonally}
  TenthsOfInchPixelsY := GetDeviceCaps(Printer.Handle, LOGPIXELSY) div 10;
  AmountPrinted := 0;

  try

   self.Enabled := false;                                                  {Disable the parent Form }
   Printer.BeginDoc;                                                       {Initiate a print job }

   AbortDlg := TAbortDlg.create(application);
   AbortDlg.lblTask.caption := 'Printing...';
   AbortDlg.show;
   Application.ProcessMessages;

   LineHeight := Printer.Canvas.TextHeight('X')+TenthsOfInchPixelsY;       {Calculate a arbitrary line height }
   if FHeader <> '' then PrintHeader;
   if FSubHeader <> '' then printSubHeader;
   PrintColumnNames;                                                       {Print the column Names }

   queryPrint.First;
   {Add the data in the fields into a TStringList in the order that they are going to be printed }
    while (not queryPrint.Eof) or Printer.Aborted do
     begin

      Application.ProcessMessages;
      with Items do
       begin
        for iInc := 0 to iArraySize do
         begin
          if FColumns[iInc] <> '' then
           begin
            AddObject(queryPrint.FieldByName(FColumns[iInc]).displayText,pointer(iWidth));
           end;
         end;
       end;

      Printer.Canvas.Font.Assign(editData.font);
      PrintLine(Items);   { Print the line }

     {Force printjob to begin a new page if printed output has exceeded the page height }
     if AmountPrinted + LineHeight > (Printer.PageHeight - (LineHeight*2)) then
      begin
       PrintFooter;
       AmountPrinted := 0;                                                {Reset to zero }
       if not Printer.Aborted then Printer.NewPage;                       {Force page eject }
       inc(iPage);
       PrintHeader;                                                       {Print the headers again }
       PrintSubHeader;
       PrintColumnNames;                                                  {Print the column names again }
      end;

     Items.Clear;                                                         {Clear this record from the TStringList }
     queryPrint.Next;                                                       {Go to the next record }
    end;

    AbortDlg.free;

    if not Printer.Aborted then
     begin
      PrintFooter;
      Printer.EndDoc;
     end;
    self.Enabled := true;

  except
    on E: Exception do MessageDlg(E.Message, mtError, [mbok], 0);
  end;

  Items.Free;                                                             {Free the TStringList }
  if isWindowVisible(self.handle) then setFocus;

end;

{print header}
procedure TPrintDlg.PrintHeader;
begin

 Printer.Canvas.Font.Assign(editHead.font);

 {Print out the Header }
 with Printer do
  begin
   if not Printer.Aborted then
    begin
     Canvas.LineTo(PageWidth,0);
     Canvas.TextOut((PageWidth div 2)-(Canvas.TextWidth(FHeader) div 2),1, FHeader);
    end;
   {Increment AmountPrinted by the LineHeight }
   AmountPrinted := AmountPrinted + LineHeight+TenthsOfInchPixelsY;
  end;

end;


{print header}
procedure TPrintDlg.PrintSubHeader;
begin

  Printer.Canvas.Font.Assign(editSub.font);

  {Print out the Header }
  with Printer do
   begin
    if not Printer.Aborted then
     begin
      Canvas.TextOut((PageWidth div 2)-(Canvas.TextWidth(FSubHeader) div 2),LineHeight,FSubHeader);
      AmountPrinted := AmountPrinted + LineHeight+TenthsOfInchPixelsY;
      Canvas.MoveTo(0,(AmountPrinted - LineHeight+TenthsOfInchPixelsY));
      Canvas.LineTo(PageWidth,(AmountPrinted - LineHeight+TenthsOfInchPixelsY));
     end;
    {Increment AmountPrinted by the LineHeight }
    Canvas.MoveTo(0,AmountPrinted);
   end;

end;



{print field names as column headings}
procedure TPrintDlg.PrintColumnNames;
var
  ColNames: TStringList;
  iInc: integer;
begin

  ColNames := TStringList.Create;                                   {Create the string list }

  Printer.Canvas.Font.Assign(editCol.font);

  with ColNames do
   begin
    { Create the column headers }
    for iInc := 0 to iArraySize do
     begin
      if FColumns[iInc] <> '' then AddObject(FColumns[iInc],pointer(iWidth));
     end;
   end;

  PrintLine(ColNames);                                              {Print the line }
  ColNames.Free;                                                    {Free the string list }

end;

{print record from table}
procedure TPrintDlg.PrintLine(Items: TStringList);
var
  OutRect: TRect;
  Inches: double;
  i, iWidth, iPage: integer;
  b: boolean;
begin

  {initialise}
  OutRect.Left := 0;                                                {left position is zero }
  OutRect.Top := AmountPrinted;                                     {Set Top to Amount printed }
  OutRect.Bottom := OutRect.Top + LineHeight;                       {Set bottom position }
  iWidth := 0;
  iPage := 0;
  b := false;

  With Printer.Canvas do
    for i := 0 to (Items.Count - 1) do
     begin
      Inches := longint(Items.Objects[i]) * 0.1;                    {Get inches }
      OutRect.Right := OutRect.Left + round(PixelsInInchx * Inches);{Determine Right position }

      {check data against width of page}
      iWidth := outrect.right;
      iPage := printer.pageWidth;
      if iWidth > iPage then
       begin
        {show clipping message - bool determines if message has already been displayed}
        if not b then
         begin
          AbortDlg.lblTask.caption := 'Clipping data...';
          application.processMessages;
          messageBeep(1);
          b := true;
         end;
       end;

      if not Printer.Aborted then
       begin
         TextRect(OutRect, OutRect.Left, OutRect.Top, Items[i]);     {Print the line }
       end;
      OutRect.Left := OutRect.Right;                                {Set left to Right }
      iWidth := iWidth + OutRect.Left + round(PixelsInInchx * Inches);
     end;

  { Increment the amount printed }
  AmountPrinted := AmountPrinted + TenthsOfInchPixelsY * 2;

end;


{print footer}
procedure TPrintDlg.PrintFooter;
var
  OutRect: TRect;
  Inches: double;
  i: integer;
begin

  OutRect.Left := 0;                                                {left position is zero }
  OutRect.Top :=  Printer.PageHeight - LineHeight;                  {Set Top to one line less than page height}
  OutRect.Bottom := OutRect.Top + LineHeight;                       {Set bottom position }

  With Printer.Canvas do
   begin
    if not Printer.Aborted then
     TextRect(OutRect, OutRect.Left, OutRect.Top, (DateTimeToStr(Now) +', Page '+ IntToStr(iPage)));
   end;

  { Increment the amount printed }
  AmountPrinted := AmountPrinted + TenthsOfInchPixelsY * 2;

end;


{***buttons*********************************************************************}

{print}
procedure TPrintDlg.BitBtn2Click(Sender: TObject);
begin
 printTab;
end;

{setup}
procedure TPrintDlg.BitBtn1Click(Sender: TObject);
begin
 printerSetupDialog1.execute;
end;

{conditions}
procedure TPrintDlg.BitBtn5Click(Sender: TObject);
begin
 if QueryDlg.showModal = mrOK then
  begin
   FStrings := QueryDlg.lstFields.Items;
   FSQL := QueryDlg.FSQL;
   SetQuery;
   SetColumns;
  end;
end;


{close}
procedure TPrintDlg.BitBtn3Click(Sender: TObject);
begin
 close;
end;

{help}
procedure TPrintDlg.BitBtn4Click(Sender: TObject);
begin
 Application.HelpCommand(HELP_CONTEXT,HelpContext);
end;


{***form's preferences*********************************************************}

procedure TPrintDlg.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
 {restore settings}
 printer.orientation := FOrientation;
end;

end.
