{ Written by; Steve Trefethen

The following is a set of Delphi components to access (at least some of)
the networking dialogs that are build into WFW.  These components
provide access to both network drives and printers.

This file is provided "as is" without and warranty either expressed
or implied.  These components have been tested only on WFW 3.11.

These components may be distributed freely.  If you add to them or
enhance them I would appreciate it if you could send me a copy.

Enjoy!

}
unit NetDlgs;

interface

uses
  SysUtils, WinTypes, WinProcs, Classes, Forms;

type

  { TNetServerBrowseDialog }

  TNetServerBrowseDialog = class(TComponent)
  private
    FLocalName,
    FConnectName: PChar;
    NetDriver: THandle;
    WNETSERVERBROWSEDIALOG : Function(hwndParent : hWnd; lpszSectionName, lpszBuffer : PChar;
                                      cbBuffer : Word; dlFlags : LongInt) : Word;
    function GetLocalName: String;
    function GetConnectName: String;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function Execute: Boolean;
    property LocalName: String read GetLocalName;
    property ConnectName: String read GetConnectName;
  end;

  { TNetShareDialog }

  TNetShareDialog = class(TComponent)
  private
    FPrinter: Boolean;
    FPathName: PChar;
    NetDriver: THandle;
    function GetPathName: String;
    procedure SetPathName(Value: String);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function Execute: Boolean; virtual; abstract;
  published
    property PathName: String read GetPathName write SetPathName;
    property Printer: Boolean read FPrinter write FPrinter;
  end;

  { TNetShareAsDialog }

  TNetShareAsDialog = class(TNetShareDialog)
  private
    WNETSHAREASDIALOG : Function(hwndParent : HWND; iType : Word; lpszPath : PChar) : Word;
  public
    function Execute: Boolean; virtual;
  end;

  { TNetStopShareDialog }

  TNetStopShareDialog = class(TNetShareDialog)
  private
    WNETSTOPSHAREDIALOG : Function(hwndParent : Hwnd; iType : Word; lpszPath : PChar) : Word;
  public
    function Execute: Boolean;
  end;

  { TNetConnectDialog }

  TNetConnectDialog = class(TComponent)
  private
    FPrinter: Boolean;
    NetDriver: THandle;
    CONNECTDIALOG : Function(hwndParent : hWnd; iType : Word) : Word;
  public
    function Execute: Boolean;
  published
    property Printer: Boolean read FPrinter write FPrinter;
  end;

procedure Register;
Function NDdeGetNodeName(lpszNodeName : PChar; cchNodeName : LongInt) : Word;

implementation

const
  StringSize = 255;
  C_WNetShareAsDialog      = 141;
  C_WNetStopShareDialog    = 142;
  C_WNetServerBrowseDialog = 146;
  C_I_CONNECTIONDIALOG     = 533;

function CheckNetDriver: THandle;
const
  User = 'USER';
var
  UserHandle: THandle;
  WNetGetCaps: function (Flags: Word): Word;
  Handle: hWnd;
begin
  Result := 0;
  UserHandle := GetModuleHandle(User);
  if UserHandle <> 0 then begin
    @WNetGetCaps := GetProcAddress(UserHandle, 'WNETGETCAPS');
    if @WNetGetCaps <> nil then
      Result := WNetGetCaps(Word(-1))
    else
      raise Exception.Create('GetProcAddress failed');
  end else
    raise Exception.Create('GetModuleHandle failed');
end;

{ TNetServerBrowseDialog }

constructor TNetServerBrowseDialog.Create(AOwner: TComponent);
var
  S: String;
begin
  inherited Create(AOwner);
  FLocalName := nil;
  FConnectName := nil;
  NetDriver := 0;
end;

destructor TNetServerBrowseDialog.Destroy;
begin
  StrDispose(FLocalName);
  StrDispose(FConnectName);
  inherited Destroy;
end;

function TNetServerBrowseDialog.Execute: Boolean;
var
  CConnectName,
  CLocalName: array[0..StringSize + 1] of char;
  Handle: hWnd;
begin
  { Check to see if NETDDE.EXE is loaded }
  Handle := FindWindow('NetDDEMainWDW', 'NetDDE');
  if Handle = 0 then
  begin
    Handle := WinExec('NETDDE.EXE',sw_ShowNormal);  { Attempt to load it }
    if Handle < 32 then
      raise Exception.Create('Unable to load Network DDE support');
  end;
  if Handle <> 0 then
  begin
    NDDEGetNodeName(CLocalName, StringSize);
    { is network access enabled? }
    NetDriver := CheckNetDriver;
    if NetDriver <> 0 then
    begin
      { Then get the address of the Browse Dialog }
      @WNetServerBrowseDialog := GetProcAddress(NetDriver, MakeIntResource(C_WNetServerBrowseDialog));
      if Owner is TForm then
        if WNetServerBrowseDialog(TForm(Owner).Handle, CLocalName,
                                  CConnectName, StringSize, 0) <> 0 then
          Result := False
        else
        begin
          Result := True;
          FLocalName := StrAlloc(StrLen(CLocalName)+1);
          StrCopy(FLocalName, CLocalName);
          FConnectName := StrAlloc(StrLen(CConnectName)+1);
          StrCopy(FConnectName, CConnectName);
        end
      else
        raise Exception.Create('Invalid dialog owner handle.  Should be a TForm');
    end
    else
      raise Exception.Create('Net Driver handle not found');
  end;
end;

function TNetServerBrowseDialog.GetLocalName: String;
begin
  if FLocalName = nil then
    Result := ''
  else
    Result := StrPas(FLocalName);
end;

function TNetServerBrowseDialog.GetConnectName: String;
begin
  if FConnectName = nil then
    Result := ''
  else
    Result := StrPas(FConnectName);
end;

{ TNetShareDialog }

constructor TNetShareDialog.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FPathName := nil;
  NetDriver := 0;
end;

destructor TNetShareDialog.Destroy;
begin
  StrDispose(FPathName);
  inherited Destroy;
end;

function TNetShareDialog.GetPathName: String;
begin
  if FPathName = nil then
    Result := ''
  else
    Result := StrPas(FPathName);
end;

procedure TNetShareDialog.SetPathName(Value: String);
begin
  if FPathName <> nil then
    StrDispose(FPathName);
  FPathName := StrAlloc(Length(Value)+1);
  StrPCopy(FPathName, Value);
end;

{ TNetShareAsDialog }

function TNetShareAsDialog.Execute: Boolean;
begin
  if FPathName = nil then
    raise Exception.Create('No PathName specified');
  NetDriver := CheckNetDriver;
  if NetDriver <> 0 then
  begin
    { Then get the address of the Browse Dialog }
    @WNetShareAsDialog := GetProcAddress(NetDriver, MakeIntResource(C_WNetShareAsDialog));
    if Owner is TForm then
    begin
      if Printer then
        WNetShareAsDialog(TForm(Owner).Handle, 3, FPathName)
      else
        WNetShareAsDialog(TForm(Owner).Handle, 1, FPathName);
    end
    else
      raise Exception.Create('Invalid dialog owner handle.  Should be a TForm');
  end
  else
    raise Exception.Create('Net Driver handle not found');
end;

{ TNetStopShareDialog }

function TNetStopShareDialog.Execute: Boolean;
begin
  NetDriver := CheckNetDriver;
  if NetDriver <> 0 then
  begin
    { Then get the address of the Browse Dialog }
    @WNetStopShareDialog := GetProcAddress(NetDriver, MakeIntResource(C_WNetStopShareDialog));
    if (Owner is TForm) then
    begin
      if Printer then
        WNetStopShareDialog(TForm(Owner).Handle, 3, FPathName)
      else
        WNetStopShareDialog(TForm(Owner).Handle, 1, FPathName);
    end
    else
      raise Exception.Create('Invalid dialog owner handle.  Should be a TForm');
  end
  else
    raise Exception.Create('Net Driver handle not found');
end;

{ TNetConnectDialog }

function TNetConnectDialog.Execute: Boolean;
begin
  NetDriver := CheckNetDriver;
  if NetDriver <> 0 then
  begin
    { Then get the address of the Browse Dialog }
    @ConnectDialog := GetProcAddress(NetDriver, 'WNETCONNECTDIALOG');
    if Owner is TForm then
    begin
      if Printer then
        ConnectDialog(TForm(Owner).Handle, 3)  { 1 = Drive 3 = Printer }
      else
        ConnectDialog(TForm(Owner).Handle, 1);
    end
    else
      raise Exception.Create('Invalid dialog owner handle.  Should be a TForm');
  end
  else
    raise Exception.Create('Net Driver handle not found');
end;

procedure Register;
begin
  RegisterComponents('Dialogs', [TNetServerBrowseDialog,TNetShareAsDialog,
                                 TNetStopShareDialog,TNetConnectDialog]);
end;

Function NDdeGetNodeName; external 'NDDEAPI' Index 106;

end.
