unit MBUtils;

{ compiler options here }

interface

uses
   SysUtils, WinTypes, WinProcs, Messages, Graphics, Controls,
   Forms, Classes, DsgnIntf, StdCtrls, Dialogs, MMSystem,
   DbiTypes, DbiProcs, DbiErrs, DB, DBTables;

{const declarations here }

type

   {Exceptions for MBUtils}
   EMBCounter = class(Exception);
   EMBRecordLock = class(Exception);

   MBError = (mbeSuccess, mbeNotActive, mbeCounterNotFound, mbeBusy,
      mbeNameExists, mbeAtMax, mbeAtMin, mbeParameterError);

   {TMBCounter}
   TMBCounter = class(TComponent)
   private
      { Private declarations }
      FCounterTable : TTable;
      FCounterKeyValue : String;    {key for the current counter}
      FCounterValueField : String;  {field name of the current value field}
      FIncrementValueField : String;{field name of the increment value field}
      FCounterKeyField: String;     {field name of the counter key value field}
      FMaxValueField: String;       {field name of the max allowed value field}
      FMinValueField: String;       {field name of the min allowed value field}
      FRetryAttempts: Integer;      {maximum times will try for record lock}
      FRetryWaitMin: LongInt;       {minimun number of MS to wait 4 retry}
      FRetryWaitMax: LongInt;       {maximum number of MS to wait 4 retry}
      FLastError: MBError;
      function GetActive: Boolean;
      procedure SetActive(const Value: Boolean);
      function GetDatabaseName: TSymbolStr;
      procedure SetDatabaseName(const Value: TSymbolStr);
      function GetTableName: TFileName;
      procedure SetTableName(const Value: TFileName);
      function GetIndexName: TIndexName;
      procedure SetIndexName(const Value: TIndexName);
      function GetIndexFieldNames: String;
      procedure SetIndexFieldNames(const Value: String);
  protected
    { Protected declarations }

  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

  published
    { Published declarations }
    property DatabaseName: TSymbolStr read GetDatabaseName
       write SetDatabaseName;
    property TableName: TFileName read GetTableName write SetTableName;
    property IndexFieldNames: String read GetIndexFieldNames
       write SetIndexFieldNames;
    property IndexName: TIndexName read GetIndexName write SetIndexName;
    property CounterKeyValue: String read fCounterKeyValue
       write fCounterKeyValue;
    property Active: Boolean read GetActive write SetActive;
    property CounterKeyField: String read FCounterKeyField
       write FCounterKeyField;
    property IncrementValueField: String read FIncrementValueField
       write FIncrementValueField;
    property MaxValueField: String read FMaxValueField write FMaxValueField;
    property MinValueField: String read FMinValueField write FMinValueField;
    property CounterValueField: String read FCounterValueField
       write fCounterValueField;
    property RetryWaitMin: LongInt read FRetryWaitMin write FRetryWaitMin
       default 500;
    property RetryWaitMax: LongInt read FRetryWaitMax write FRetryWaitMax
       default 1000;
    property RetryAttempts: Integer read FRetryAttempts write FRetryAttempts
       default 5;
    procedure createCounter(const initialValue: LongInt;
       const increment: LongInt; const minValue: LongInt;
       const maxValue: LongInt);
    procedure getCounterInfo(var currentValue: LongInt; var increment: LongInt;
       var minValue: LongInt; var maxValue: LongInt);
    procedure setCounterInfo(const currentValue: LongInt;
       const increment: LongInt; const minValue: LongInt;
       const maxValue: LongInt);
    function isCounter: Boolean;
    function nextCounterValue: LongInt;
    function RollBackValue(const Value: LongInt): Boolean;
    function lastCounterError: MBError;
  end; {TMBCounter}

procedure MBSleep(const SleepTime: LongInt; const RandomTime: LongInt);

procedure MBLockRecord(var Table: TTable; const MaxRetry, MinWaitMS,
   MaxWaitMS: LongInt);

{ Sleep procedure implemented with a program loop }

procedure Register;

implementation

{ TMBCounter }

constructor TMBCounter.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   FCounterTable := TTable.Create(Self);
   FRetryAttempts:= 5;
   FRetryWaitMin:= 500;
   FRetryWaitMax:= 1000;
end;

destructor TMBCounter.Destroy;
begin
   FCounterTable.Free;
   inherited Destroy;
end;

procedure TMBCounter.SetDatabaseName(const Value: TSymbolStr);
begin
   FCounterTable.DatabaseName := Value;
end;

function TMBCounter.GetDatabaseName: TSymbolStr;
begin
   Result := FCounterTable.DatabaseName;
end;

procedure TMBCounter.SetIndexName(const Value: TIndexName);
begin
   FCounterTable.IndexName := Value;
end;

function TMBCounter.GetIndexName: TIndexName;
begin
   Result := FCounterTable.IndexName;
end;

procedure TMBCounter.SetIndexFieldNames(const Value: string);
begin
   FCounterTable.IndexFieldNames := Value;
end;

function TMBCounter.GetIndexFieldNames: String;
begin
   Result := FCounterTable.IndexFieldNames;
end;

procedure TMBCounter.SetActive(const Value: Boolean);
begin
   FCounterTable.Active := Value;
end;

function TMBCounter.GetActive: Boolean;
begin
   Result := FCounterTable.Active;
end;

procedure TMBCounter.SetTableName(const Value: TFileName);
begin
   FCounterTable.TableName := Value;
end;

function TMBCounter.GetTableName: TFileName;
begin
   Result := FCounterTable.TableName;
end;

function TMBCounter.LastCounterError: MBError;
begin
   Result := FLastError;
end;

function TMBCounter.RollBackValue(const Value: LongInt): Boolean;
var
   newValue: LongInt;
begin
   { Verify the counter key field name }
   if Length(FCounterKeyField) = 0 then
      begin
         FLastError := mbeParameterError;
         raise EMBCounter.Create('The CounterKeyField cannot be blank!');
      end;
   { Verify the Counter field name }
   if Length(FCounterValueField) = 0 then
      begin
         FLastError := mbeParameterError;
         raise EMBCounter.Create('The CounterValueField cannot be blank!');
      end;
   { See if the Counter Key Value exists }
   if isCounter then
      begin
         { Try to get a record lock }
         MBLockRecord(FCounterTable, FRetryAttempts, FRetryWaitMin,
            FRetryWaitMax);
         { verify the specified value }
         if Value = FCounterTable.FieldByName(FCounterValueField).AsInteger then
            begin
               { calc the prior value }
               if Length(FIncrementValueField) > 0 then
                  newValue :=
                     FCounterTable.FieldByName(FCounterValueField).AsInteger -
                     FCounterTable.FieldByName(FIncrementValueField).AsInteger
               else
                  newValue :=
                     FCounterTable.FieldByName(FCounterValueField).AsInteger -
                     1;
               FCounterTable.FieldByName(FCounterValueField).AsInteger :=
                  NewValue;
               FCounterTable.Post;
               FLastError := mbeSuccess;
               Result := True;
            end
         else
            begin
               FCounterTable.Cancel;
               Result := False;
            end;
         end
      else
         begin
             FLastError := mbeCounterNotFound;
             raise EMBCounter.Create('The specified counter was not found: '+
                FCounterKeyValue);
         end;
end;

function TMBCounter.NextCounterValue: LongInt;
begin
   { Verify the counter key field name }
   if Length(FCounterKeyField) = 0 then
      begin
         FLastError := mbeParameterError;
         raise EMBCounter.Create('The CounterKeyField cannot be blank!');
      end;
   { Verify the Counter field name }
   if Length(FCounterValueField) = 0 then
      begin
         FLastError := mbeParameterError;
         raise EMBCounter.Create('The CounterValueField cannot be blank!');
      end;
   { See if the Counter Key Value exists }
   if isCounter then
      begin
         MBLockRecord(FCounterTable, FRetryAttempts, FRetryWaitMin,
            FRetryWaitMax);
         { calc the next value }
         if Length(FIncrementValueField) > 0 then
            Result := FCounterTable.FieldByName(FCounterValueField).AsInteger +
               FCounterTable.FieldByName(FIncrementValueField).AsInteger
         else
            Result := FCounterTable.FieldByName(FCounterValueField).AsInteger +
               1;
         { check if that value exceeds the max if a max is specified }
         if length(FMaxValueField) > 0 then
            if Result > FCounterTable.FieldByName(FMaxValueField).AsInteger then
               begin
                  FCounterTable.Cancel;
                  result := 0;
                  FLastError := mbeAtMax;
                  raise EMBCounter.Create('The next value would exceed the' +
                     ' maximum value for that counter: ' + FCounterKeyValue);
               end;
         if length(FMinValueField) > 0 then
            if Result < FCounterTable.FieldByName(FMinValueField).AsInteger then
               begin
                  FCounterTable.Cancel;
                  result := 0;
                  FLastError := mbeAtMin;
                  raise EMBCounter.Create('The next value would exceed the' +
                     ' minimum value for that counter: ' + FCounterKeyValue);
               end;
         FCounterTable.FieldByName(FCounterValueField).AsInteger := Result;
         FCounterTable.Post;
         FLastError := mbeSuccess;
      end { if gotoKey then}
   else
      begin
          FLastError := mbeCounterNotFound;
          raise EMBCounter.Create('The specified counter was not found: '+
             FCounterKeyValue);
      end;
end;

procedure TMBCounter.getCounterInfo(var currentValue: LongInt;
   var increment: LongInt; var minValue: LongInt; var maxValue: LongInt);
begin
   if isCounter then
      begin
         if Length(FCounterValueField) > 0 then
            currentValue :=
               FCounterTable.FieldByName(FCounterValueField).AsInteger
         else
            currentValue := 0;
         if Length(FIncrementValueField) > 0 then
            increment :=
               FCounterTable.FieldByName(FIncrementValueField).AsInteger
         else
            increment := 0;
         if Length(FMinValueField) > 0 then
            minValue := FCounterTable.FieldByName(FMinValueField).AsInteger
         else
            minValue := 0;
         if Length(FMaxValueField) > 0 then
            maxValue := FCounterTable.FieldByName(FMaxValueField).AsInteger
         else
            maxValue := 0;
         FLastError:= mbeSuccess;
      end { if isCounter else }
   else
      begin
          FLastError := mbeCounterNotFound;
          raise EMBCounter.Create('The specified counter was not found: '+
             FCounterKeyValue);
      end;
end;

procedure TMBCounter.createCounter(const initialValue: LongInt;
   const increment: LongInt; const minValue: LongInt; const maxValue: LongInt);
begin
   if isCounter then
      raise EMBCounter.Create('Counter Name already exists: ' +
         FCounterKeyValue)
   else
      begin
         FCounterTable.Insert;
         if Length(FCounterKeyField) > 0 then
            FCounterTable.FieldByName(FCounterKeyField).AsString :=
               UpperCase(FCounterKeyValue);
         if Length(FCounterValueField) > 0 then
            FCounterTable.FieldByName(FCounterValueField).AsInteger :=
               initialValue;
         if Length(FIncrementValueField) > 0 then
            FCounterTable.FieldByName(FIncrementValueField).AsInteger :=
               increment;
         if Length(FMinValueField) > 0 then
            FCounterTable.FieldByName(FMinValueField).AsInteger := minValue;
         if Length(FMaxValueField) > 0 then
            FCounterTable.FieldByName(FMaxValueField).AsInteger := maxValue;
         FCounterTable.Post;
         FLastError := mbeSuccess;
      end; { if isCounter else }
end;

procedure TMBCounter.setCounterInfo(const CurrentValue: LongInt;
   const increment: LongInt; const minValue: LongInt; const maxValue: LongInt);
begin
   if isCounter then
      begin
         MBLockRecord(FCounterTable, FRetryAttempts, FRetryWaitMin,
            FRetryWaitMax);
         FCounterTable.FieldByName(FCounterValueField).AsInteger :=
            currentValue;
         FCounterTable.FieldByName(FIncrementValueField).AsInteger := increment;
         FCounterTable.FieldByName(FMinValueField).AsInteger := minValue;
         FCounterTable.FieldByName(FMaxValueField).AsInteger := maxValue;
         FCounterTable.Post;
         FLastError := mbeSuccess;
      end { if isCounter }
   else
      begin
          FLastError := mbeCounterNotFound;
          raise EMBCounter.Create('The specified counter was not found: '+
             FCounterKeyValue);
      end;
end;

function TMBCounter.isCounter: Boolean;
begin
   if FCounterTable.Active then
      begin
         FCounterTable.setKey;
         FCounterTable.FieldByName(FCounterKeyField).AsString :=
            UpperCase(FCounterKeyValue);
         if FCounterTable.gotoKey then
            Result := TRUE
         else
            Result := FALSE;
         FLastError := mbeSuccess;
      end { if table.Active then }
   else
      begin
         FLastError := mbeNotActive;
         raise EMBCounter.Create('The counter table is not active: ' +
            FCounterTable.TableName + '(' + name + ')');
      end;
end;

{ End TMBCounter }

procedure MBSleep(const SleepTime: LongInt; const RandomTime: LongInt);
var
  MSStart, MSEnd: LongInt;
  targetTime: LongInt;
begin
   MSStart := timeGetTime;
   { Validate parameters }
   if ((SleepTime > 0) and (RandomTime >= 0)) then
      begin
        if RandomTime = 0 then
            targetTime := SleepTime
         else
            begin
               {Adjust the specified time by +/- RandomTime}
               randomize;
               targetTime := SleepTime - RandomTime + random(2 * RandomTime);
            end; {else begin }
         while TRUE do
            begin
               Application.ProcessMessages; {Be a polite windows user}
               MSEnd := timeGetTime; {Check elapsed MS}
               if (MSEnd - MSStart) >= targetTime then break;
            end; {while true do }
      end; {if sleeptime ... }
end;

procedure MBLockRecord(var Table: TTable; const MaxRetry, MinWaitMS,
   MaxWaitMS: LongInt);
var
   lockAttempts: Integer;
   WaitMS, RandomMS: LongInt;
   errInfo: DbiErrInfo;
begin
   lockAttempts := 1; {Initialize the lock attempts for this try}
   while True do
      try
         Table.edit;
         break;
      except
         on E:EDatabaseError do
            begin
               { Check the error code for error category ERRCAT_LOCKCONFLICT}
               DbiGetErrorInfo(True, errInfo);
               if (ErrCat(errInfo.IError) <> ERRCAT_LOCKCONFLICT) then
                  raise { EDatabaseError wasn't for a record lock }
               else
                  begin
                     if lockAttempts > MaxRetry then
                        begin
                           raise EMBRecordLock.Create('Retry Count Exceeded: ' +
                              E.Message)
                        end
                     else
                        begin
                           { convert from min/max MS to avg/random MS }
                           WaitMS := (MinWaitMS + MaxWaitMS) div 2;
                           RandomMS := (MaxWaitMS - MinWaitMS) div 2;
                           MBSleep(WaitMS, RandomMS);
                           lockAttempts := lockAttempts + 1;
                       end;
                  end; { errCat <> LockConflict }
             end; { on E: EDatabaseError do }
      end; { try...except }
end;

{ TDBStringProperty }

type
  TDBStringProperty = class(TStringProperty)
  public
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValueList(List: TStrings); virtual; abstract;
    procedure GetValues(Proc: TGetStrProc); override;
  end;

function TDBStringProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paValueList, paSortList, paMultiSelect];
end;

procedure TDBStringProperty.GetValues(Proc: TGetStrProc);
var
  I: Integer;
  Values: TStringList;
begin
  Values := TStringList.Create;
  try
    GetValueList(Values);
    for I := 0 to Values.Count - 1 do Proc(Values[I]);
  finally
    Values.Free;
  end;
end;

{ TDatabaseNameProperty }

type
  TDatabaseNameProperty = class(TDBStringProperty)
  public
    procedure GetValueList(List: TStrings); override;
  end;

procedure TDatabaseNameProperty.GetValueList(List: TStrings);
var
  oldCursor, waitCursor: HCursor;
begin
  waitCursor := LoadCursor(0, IDC_WAIT);
  OldCursor := SetCursor(waitCursor);
  Session.GetDatabaseNames(List);
  SetCursor(OldCursor);
end;

{ TTableNameProperty }

type
  TTableNameProperty = class(TDBStringProperty)
  public
    procedure GetValueList(List: TStrings); override;
  end;

procedure TTableNameProperty.GetValueList(List: TStrings);
var
  Table: TTable;
  MBCounter: TMBCounter;
  oldCursor, waitCursor: HCursor;
begin
  MBCounter := GetComponent(0) as TMBCounter;
  Table := TTable.Create(NIL);
  TRY
    waitCursor := LoadCursor(0, IDC_WAIT);
    OldCursor := SetCursor(waitCursor);
    Session.GetTableNames(MBCounter.DatabaseName, '', True, False, List);
    SetCursor(OldCursor);
  FINALLY
     Table.Free;
  END;
end;

{ TIndexNameProperty }

type
  TIndexNameProperty = class(TDBStringProperty)
  public
    procedure GetValueList(List: TStrings); override;
  end;

procedure TIndexNameProperty.GetValueList(List: TStrings);
var
  Table: TTable;
  MBCounter: TMBCounter;
  oldCursor, waitCursor: HCursor;
begin
  MBCounter := GetComponent(0) as TMBCounter;
  Table := TTable.Create(NIL);
  TRY
    Table.DatabaseName := MBCounter.DatabaseName;
    if MBCounter.TableName <> '' then
    begin
       Table.TableName := MBCounter.TableName;
       waitCursor := LoadCursor(0, IDC_WAIT);
       OldCursor := SetCursor(waitCursor);
       Table.GetIndexNames(List);
       SetCursor(OldCursor);
    end;
  FINALLY
    Table.Free;
  END;
end;

{ TIndexFieldNamesProperty }

type
  TIndexFieldNamesProperty = class(TDBStringProperty)
  public
    procedure GetValueList(List: TStrings); override;
  end;

procedure TIndexFieldNamesProperty.GetValueList(List: TStrings);
var
  Table: TTable;
  MBCounter: TMBCounter;
  oldCursor, waitCursor: HCursor;
  I: Integer;
begin
  MBCounter := GetComponent(0) as TMBCounter;
  Table := TTable.Create(NIL);
  TRY
    Table.DatabaseName := MBCounter.DatabaseName;
    if MBCounter.TableName <> '' then
    begin
       Table.TableName := MBCounter.TableName;
       waitCursor := LoadCursor(0, IDC_WAIT);
       OldCursor := SetCursor(waitCursor);
       with Table do
       begin
         IndexDefs.Update;
         for I:= 0 to IndexDefs.Count - 1 do
           with IndexDefs[I] do
             if not (ixExpression in Options) then List.Add(Fields);
       end;
       SetCursor(OldCursor);
    end;
  FINALLY
    Table.Free;
  END;
end;

{ TFieldNameProperty }

type
  TFieldNameProperty = class(TDBStringProperty)
  public
    procedure GetValueList(List: TStrings); override;
  end;

procedure TFieldNameProperty.GetValueList(List: TStrings);
var
  Table: TTable;
  MBCounter: TMBCounter;
  oldCursor, waitCursor: HCursor;
begin
  MBCounter := GetComponent(0) as TMBCounter;
  Table := TTable.Create(NIL);
  TRY
    Table.DatabaseName := MBCounter.DatabaseName;
    if MBCounter.TableName <> '' then
    begin
      Table.TableName := MBCounter.TableName;
      waitCursor := LoadCursor(0, IDC_WAIT);
      OldCursor := SetCursor(waitCursor);
      Table.GetFieldNames(List);
      SetCursor(OldCursor);
    end;
  FINALLY
    Table.Free;
  END;
end;

{ TCounterKeyValueProperty }

type
  TCounterKeyValueProperty = class(TDBStringProperty)
  public
    procedure GetValueList(List: TStrings); override;
  end;

procedure TCounterKeyValueProperty.GetValueList(List: TStrings);
var
  Table: TTable;
  MBCounter: TMBCounter;
  oldCursor, waitCursor: HCursor;
begin
  MBCounter := GetComponent(0) as TMBCounter;
  Table := TTable.Create(NIL);
  TRY
    Table.DatabaseName := MBCounter.DatabaseName;
    if MBCounter.TableName <> '' then
    begin
       Table.TableName := MBCounter.TableName;
       Table.IndexName := MBCounter.IndexName;
       Table.IndexFieldNames := MBCounter.IndexFieldNames;
       waitCursor := LoadCursor(0, IDC_WAIT);
       OldCursor := SetCursor(waitCursor);
       with Table do
       begin
         Open;
         First;
         While not eof do
         begin
           if MBCounter.CounterKeyField = '' then
             List.Add(Fields[0].AsString)
           else
             List.Add(FieldByName(MBCounter.CounterKeyField).asString);
           next;
         end;
       end;
       SetCursor(OldCursor);
    end;
  FINALLY
    Table.Free;
  END;
end;

{ Component Registration }

procedure Register;
begin
  RegisterComponents('MBUtils', [TMBCounter]);
  RegisterPropertyEditor(TypeInfo(TSymbolStr), TMBCounter, 'DatabaseName',
    TDatabaseNameProperty);
  RegisterPropertyEditor(TypeInfo(TFileName), TMBCounter, 'TableName',
    TTableNameProperty);
  RegisterPropertyEditor(TypeInfo(TIndexName), TMBCounter, 'IndexName',
    TIndexNameProperty);
  RegisterPropertyEditor(TypeInfo(String), TMBCounter, 'IndexFieldNames',
    TIndexFieldNamesProperty);
  RegisterPropertyEditor(TypeInfo(String), TMBCounter, 'CounterKeyField',
    TFieldNameProperty);
  RegisterPropertyEditor(TypeInfo(String), TMBCounter, 'IncrementValueField',
    TFieldNameProperty);
  RegisterPropertyEditor(TypeInfo(String), TMBCounter, 'MinValueField',
    TFieldNameProperty);
  RegisterPropertyEditor(TypeInfo(String), TMBCounter, 'MaxValueField',
    TFieldNameProperty);
  RegisterPropertyEditor(TypeInfo(String), TMBCounter, 'CounterValueField',
    TFieldNameProperty);
  RegisterPropertyEditor(TypeInfo(String), TMBCounter, 'CounterKeyValue',
    TCounterKeyValueProperty);
end;

end.
