unit Dboutlin;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, Grids, Outline, DB, DBTables, StdCtrls, Menus ;

type
  TDBOutline = class ;

  TOutlineLink = class ( TDataLink )
  private
    FOutline : TDBOutline ;
  protected
    procedure ActiveChanged; override;
    procedure DataSetChanged; override;
    procedure DataSetScrolled ( Distance : Integer ) ; override ;
    procedure RecordChanged(Field: TField); override;
  public
    constructor Create(AOutline: TDBOutline);
  end;


  TDBOutline = class(TCustomOutline)
  private
    FOutlineLink : TOutlineLink ;
    FTableIDField : string ;
    FTableParentField : string ;
    FTableTextField : string ;
    FRootID : LongInt ;
    FAutoExpand : Boolean ;
    FOnClosedLoop : TNotifyEvent ;
    FBuilding : Boolean ;
    FClicking : Boolean ;
    FPrevState : TDataSetState ;
    FIndexName : string ;
    FIndexFields : string ;
    FDelRootID : LongInt ;
    function  GetDataSource : TDataSource ;
    procedure SetDataSource ( ADataSource : TDataSource ) ;
    function  GetTable : TTable ;
    procedure SetRootID ( ID : LongInt ) ;
    function  NeedRebuild : Boolean ;
    procedure Building ( AIndex, AParent : LongInt ) ;
    procedure CreateTree ( AIndex, AParent : LongInt ) ;
    function  AddItemTo ( AIndex : Integer ) : LongInt ;
    function  GetTableIDIndex : LongInt ;
    function  GetID ( AIndex : LongInt ) : LongInt ;
    function  GetSelectedID : LongInt ;
    procedure IndexChanged ;
    procedure Execute ( AParent : LongInt ) ;
  protected
    property  OutlineLink : TOutlineLink read FOutlineLink ;
    procedure ActiveChanged ( Value : Boolean ) ;
    procedure DataChanged;
    procedure BuildTree ;
    procedure RecordNumberChanged ;
    procedure RecordChanged(Field: TField);
    procedure Notification(AComponent: TComponent;
                           Operation: TOperation); override;
    procedure Click ; override ;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override ;
    procedure ClosedLoop ;
  public
    constructor Create ( AOwner : TComponent ) ; override ;
    destructor  Destroy ; override ;
    procedure   AddTo ( AParent : LongInt ) ;
    procedure   DeleteWithChildren ( AParent : LongInt ) ;
    property  Table : TTable read GetTable ;
    property  SelectedID : LongInt read GetSelectedID ;
    property  ID [Index: LongInt]: LongInt read GetID ;
  published
    property DataSource : TDataSource read GetDataSource write SetDataSource ;
    property TableIDField : string read FTableIDField write FTableIDField ;
    property TableParentField : string read FTableParentField write FTableParentField ;
    property TableTextField : string read FTableTextField write FTableTextField ;
    property RootID : LongInt read FRootID write SetRootID default 0 ;
    property AutoExpand : Boolean read FAutoExpand write FAutoExpand default True ;
    property OnClosedLoop : TNotifyEvent read FOnClosedLoop write FOnClosedLoop ;

    property OutlineStyle;
    property OnExpand;
    property OnCollapse;
    property Options;
    property Style;
    property ItemHeight;
    property OnDrawItem;
    property Align;
    property Enabled;
    property Font;
    property Color;
    property ParentColor;
    property ParentCtl3D;
    property Ctl3D;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClick;
    property DragMode;
    property DragCursor;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnDblClick;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property BorderStyle;
    property ItemSeparator;
    property PicturePlus;
    property PictureMinus;
    property PictureOpen;
    property PictureClosed;
    property PictureLeaf;
    property ParentFont;
    property ParentShowHint;
    property ShowHint;
    property PopupMenu;
    property ScrollBars;
  end;

procedure Register;

implementation

type
  TData = class
  public
    ID : LongInt ;
    constructor Create ( AID : LongInt ) ;
  end;

constructor TData.Create ( AID : LongInt ) ;
begin
  inherited Create ;
  ID := AID ;
end;

constructor TOutlineLink.Create ( AOutline : TDBOutline ) ;
begin
  inherited Create ;
  FOutline := AOutline ;
end;

procedure TOutlineLink.ActiveChanged;
begin
  FOutline.ActiveChanged ( Active ) ;
end;

procedure TOutlineLink.DataSetChanged;
begin
  FOutline.DataChanged ;
end;

procedure TOutlineLink.DataSetScrolled ( Distance : Integer ) ;
begin
  FOutline.RecordNumberChanged ;
end;

procedure TOutlineLink.RecordChanged(Field: TField);
begin
  FOutline.RecordChanged(Field);
end;

{ TDBOutline }

constructor TDBOutline.Create ( AOwner : TComponent ) ;
begin
  inherited Create ( AOwner ) ;
  FOutlineLink := TOutlineLink.Create ( Self ) ;
  FAutoExpand := True ;
  FBuilding := False ;
  FClicking := False ;
  FIndexName := '' ;
  FIndexFields := '' ;
  FPrevState := dsInactive ;
end;

destructor TDBOutline.Destroy ;
begin
  Clear ;
  FOutlineLink.Free ;
  FOutlineLink := nil ;
  inherited Destroy ;
end;

function TDBOutline.GetDataSource: TDataSource;
begin
  Result := FOutlineLink.DataSource;
end;

procedure TDBOutline.SetDataSource(ADataSource: TDataSource);
begin
  FOutlineLink.DataSource := ADataSource;
end;

function  TDBOutline.GetTable : TTable ;
begin
  try
    Result := FOutLineLink.DataSet as TTable ;
  except
  on EInvalidCast do raise Exception.Create ( 'TTable required for TDBOutline' ) ;
  end;
end;

procedure TDBOutline.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (FOutlineLink <> nil) and
    (AComponent = DataSource) then DataSource := nil;
end;

procedure TDBOutline.ActiveChanged ( Value : Boolean ) ;
begin
  if not Value then
  begin
    Clear ;
    FIndexName := '' ;
    FIndexFields := '' ;
    FPrevState := dsInactive ;
  end
  else
  begin
    FIndexFields := Table.IndexFieldNames ;
    FIndexName := Table.IndexName ;
    FPrevState := Table.State ;
    BuildTree ;
  end;
end;

procedure TDBOutline.BuildTree ;
begin
  if not NeedRebuild then Exit ;
  Clear ;
  Building ( 0, FRootID ) ;
  if FAutoExpand then FullExpand ;
  RecordNumberChanged ;
end;

function TDBOutline.NeedRebuild : Boolean ;
begin
{ Function BuildTree produce some side effects by changing
  following properties, fields :
  1. Table's current record ;
  2. Table.IndexFieldNames ;
  3. Table.State -> dsBrowse ;
  4. Table's buffer for searching;

  There are no ways retain and later restore State
  and search buffer ( without using private methods ).
  Therefore Building method is called only in Browse state,
  and when not used GoToKey method.

}
  Result := False ;
  if FBuilding or FClicking then Exit ;
  with Table do
  begin
    if ( UpperCase (FIndexFields) = UpperCase(IndexFieldNames) ) and
       ( UpperCase (FIndexName) = UpperCase(IndexName) )
    then
      if State = dsBrowse then Result := True ;
    FIndexFields := IndexFieldNames ;
    FIndexName := IndexName ;
    if (FPrevState = dsSetKey) and (State = dsBrowse) then
      Result := False ;       {  Catch calls of GoToKey method }
    FPrevState := State ;
  end;
end;

procedure TDBOutline.Building ( AIndex, AParent : LongInt ) ;
var
  BookMark : TBookMark ;
begin
  with Table do
  try
    FBuilding := True ;
    IndexFieldNames := FTableParentField ;
    BookMark := GetBookMark ;
    try
      DisableControls ;
      CreateTree ( AIndex, AParent ) ;
    finally
      GoToBookMark ( BookMark ) ;
      FreeBookMark ( BookMark ) ;
      EnableControls ;
    end;
  finally
    if FIndexName <> '' then
      IndexName := FIndexName
    else
      IndexFieldNames := FIndexFields ;
    FBuilding := False ;
  end;
end;

procedure TDBOutline.CreateTree ( AIndex, AParent : LongInt ) ;
var
  idx, i : LongInt ;
  List : TList ;
begin
  List := TList.Create ;
  with Table do
  try
    SetKey ;
    FieldByName ( FTableParentField ).AsInteger := AParent ;
    GoToKey ;
    while (FieldByName ( FTableParentField ).AsInteger = AParent)
          and not EOF do
    begin
      if FieldByName(FTableIDField).AsInteger = FRootID then
        ClosedLoop
      else
      begin
        idx := AddItemTo ( AIndex ) ;
        List.Add ( Items[idx] ) ;
      end;
      Next ;
    end;

    for i := 0 to List.Count - 1 do
    begin
      idx := TOutlineNode ( List[i] ).Index ; ;
      CreateTree ( idx, LongInt(Items[idx].Data) ) ;
    end;
  finally
    List.Destroy ;
  end;
end;

procedure TDBOutline.ClosedLoop ;
begin
  if Assigned ( FOnClosedLoop ) then
    FOnClosedLoop ( Self ) ;
end;

function TDBOutline.AddItemTo ( AIndex : Integer ) : LongInt ;
begin
  with Table do
    Result := AddChildObject ( AIndex, FieldByName ( FTableTextField ).AsString,
      Pointer ( FieldByName ( FTableIDField ).AsInteger ) ) ;
end;

function TDBOutline.GetTableIDIndex : LongInt ;
begin
  Result := GetDataItem ( Pointer(Table.FieldByName ( FTableIDField ).AsInteger) ) ;
end;

procedure TDBOutline.DataChanged;
begin
  BuildTree ;
end;

procedure TDBOutline.RecordNumberChanged ;
var
  AIndex : LongInt ;
begin
  if FBuilding then Exit ;

  AIndex := GetTableIDIndex ;
  if AIndex = 0 then Exit ;

  SelectedItem := AIndex ;
  if FAutoExpand and not Items[SelectedItem].IsVisible then
  begin
    Items[SelectedItem].FullExpand ;
    SelectedItem := AIndex ;
  end;
end;

procedure TDBOutline.RecordChanged(Field: TField);
var
  IDIndex : LongInt ;
begin
  if FBuilding or FClicking then Exit ;
  if Field = nil then Exit ;

  IDIndex := GetTableIDIndex ;
  if UpperCase(Field.FieldName) = UpperCase(FTableTextField) then
  begin
    if IDIndex <> 0 then
      Items[IDIndex].Text := Table.FieldByName(FTableTextField).AsString ;
  end;
end;

procedure TDBOutline.Click ;
begin
  inherited Click ;
  IndexChanged ;
end;

procedure TDBOutline.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited KeyDown(Key,Shift) ;
  IndexChanged ;
end;

procedure TDBOutline.IndexChanged ;
begin
  if SelectedItem > 0 then
    with Table do
    begin
      FClicking := True ;
      IndexFieldNames := FTableIDField ;
      SetKey ;
      FieldByName ( FTableIDField ).AsInteger := SelectedID ;
      GoToKey ;
      if FIndexName <> '' then
        IndexName := FIndexName
      else
        IndexFieldNames := FIndexFields ;
      FClicking := False ;
    end;
end;

procedure TDBOutline.SetRootID ( ID : LongInt ) ;
begin
  if FRootID <> ID then
  begin
    FRootID := ID ;
    if FOutlineLink.Active then
      BuildTree ;
  end;
end;

function TDBOutline.GetID ( AIndex : LongInt ) : LongInt ;
begin
  Result := FRootID ;
  if AIndex > 0 then
    Result := LongInt ( Items[AIndex].Data ) ;
end;

function TDBOutline.GetSelectedID : LongInt ;
begin
  Result := ID[SelectedItem] ;
end;

procedure TDBOutline.AddTo ( AParent : LongInt ) ;
begin
  with Table do
  begin
    Append ;
    FieldByName ( FTableParentField ).AsInteger := AParent ;
    Post ;
    Edit ;
  end;
end;

procedure TDBOutline.DeleteWithChildren ( AParent : LongInt ) ;
begin
  FDelRootID := AParent ;
  try
    Table.DisableControls ;
    Execute ( AParent ) ;
  finally
    Table.EnableControls ;
  end;
end;

procedure TDBOutline.Execute ( AParent : LongInt ) ;
var
  List : TList ;
  i : Integer ;
begin
  List := TList.Create ;

  with Table do
  try
    IndexFieldNames := FTableParentField ;
    SetKey ;
    FieldByName ( FTableParentField ).AsInteger := AParent ;
    GoToKey ;
    while (FieldByName ( FTableParentField ).AsInteger = AParent)
          and not EOF do
    begin
      if FieldByName(FTableIDField).AsInteger = FDelRootID then
        ClosedLoop
      else
        List.Add ( TData.Create( FieldByName(FTableIDField).AsInteger ) ) ;
      Next ;
    end;

    IndexFieldNames := FTableIDField ;
    SetKey ;
    FieldByName ( FTableIDField ).AsInteger := AParent ;
    if GoToKey then
      Delete ;

    for i := 0 to List.Count -1 do
      Execute ( TData(List[i]).ID ) ;
  finally
    List.Destroy ;
  end;
end;

procedure Register;
begin
  RegisterComponents('Data Controls', [TDBOutline]);
end;

end.
