unit Acropspl;

interface

{ A silly wrapper component to hide the real code in the BaseASpl unit }

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs;


type
  TAcropSpell = class(TComponent)
  private
    { Private declarations }
    DictDataPtr         : pointer;
    FIsDictionaryOpen   : boolean;
    FDictionaryMain     : string;
    FDictionaryUser     : string;
    FMaxSuggestions     : byte;
  protected
    { Protected declarations }
  public
    { Public declarations }
    constructor Create(AOwner : TComponent); override;
    procedure Free;
    function OpenDictionary(Filename : STRING) : boolean;
    function OpenUserDictionary(Filename : STRING) : integer;
    procedure CloseDictionaries;
    function GoodWord(TheWord : STRING) : BOOLEAN;
    function AddWord(TheWord : STRING; DictID : integer) : BOOLEAN;
    function SuggestCloseMatch(TheWord : STRING) : TStringList;
    function SuggestPhoneme(TheWord : STRING) : TStringList;
    function GetUserDictionaryList(DictID : integer) : TStringList;
    function CloseUserDictionary(DictID : INTEGER) : BOOLEAN;
    function DeleteUserDictionary(DictID : integer) : BOOLEAN;
    procedure DeleteUserDictionaries;
    procedure CloseUserDictionaries;
    function BuildUserDictionary(Filename : STRING; WordList : TStringList) : INTEGER;
    function IsDictionaryOpen : boolean;
    procedure SetMaxSuggestions(Max : byte);
  published
    { Published declarations }
    property DictionaryName : string read FDictionaryMain write FDictionaryMain;
    property DictionaryUser : string read FDictionaryUser write FDictionaryUser;
    property MaxSuggestions : byte read FMaxSuggestions write SetMaxSuggestions default 10;
  end;

procedure Register;

implementation

uses BaseASpl;

procedure Register;
begin
  RegisterComponents('Samples', [TAcropSpell]);
end;


constructor TAcropSpell.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  FIsDictionaryOpen   := FALSE;
  FDictionaryMain     := 'acrop.dct';
  FDictionaryUser     := 'custom.dct';
  FMaxSuggestions     := 10;
  BaseASpl.InitDictionaryData(DictDataPtr);
end;

procedure TAcropSpell.Free;
begin
  if FIsDictionaryOpen then
    CloseDictionaries;
  BaseASpl.ReleaseDictionaryData(DictDataPtr);
  inherited Free;
end;

procedure OpenDefaultDictionary(AOwner : TAcropSpell);
begin
  with AOwner do
    begin
      FIsDictionaryOpen := OpenDictionary(FDictionaryMain);
    end;
end;

procedure TAcropSpell.SetMaxSuggestions(Max : byte);
begin
  if Max > 30 then
    Max := 30;
  FMaxSuggestions := Max;
end;

function TAcropSpell.OpenDictionary(Filename : STRING) : boolean;
begin
  OpenDictionary := FALSE;
  if FIsDictionaryOpen then
    exit;
  if BaseASpl.OpenDictionary(DictDataPtr, Filename) then
    begin
      FIsDictionaryOpen := TRUE;
      FDictionaryMain   := Filename;
      OpenDictionary    := TRUE;
    end;
end;

function TAcropSpell.OpenUserDictionary(Filename : STRING) : integer;
begin
  OpenUserDictionary := BaseASpl.OpenUserDictionary(DictDataPtr, Filename);
end;

procedure TAcropSpell.CloseDictionaries;
begin
  if not FIsDictionaryOpen then
    exit;
  BaseASpl.CloseDictionaries(DictDataPtr);
  FIsDictionaryOpen := FALSE;
end;

function TAcropSpell.GoodWord(TheWord : STRING) : BOOLEAN;
begin
  if not FIsDictionaryOpen then
    OpenDefaultDictionary(Self);
  GoodWord := BaseASpl.GoodWord(DictDataPtr, TheWord);
end;

function TAcropSpell.AddWord(TheWord : STRING; DictID : integer) : BOOLEAN;
begin
  if not FIsDictionaryOpen then
    OpenDefaultDictionary(Self);
  AddWord := BaseASpl.AddWord(DictDataPtr, TheWord, DictID);
end;

function TAcropSpell.SuggestCloseMatch(TheWord : STRING) : TStringList;
begin
  if not FIsDictionaryOpen then
    OpenDefaultDictionary(Self);
  SuggestCloseMatch := BaseASpl.SuggestCloseMatch(DictDataPtr, TheWord, FMaxSuggestions);
end;

function TAcropSpell.SuggestPhoneme(TheWord : STRING) : TStringList;
begin
  if not FIsDictionaryOpen then
    OpenDefaultDictionary(Self);
  SuggestPhoneme := BaseASpl.SuggestPhoneme(DictDataPtr, TheWord, FMaxSuggestions);
end;

function TAcropSpell.GetUserDictionaryList(DictID : integer) : TStringList;
begin
  GetUserDictionaryList := BaseASpl.GetUserDictionaryList(DictDataPtr, DictID);
end;

function TAcropSpell.CloseUserDictionary(DictID : INTEGER) : BOOLEAN;
begin
  CloseUserDictionary := BaseASpl.CloseUserDictionary(DictDataPtr, DictID);
end;

function TAcropSpell.DeleteUserDictionary(DictID : integer) : BOOLEAN;
begin
  DeleteUserDictionary := BaseASpl.DeleteUserDictionary(DictDataPtr, DictID);
end;

procedure TAcropSpell.DeleteUserDictionaries;
begin
  BaseASpl.DeleteUserDictionaries(DictDataPtr);
end;

procedure TAcropSpell.CloseUserDictionaries;
begin
  BaseASpl.CloseUserDictionaries(DictDataPtr);
end;

function TAcropSpell.BuildUserDictionary(Filename : STRING; WordList : TStringList) : integer;
begin
  BuildUserDictionary := BaseASpl.BuildUserDictionary(DictDataPtr, Filename, WordList);
end;

function TAcropSpell.IsDictionaryOpen : boolean;
begin
  IsDictionaryOpen := FIsDictionaryOpen;
end;

end.
