/*
 AdsLink.cpp
 Portions copyright  1995, Bernd R. Fix. All Rights Reserved.

 Description: ADS application link class

 Notes:
      The CAdsLink class is a utility class encapsulating the ADS
      'dispatch loop'.  It controls the initialisation of the link
      with AutoCAD, registers the application for extended entity
      data, if required, and administrates the request codes sent
      by AutoCAD via the dispatch loop.

      It should not need to be customised by developers, unless of
      course the ADS API changes with regard to the definition, etc,
      of command functions.

 Revisions:
      Jun 96  DG  Initial issue

*/



#include "stdafx.h"
#include "AdsLink.h"
#include "AdsApp.h"
#include "AdsCmd.h"


#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif


extern "C" {
   int ads_winit(HINSTANCE hInstance, char *pszAppName);
}



//// Construction/destruction
CAdsLink::CAdsLink(const char *pszRegapp /*=NULL*/)
   : m_bIsInitialized(FALSE), m_csRegapp(pszRegapp)
{
}


CAdsLink::~CAdsLink()
{
   while (!m_cCommandList.IsEmpty()) {
      delete m_cCommandList.RemoveHead();
   }
}


//// Member functions
BOOL CAdsLink::InitLink(CAdsApp *pApp)
{
   ASSERT(!m_bIsInitialized);
   
   m_pApp = pApp;
   ASSERT(m_pApp != NULL);

   char pszExeName[_MAX_PATH] = "";
   strcpy(pszExeName, m_pApp->m_pszExeName);
	
	m_bIsInitialized = (ads_winit(m_pApp->m_hInstance, pszExeName) == RTNORM);
	return m_bIsInitialized;
}


BOOL CAdsLink::DefineCmd(const char *pszCommand,
                         PFNADSCOMMAND pfnCommand,
                         BOOL bToBeRegistered /*= FALSE*/,
                         const char *pszHelpTopic /*= NULL*/)
{
   ASSERT(m_bIsInitialized);

	CAdsCommand *pCommand = new CAdsCommand(pszCommand, pfnCommand,
	                                        bToBeRegistered, pszHelpTopic);
   ASSERT(pCommand != NULL);

	return (m_cCommandList.AddTail(pCommand) != NULL);
}


BOOL CAdsLink::GetAdsRequest(int& nAdsStatus)
{
   static int nAppStatus = RSRSLT;

   nAdsStatus = ads_link(nAppStatus);

   nAppStatus = RSRSLT;
   switch (nAdsStatus) {
      case RQXLOAD:
         ASSERT(m_bIsInitialized);
         RegisterApp();

         nAppStatus = (ParseFunctions(kLoad) == TRUE) ? RSRSLT : RSERR;
         break;

      case RQSUBR:
         ASSERT(m_bIsInitialized);
         nAppStatus = (ExecuteCommand() == TRUE) ? RSRSLT : RSERR;
         break;

      case RQXUNLD:
         nAppStatus = (ParseFunctions(kUnload) == TRUE) ? RSRSLT : RSERR;
         m_bIsInitialized = FALSE;
         break;

      case RQSAVE:
      case RQQUIT:
      case RQEND:
      case RQCFG:
      default:
         m_pApp->OnRequest(nAdsStatus);
         break;
   }

   return (nAppStatus == RSRSLT) ? TRUE : FALSE;
}


void CAdsLink::Break()
{
   if (m_bIsInitialized) {
      ads_link(RSERR);
      m_bIsInitialized = FALSE;
   }
}


//// Helper functions
BOOL CAdsLink::RegisterApp()
{
   BOOL bRegisteredOk = FALSE;

   if (!m_csRegapp.IsEmpty()) {
      struct resbuf *pRegappDef = ads_tblsearch("APPID", m_csRegapp, FALSE);

      if (pRegappDef != NULL) {
         ads_relrb(pRegappDef);
         pRegappDef = NULL;
         bRegisteredOk = TRUE;
      }
      else if (ads_regapp(m_csRegapp) == RTNORM) {
         bRegisteredOk = TRUE;
      }
   }
   else {
      bRegisteredOk = TRUE;
   }

   return bRegisteredOk;
}


BOOL CAdsLink::ParseFunctions(int eAction)
{
   BOOL bResult = TRUE;
   POSITION nCmdPosition = m_cCommandList.GetHeadPosition();
   CAdsCommand *pCommand = NULL;


   for (int nCmdNumber = 1; nCmdPosition != NULL; nCmdNumber++) {
      pCommand = (CAdsCommand*)m_cCommandList.GetNext(nCmdPosition);
      ASSERT(pCommand != NULL);

      if (   (eAction == kLoad && pCommand->Define() == FALSE)
          || (eAction == kUnload && pCommand->Undefine() == FALSE))
      {
         bResult = FALSE;
      }
   }

   return bResult;
}


int CAdsLink::ExecuteCommand()
{
   int nCmdIndex = ads_getfuncode();
   CAdsCommand *pCommand = NULL;
   int nCmdResult = RSERR;
   BOOL bCommandFound = FALSE;


   if (nCmdIndex != RTERROR) {
      for (POSITION nCmdPosition = m_cCommandList.GetHeadPosition();
           nCmdPosition != NULL; )
      {
         pCommand = (CAdsCommand*)m_cCommandList.GetNext(nCmdPosition);

         if (pCommand->DefunCode() == nCmdIndex) {
            nCmdResult = pCommand->Run();
            bCommandFound = TRUE;
            break;
         }
      }

      ASSERT(bCommandFound == TRUE);
   }

   return nCmdResult;
}
