/*
 AdsCmd.cpp

 Description: ADS command class

 Notes:
      The CAdsCommand class is a utility class providing an interface
      between AutoCAD and the actual command function.  It takes care
      of defining and registering the function, etc.

      It should not need to be customised by developers, unless of
      course the ADS API changes with regard to the definition, etc,
      of command functions.

 Revisions:
      Jun 96  DG  Initial issue
*/



#include "stdafx.h"
#include "AdsApp.h"
#include "AdsCmd.h"


#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

extern CAdsApp theApp;
extern "C" {
   HWND adsw_hwndAcad;
}


short CAdsCommand::m_nLastDefunCode = 0;



//// Construction/destruction
CAdsCommand::CAdsCommand(const char *pszCommand,
                         PFNADSCOMMAND pfnCommand,
                         BOOL bToBeRegistered /*= FALSE*/,
                         const char *pszHelpTopic /*= NULL*/)
   : m_csCommand(pszCommand), m_pfnCommand(pfnCommand),
     m_bToBeRegistered(bToBeRegistered),
     m_csHelpTopic(pszHelpTopic), m_nDefunCode(0)
{
   m_csCommand.MakeUpper();

   ASSERT(   m_csHelpTopic.IsEmpty()
          || m_csCommand.Left(2) == "C:");
}


CAdsCommand::~CAdsCommand()
{
   if (IsDefined()) {
      Undefine();
   }
}


//// Member functions
BOOL CAdsCommand::Define()
{
   if (ads_defun(m_csCommand, (short)(m_nLastDefunCode + 1)) == RTNORM) {
      m_nDefunCode = ++m_nLastDefunCode;

      if (m_bToBeRegistered) {
         if (ads_regfunc(m_pfnCommand, m_nDefunCode) == RTERROR) {
            /* Reset the flag so we don't bother trying to denregister
               it when the application is unloaded.
            */
            m_bToBeRegistered = FALSE;
         }
      }

      if (!m_csHelpTopic.IsEmpty()) {
         char *pszCommand = m_csCommand.GetBuffer(m_csCommand.GetLength());
         char *pszHelpTopic = m_csHelpTopic.GetBuffer(m_csHelpTopic.GetLength());
         char pszHelpFile[_MAX_PATH] = "";
         strcpy(pszHelpFile, theApp.m_pszHelpFilePath);

         ads_setfunhelp(pszCommand, pszHelpFile, pszHelpTopic, 0);

         m_csCommand.ReleaseBuffer();
         m_csHelpTopic.ReleaseBuffer();
      }
   }

   return (m_nDefunCode != 0);
}


BOOL CAdsCommand::Undefine()
{
   if (ads_undef(m_csCommand, m_nDefunCode) == RTNORM) {
      if (m_bToBeRegistered) {
         ads_regfunc(NULL, m_nDefunCode);
      }

      m_nDefunCode = 0;
   }

   return (m_nDefunCode == 0);
}


short CAdsCommand::DefunCode() const
{
   return m_nDefunCode;
}


int CAdsCommand::Run() const
{
   int nReturnCode = (*m_pfnCommand)();

   /* Pass focus back to AutoCAD to make sure it is properly activated */
   VERIFY(SetForegroundWindow(adsw_hwndAcad));

   return nReturnCode;
}


BOOL CAdsCommand::IsDefined() const
{
   return (m_nDefunCode != 0);
}
