/*
	MATHDXF.C
	JK Jordan, 10/24/92    see help strings below for info
*/

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <io.h>
#include <string.h>

#define DEFAULTDXFFILE  "math.dxf"
#define OBJECTNAME      "OBJ01"

#define MAXXYBOXDIMENSION (float)100.0   /* 3D Studio units */
#define DEFAULTSCALEFAC (float)0.5       /* ratio of height to MAXXYBOXDIMENSION */

/* prototypes */
void makedxf(float unitlength, float zscalefac, int nrows, int ncols);
void doheader(void);
void dopolyline(void);
void dovertexP( float x, float y, float z);
void dovertexF( int ixa, int ixb, int ixc);
void doeof(void);
void wrs( char *string );
void wrf( float f );
void wri( int i );

/* global variables */
static float **n;
static FILE *dxffile;


/*---------------------------------------------------------------------------*/
main(int argc, char*argv[])
	{
	FILE   *mathprnfile;
	int    i, j;
	float  z;
	float  max_z;
	float  scalefac;
	float  zscalefac;
	int    max_xy;
	float  unitlength;
	char   infilename[80], outfilename[80];
	char   c;
	int    ncols, nrows;
	int    displayflag;



	/* ---use command line parameters if available */
	if( (argc==1) || (strchr(argv[1],'?')!=NULL ) )
		{
		printf("\nMATHDXF: convert an ASCII table of numbers into a DXF file.");
		printf("\n    This program was developed to convert a two-dimensional array");
		printf("\n    of values representing a 3D surface in Mathcad, however it");
		printf("\n    should convert any rectangular ASCII table of values.  The");
		printf("\n    Mathcad command WRITEPRN(myfile):=L will create the data file");
		printf("\n    MYFILE.PRN from the matrix L.  The converted DXF file will");
		printf("\n    contain a polyface mesh of a single object named OBJ01 suitible");
		printf("\n    for loading into 3D Studio or DXFIN'ing into Autocad.");
		printf("\n");
		printf("\n    Usage:");
		printf("\n       mathdxf <infile>.prn [<dxffile>.dxf [<scale factor> [ /d ]]]");
		printf("\n");
		printf("\n         - <infile> is the ASCII input file, space delimited.  No default.");
		printf("\n         - mesh file <dxffile> will be created.  Default: MATH.DXF.");
		printf("\n         - <scale factor> is the relative height of the bounding box,");
		printf("\n           i.e., a factor of 1.0 will result in an object height equal");
		printf("\n           to the larger of the width and length.  Default: 0.5");
		printf("\n");
		printf("\n    Ex:   mathdxf myfile.prn            converts MYFILE.PRN to MATH.DXF");
		printf("\n          mathdxf myfile.prn test.dxf           converts it to TEST.DXF");
		printf("\n          mathdxf myfile.prn test.dxf 0.5        ...and sets the height");
		printf("\n                                                      to half the width");
		printf("\nJohn K Jordan (JKJ) [71174,711] (615)482-4698    Note: this is FREEware.\n");
		exit(0);
		}

	else
		strcpy(infilename,argv[1]);

	if( argc>2 )
		strcpy(outfilename,argv[2]);
	else
		strcpy(outfilename,DEFAULTDXFFILE);

	if( argc>3 )
		scalefac = (float)atof(argv[3]);
	else
		scalefac = DEFAULTSCALEFAC;

	// note: a 'd' as the 4th parm will display data before DXF'ing
	if( argc>4 && strchr(strlwr(argv[4]),'d') )
		displayflag=1;
	else
		displayflag=0;

	/* ---open files */
	if((mathprnfile=fopen(infilename,"rt"))==NULL)
		{
		printf("ERROR opening .prn for input\n\n");
		exit(0);
		}

	if((dxffile = fopen(outfilename, "wt"))==NULL)
		{
		printf("ERROR opening .dxf file for output\n\n");
		fclose(mathprnfile);
		exit(0);
		}

	/* first, count how many columns and rows are in this data file */
	ncols=nrows=0;
	c = 'x';
	while(1)
		{
		if( fscanf(mathprnfile, " %f",&z)==EOF )     //read one value so we can count columns
			{
			if( c!='\n' )                             //if last row doesn't have a '\n'
				nrows++;
			break;
			}
		if( nrows==0)                                //count cols in first row only
			ncols++;

		while( (c=(char)getc(mathprnfile))==' ' || (c=='\t') )
			;                                         //skip any blanks before '\n'
		if( c=='\n' )                                //count end-of-lines to count rows
			nrows++;
		ungetc(c,mathprnfile);
		}
	fseek(mathprnfile,0L,SEEK_SET);                 //reset file pointer to beginning

	/* ---allocate space for 2d floating point array */
	if( (n = (float **)calloc( nrows, sizeof(float *) ))==NULL )
		exit(0);
	for(j=0; j<nrows; j++ )
		if( (*(n+j)=(float *)calloc( ncols, sizeof(float) ) )==NULL )
			exit(0);

	/* ---read math data file */
	printf("\nMathcad PRN to Autocad DXF\n");
	printf("\n   reading PRN file %s, %d rows of %d columns...",infilename,nrows,ncols);
	max_z = (float)-1.0e6;
	for( j=0; j<nrows; j++ )
		{
		for( i=0; i<ncols; i++ )
			{
			if( fscanf(mathprnfile, "%f ",&z)==EOF )	z = (float)0.0;
			if( z < (float)1.0e-6 )                   z = (float)0.0;
			n[j][i]=z;
			max_z = max( z, max_z );                  //get maximum z for vert scaling
			}
		}

	fclose(mathprnfile);
	printf("\n");

	/* ---display data */
	if( displayflag )
		{
		printf("\n");
		for( j=0; j<nrows; j++ )
			{
			for( i=0; i<ncols; i++ )
				{
				printf("  %f", n[j][i]);
				}
			printf("\n");
			}
		printf("\n");
		}


	/* scale such that the bounding box height is scalefac*width */
	max_xy = max(ncols-1, nrows-1);
	unitlength = MAXXYBOXDIMENSION / max_xy;
	zscalefac = (MAXXYBOXDIMENSION * scalefac) / max_z;


	/* ---build the DXF file */
	printf("\n   creating DXF file %s...",outfilename);
	makedxf(unitlength, zscalefac, nrows, ncols);


	/* ---close everything down */
	fclose(dxffile);

	for(j=0; i<ncols; i++)
		free(*(n+j));
	free(n);

	printf("\n\n   done.\n");

	return(0);
	}

/*---------------------------------------------------------------------------*/
void makedxf(float unitlength, float zscalefac, int nrows, int ncols)
	{
	int i,j;

	doheader();
	dopolyline();

	for( j=0; j<nrows; j++ )
		for( i=0; i<ncols; i++ )
			dovertexP(i*unitlength, j*unitlength, n[j][i]*zscalefac);


	for( j=0; j<nrows-1; j++ )
		for( i=0; i<ncols-1; i++ )
			{
			dovertexF( i+1+ncols*j, -(i+2+ncols*j), i+1+ncols*(j+1) );
			dovertexF( i+2+ncols*j, i+2+ncols*(j+1), -(i+1+ncols*(j+1)) );
			}

	doeof();
	}

/*---------------------------------------------------------------------------*/
void doheader(void)
	{
	wrs("0");
	wrs("SECTION");
	wrs("2");
	wrs("ENTITIES");
	}
/*---------------------------------------------------------------------------*/
void dopolyline(void)
	{
	wrs("0");
	wrs("POLYLINE");
	wrs("8");
	wrs(OBJECTNAME);
	wrs("66");
	wrs("1");
	wrs("70");
	wrs("64");
	wrs("71");
	wrs("999");
	wrs("72");
	wrs("999");
	}
/*---------------------------------------------------------------------------*/
void dovertexP( float x, float y, float z)
	{
	wrs("0");
	wrs("VERTEX");
	wrs("8");
	wrs(OBJECTNAME);
	wrs("10");
	wrf(x);
	wrs("20");
	wrf(y);
	wrs("30");
	wrf(z);
	wrs("70");
	wrs("192");
	}
/*---------------------------------------------------------------------------*/
void dovertexF( int ixa, int ixb, int ixc)
	{
	wrs("0");
	wrs("VERTEX");
	wrs("8");
	wrs(OBJECTNAME);
	wrs("10");
	wrs("0");
	wrs("20");
	wrs("0");
	wrs("30");
	wrs("0");
	wrs("70");
	wrs("128");
	wrs("71");
	wri(ixa);
	wrs("72");
	wri(ixb);
	wrs("73");
	wri(ixc);
	}
/*---------------------------------------------------------------------------*/
void doeof(void)
	{
	wrs("0");
	wrs("SEQEND");
	wrs("8");
	wrs(OBJECTNAME);
	wrs("0");
	wrs("ENDSEC");
	wrs("0");
	wrs("EOF");
	}
/*---------------------------------------------------------------------------*/
void wrs( char *string )
	{
	fprintf(dxffile, "%s\n", string);
	}
/*---------------------------------------------------------------------------*/
void wrf( float f )
	{
	fprintf(dxffile, "%g\n", f);
	}
/*---------------------------------------------------------------------------*/
void wri( int i )
	{
	fprintf(dxffile, "%d\n", i);
	}
/*---------------------------------------------------------------------------*/
