;---------------------- C-SECT.LSP -----------------------------------------
;  This function will allow you to create a 3D object in by defining the 
;  cross sections (C-SECT) that make up the object.
;  
;  Imagine taking a C.A.T. scan of a 3D object and connecting the outline
;  contours of each of the slices with 3DFACEs. These 3DFACEs would define
;  the external surface of the object.
;
;  For this function a cross section is a PLINE that defines the surface
;  contour at a specific Z-AXIS elevation. The PLINEs can be open or closed, 
;  decurved or splined.
;
;  The only qualification is that ALL of the PLINEs (cross sections) MUST 
;  have the same number of vertices. At first this would appear to be a
;  severly limiting factor but because the function can be used on different
;  groups of PLINEs it is not as bad as it first appears.
;      
;  Written by  Kurt Rauchenberger   73477,2260                     11/27/88
;
;  Acknowledgment:
;	Some of the Ideas for building the 3DFACEs and moving around the
;	PLINEs came from looking at the R-SURF.LSP function on the Bonus
;	Disk. I hope that the folks who wrote that function don't mind that
;	I borrowed one of their subfunctions (vernext). Thanks. 
;
;  GLOBAL VARIABLES
;  -------------------------------------------------------------------------
;  hlist		This list will contain the heights of the selected
;			PLINEs sorted from lowest Z-AXIS elevation to highest
;
;  keylist		This is a list which contains sublists of each of the
;                       selected PLINEs:
;                       	entity name 
;                       	height (Z-AXIS elevation)
;			It is used to map the sorted heights in hlist to a
;			specific PLINE entity name.  
;----------------------------------------------------------------------------


;  Build a 3D point based on a vector entity name
(defun build_3dpt(vname / elist x y z)
	(setq elist (entget vname))
	(setq x (cadr (assoc 10 elist)))
	(setq y (caddr(assoc 10 elist)))
	(setq z (cdr(assoc 38 elist)))
	(list x y z)
) 

; Draw a 3Dface given the four corner points in a Clockwise order
(defun face(p1 p2 p3 p4)
	(command "3DFACE" 
			p1 
			p2 
			p3 
			p4 
			""
	)
)

;  Get next polyline vertex, ignoring spline control points
; *** This function is from R-Surf.lsp on the Bonus Disk ****
(defun vertnext (curr / v ve)
    (setq ve (entget (setq v (entnext curr))))
    (while (and (= (cdr (assoc 0 ve)) "VERTEX")
                (= (logand (cdr (assoc 70 ve)) 16) 16))
       (setq ve (entget (setq v (entnext v))))
    )
    v
)

; Draw 3Dfaces connecting 2 Plines that are at different elevations

(defun do_faces(plist1 plist2 / v1 v2 v3 v4 p1 p2 p3 p4 ep3 ep4)

   ; Initialize the vertices and retrieve vertex data
    (if (= (logand (cdr (assoc 70 (entget plist1))) 1) 1)
        (setq cflag 1) ;Set the flag if it's a closed Pline
    )
   (setq v1 (vertnext plist1))
   (setq p1 (build_3dpt v1))
   (setq v2 (vertnext v1))
   (setq p2 (build_3dpt v2))
   (setq v3 (vertnext plist2))
   (setq p3 (build_3dpt v3))
   (setq v4 (vertnext v3))
   (setq p4 (build_3dpt v4))
   (setq ep4 p1 ep3 p3) ;Save these 3Dpts in case it's a closed Pline
   (face p1 p2 p4 p3) ;Draw the first 3DFACE
   (while (= (cdr (assoc 0 (entget (vertnext v2)))) "VERTEX")
	   (setq p1 (build_3dpt v2))
   	   (setq v2 (vertnext v2))
	   (setq p2 (build_3dpt v2))
	   (setq p3 (build_3dpt v4))
	   (setq v4 (vertnext v4))
	   (setq p4 (build_3dpt v4))
	   (face p1 p2 p4 p3)
    )
    ; If it's a closed polyline do the last face
    (if  cflag
	   (face p2 p4 ep3 ep4)
    )	
)

; Get the PLINEs that will be used to make the 3D object

(defun get_sects( / s1 slength scount elist templist h ename)

	(terpri)
	(prompt "Select the Pline cross sections:")
	(setq s1 (ssget))
	(setq slength (sslength s1))

; Get the height of each PLINE and Sort them from lowest to highest
	(setq keylist nil)
	(setq scount 0)
	(repeat slength
		(setq templist nil)
		(setq templist (cons (setq ename (ssname s1 scount)) templist))
		(setq elist (entget ename))
		(if (NULL (setq h (cdr (assoc 38 elist))))
		 (setq h 0.0);If there isn't an elevation defined set h to 0.0		
		)
		(setq hlist (cons h hlist))
		(setq templist (cons h templist))
		(setq keylist (cons templist keylist))
		(setq scount (1+ scount))
	)
	(setq hlist (sort hlist))
)

; Sorts a list

(defun sort (slist)
 (if (null slist)   
  (assoc "slist" slist)
  (cons (apply 'min slist) (sort (remove (apply 'min slist) slist)))
 )
)


; Removes an the first occurence of an atom from a list
; and returns the list minus that atom

(defun remove (item list)
 (cond
  ((null list)	 'list)
  ((eq item (car list)) (cdr list))
  (t (cons (car list) (remove item (cdr list)))) 		
 )	
)

; The Main function

(defun C:C-SECT( / h list1 list2)
	(get_sects) 		 ;Get the cross sections
	(setq h (car hlist))	 ;Set h equal to the lowest height from hlist
	(setq hlist (cdr hlist)) ;Remove the lowest height from hlist
				 ;Map the height (h) to the PLINE entity name
	(setq list2 (cadr (assoc h keylist))) 
	(while hlist		 ;As long as there are PLINEs do the loop
	   (setq list1 list2)	 ;Set the first PLINE equal to the Previous
	   (setq h (car hlist))	 ;Set h equal to the lowest height from hlist
	   (setq hlist (cdr hlist)) ;Remove the lowest height from hlist
				 ;Map the height (h) to the Pline entity name	
	   (setq list2 (cadr (assoc h keylist)))
	   (if list2		 ;If there is are 2 entity names connect the 
				 ; 2 PLINEs with 3DFACEs.
	    (do_faces list1 list2)
	   )
	)
)


