/*
 * ACCTIMER.C -- Timer DLL for Access
 *
 * Based on TIMERDLL.C written by Paul Vick. His original comments
 * are included below. Other stuff within the code has been changed
 * to reflect its operation.
 * -----------------------------------------------------------------------
 * Questions? Comments? Write to Paul Vick (paulv) <=== SEE BELOW 
 * There are various places in the code which note where changes may be
 * made. They are marked CONSIDER.
 *
 * LIMITATIONS: This is just the skeleton of what can be done. There is no
 * choice in what script gets called and you can only have one timer.
 *
 * HOW IT'S DONE: Pretty simple. I set up a DDE conversation with Access.
 * Once a timer is set up, each timer callback generates a DDE execute.
 *
 * WARNING: This hasn't been _that_ extensively tested. No guarentees.
 *
 * -----------------------------------------------------------------------
 * NEW VERSION:
 *
 * Now called ACCTIMER. This version has time intervals in seconds instead
 * of milliseconds. Nothing happens in milliseconds with Access (grin)!
 * It also accepts up to 8 timers, each with its own interval in seconds
 * and a specified macro to run. See SetAccessTimer() and KillAccessTimer
 * comments for calling params.
 *
 * When the DLL is loaded, a 1 second internal timer is started. At each
 * tick, the list of timers is checked for active ones, and each is serviced
 * in turn as needed. The internal timer runs for the entire time the DLL
 * stays loaded.
 *
 * WARNING: I do not know whether Access ACKnowledges DDE macro requests
 * when the macro starts or when the macro finishes. In any case, there
 * is the potential for problems if a timer interval is shorter than the
 * time needed for the corresponding macro to complete. CAVEAT! I also do
 * not know whether Access can accept additional DDE macro execution
 * requests while it is running a macro from a previous DDE request.
 * TIMING BUGS ARE A REAL POSSIBILITY.
 *
 * In order to make things as safe as possible, the timer service routine
 * will not send a macro execute DDE request if one is already active.
 * The timer service routine could be entered while still active from
 * the previous 1-second tick's call. Calling DDE to send a request on
 * a connection with an already active transaction seems like asking for
 * trouble. So if the timer service routine is entered while a DDE
 * transaction is active, the timers will be decremented until they reach
 * 0, but no DDE requests will be made. This should approximate real-time
 * behavior.
 * 
 * Bob Denny	02-May-93	Change to be able to time in seconds, and
 *							to specify the name of the script to run
 *							when the timer goes off. Also modified for
 *							TCWIN 3.1.
 * Bob Denny	04-May-93	Add capability to handle multiple timers (8)
 *							and to really check for bogus arguments.
 */

#include <windows.h>
#include <ddeml.h>

#define MAX_TIMERS		8		// Maximum number of timers
#define MAX_MACRO_NAME	64		// Max length of macro name
#define SZACCESSNAME "MSAccess"	// Access DDE server name

//
// DLL level globals
//
TIMERPROC lpfnAccessTimerProc;	// Timer callback
HINSTANCE hinst;				// Instance of DLL
HCONV hconv = NULL;				// DDE Conversation handle
DWORD idInst = 0L;				// DDE Registration instance
FARPROC lpDdeProc;				// DDE callback
BOOL fInit = FALSE;				// Have we established DDE link?
HSZ hszAccess, hszSystem;		// String handles for app/topic
UINT hTimer = NULL;				// Windows Timer handle
BOOL fDDEActive = FALSE;		// Synchronization flag

//
// Timer-level variables
//
int iIntervalSec[MAX_TIMERS];	// Interval, seconds
int iSecsRemaining[MAX_TIMERS];	// Seconds remaining till macro run
								// Names of Access macro to execute
char szMacroName[MAX_TIMERS][MAX_MACRO_NAME];
int dMacroNameLen[MAX_TIMERS];	// Length of macro name string (long)

//
// Function Prototypes
//
VOID CALLBACK AccessTimerProc (HWND hwnd, UINT msg,
							   UINT idTimer, DWORD dwTime);
HDDEDATA CALLBACK TimerDdeCallback (UINT type, UINT fmt, HCONV hconv,
									HSZ hsz1, HSZ hsz2, HDDEDATA hData,
									DWORD dwData1, DWORD dwData2);
BOOL FAR PASCAL SetAccessTimer (int iTimer, int iSecs, LPSTR lpszName);
void FAR PASCAL KillAccessTimer(int iTimer);


/*
 * Initialization entry point for the DLL
 */
#pragma argsused
int FAR PASCAL LibMain (HINSTANCE hinst, WORD wDataSeg,
						WORD cbHeapSize, LPSTR lpszCmdLine)
{
	UINT i;

	if (cbHeapSize != 0)
		UnlockData(0);

	for(i=0; i<MAX_TIMERS; i++)
		iIntervalSec[i] = NULL;		// Mark timers inactive

	/*
	 * Register with DDEML and start conversation with Access
	 * on system topic.
     */
	lpDdeProc = MakeProcInstance((FARPROC)TimerDdeCallback, hinst);
	if (DdeInitialize((LPDWORD) &idInst, (PFNCALLBACK) lpDdeProc,
						APPCMD_CLIENTONLY, 0L) != DMLERR_NO_ERROR)
		return FALSE;
	if ((hszAccess = DdeCreateStringHandle(idInst, SZACCESSNAME, 0)) == NULL)
		return FALSE;
	if ((hszSystem = DdeCreateStringHandle(idInst, SZDDESYS_TOPIC, 0)) == NULL)
	{
		DdeFreeStringHandle(idInst, hszAccess);
		return FALSE;
    }
	if ((hconv = DdeConnect(idInst, hszAccess, hszSystem, NULL)) == NULL)
	{
		DdeFreeStringHandle(idInst, hszAccess);
		DdeFreeStringHandle(idInst, hszSystem);
		return FALSE;
	}
	fInit = TRUE;

	/*
	 * Set up and start a 1 second timer. Initialize the interval
	 * and countdown variables. Make a copy of the macro name and
     * get the needed long length of it.
	 */
	lpfnAccessTimerProc =
			(TIMERPROC) MakeProcInstance((FARPROC)AccessTimerProc, hinst);
	if ((hTimer = SetTimer(NULL, 1, 1000, lpfnAccessTimerProc)) == NULL)
		return FALSE;

	return TRUE;
}

/*
 * The callback function that Windows calls everytime the timer goes off.
 * NEW VERSION: This is called once per second, and counts down the
 *				seconds value till it goes to zero. At that point, it
 *				runs each ACCESS macro for whom the interval has elapsed,
 *				then resets the seconds remaining count back to its timeout
 *				value.
 */
#pragma argsused
VOID CALLBACK AccessTimerProc (HWND hwnd, UINT msg,
							   UINT idTimer, DWORD dwTime)
{
	UINT i;

	for(i=0; i<MAX_TIMERS; i++)
	{
		if(iIntervalSec[i] != 0)		// If this timer is active 
		{
			if(iSecsRemaining[i] > 0)	// Always count down to 0
				iSecsRemaining[i] -= 1;
			else if (!fDDEActive)		// Run macro if DDE not active
			{
            	fDDEActive = TRUE;		// Set synch flag
				iSecsRemaining[i] = iIntervalSec[i];
				//
				// Transaction is synchronous with 5 sec. timeout.
                //
				DdeClientTransaction((LPSTR)szMacroName[i],
										dMacroNameLen[i],
										hconv,
										NULL,
										CF_TEXT,
										XTYP_EXECUTE,
										5,
										NULL);
				fDDEActive = FALSE;		// Reset synch flag
			}
		}
    }
}

/*
 * Callback for the DLL's DDE. Since we don't accept any messages,
 * ignore everything except a message saying that Access has
 * disconnected from us.
 */
#pragma argsused
HDDEDATA CALLBACK TimerDdeCallback (UINT type, UINT fmt, HCONV hconv,
									HSZ hsz1, HSZ hsz2, HDDEDATA hData,
									DWORD dwData1, DWORD dwData2)
{
	/*
	 * Ignore all DDE messages, except the ones telling us we've been
     * disconnected...
	 */
	if ((type & XTYP_DISCONNECT) != 0)
    {
		KillAccessTimer(-1);
		hconv = NULL;
	}

    return(NULL);				// Return is ignored
}

/*
 * SetAccessTimer
 *
 * DLL Call point to set up the timer.
 *
 * Inputs:	Timer number (0-7)
 *			Number of seconds between macro runs (int)
 *			Name of macro to run (lpsz)
 *
 * Returns: FALSE if _anything_ failed. TRUE otherwise
 */

BOOL FAR PASCAL SetAccessTimer (int iTimer, int iSecs, LPSTR lpszName)
{
	//
	// Safety checks. Get length of macro name for later.
    //
	if(iTimer < 0 || iTimer >= MAX_TIMERS || iSecs <= 0 || lpszName == NULL)
		return(FALSE);						// Bogus args
	if (iIntervalSec[iTimer] != 0)
		return(FALSE);						// Already active!
	if((dMacroNameLen[iTimer] = (DWORD)lstrlen(lpszName) + 1) <= 1)
		return(FALSE);						// Empty macro name
	if(dMacroNameLen[iTimer] > MAX_MACRO_NAME)
    	return(FALSE);						// Macro name too long

	// Initialize interval and counter
	iIntervalSec[iTimer] = iSecsRemaining[iTimer] = iSecs;
	// Copy macro name to global
	lstrcpy((LPSTR)szMacroName[iTimer], lpszName);

	return(TRUE);	
}

/*
 * KillAccessTimer()
 *
 * Inputs:	Timer number (0-7), negative # means kill all timers
 *
 * Kills a timer. If neg# is given, all active timers are killed.
 */

void FAR PASCAL KillAccessTimer(int iTimer)
{
	UINT i;

	if(iTimer >= MAX_TIMERS)
		return;
 
	if(iTimer < 0)
		for(i=0; i<MAX_TIMERS; i++)
			iIntervalSec[i] = 0;
	else
    	iIntervalSec[iTimer] = 0;
}

/*
 * DLL cleanup function
 */
int FAR PASCAL WEP (int nParameter)
{
	if (nParameter == WEP_FREE_DLL)
	{
    	KillAccessTimer(-1);		// Make sure no more macros run
		KillTimer(NULL, hTimer);	// Now kill the 1sec timer
		if (hconv != NULL)			// Safely tear down DDE link
			DdeDisconnect(hconv);
		if (fInit)					// Safely dispose string handles
		{
			DdeFreeStringHandle(idInst, hszAccess);
			DdeFreeStringHandle(idInst, hszSystem);
			DdeUninitialize(idInst);
		}
	}
	return 1;
}

