/****************************************************************************
**	File:	xafp2.C
**
**	Desc:	Example use of AFPScanFileInformation.
**
**	Disclaimer:
**
**		Novell, Inc. makes no representations or warranties with respect to
**		any NetWare software, and specifically disclaims any express or
**		implied warranties of merchantability, title, or fitness for a
**		particular purpose.  
**
**		Distribution of any NetWare software is forbidden without the
**		express written consent of Novell, Inc.  Further, Novell reserves
**		the right to discontinue distribution of any NetWare software.
**
**		Novell is not responsible for lost profits or revenue, loss of use
**		of the software, loss of data, costs of re-creating lost data, the
**		cost of any substitute equipment or program, or claims by any party
**		other than you.  Novell strongly recommends a backup be made before
**		any software is installed.   Technical support for this software
**		may be provided at the discretion of Novell.
**
**
**	QMK386 options used:
**
**		/ip - Appletalk
**
**	Programmers:
**
**		Ini	Who						Firm
**		-----------------------------------------------------------------------
**		ABJ	Adam B. Jerome			Novell Developer Support.
**
**	History:
**
**		When		Who	What
**		-----------------------------------------------------------------------
**		08-24-94	ABJ	First code.
*/

/****************************************************************************
**	Include headers, macros, function prototypes, etc.
*/
	/*------------------------------------------------------------------------
	**	ANSI
	*/
	#include <stdlib.h>	/*	exit()	*/
	#include <stdio.h>	/* printf()	*/
	#include <string.h>	/* strlen(), strchr() strrchr() */

	/*------------------------------------------------------------------------
	**	NetWare
	*/
	#include <nwafp.h>	/* AFPOpenFileFork()		*/
	#include <nwconn.h>	/* GetFileServerID()		*/
	#include <nwenvrn.h>	/* GetFileServerName()	*/
	#include <nwdir.h>	/* GetVolumeNumber();	*/
	#include <errno.h>	/* ESUCCESS					*/
	#include <niterror.h>/*	UNKNOWN_FILE_SERVER	*/	

	/*------------------------------------------------------------------------
	**	xafp2
	*/
	#define	AFP_MODE_READ			0x01
	#define	AFP_MODE_WRITE			0x02
	#define	AFP_MODE_DENYREAD		0x04
	#define	AFP_MODE_DENYWRITE	0x08

	#define	AFP_SEARCH_HIDDEN		0x02
	#define	AFP_SEARCH_SYSTEM		0x04
	#define	AFP_SEARCH_DIR			0x08
	#define	AFP_SEARCH_FILES		0x10

	#define	AFP_REQUEST_ALL		0x3FFF

	#define  AFP_ENTRYID_VOLROOT 	0x00000001L

/****************************************************************************
** Convert a DOS style path to a Apple style path.
**
**	Shifts string right one byte, NOT including NULL terminator.
**		(terminator not needed in an apple string.)
**	Inserts length of string in first byte.
**	Converts slashes to '\0's.
**
**	Return values:
**
**			0	Success.
**			1	String too long.
*/
int StrDOStoAFP(char *outStr, char *inStr)
	{
	char	*oldStr	= NULL;
	int	 cCode;
	int	 sLen;
	char	*cp;

	/*------------------------------------------------------------------------
	**	Determine length of string and verify length bounds.
	*/
	sLen=strlen(inStr);
	if(sLen > 254)
		{
		cCode = 1;
		goto END;
		}

	/*------------------------------------------------------------------------
	**	Set string length and move string back to origional buffer (+1).
	*/
	*outStr = (char)sLen;
	++outStr;
	strncpy(outStr, inStr, sLen);
	
	/*------------------------------------------------------------------------
	**	Convert forward slashes to backslashes,
	**	then back slashes to 0x00 from the right side first.
	*/
	while((cp=strchr(outStr, '/')) != NULL) *cp = '\\';
	while((cp=strrchr(outStr, '\\')) != NULL) *cp = '\0';

	cCode=0;

END:
	/*------------------------------------------------------------------------
	**	Return temp buffer to OS.
	*/
	if(oldStr != NULL) free(oldStr);

	return(cCode);
	}

/****************************************************************************
** Directory listing.
**
**	Input:	fileserverID	ID of fileserver to scan.
**
**				volumeNumber	Volume number to scan.
**
**				pathName			AFP style directory path to scan.
*/
int AFP_Dir(WORD fileServerID, int volumeNumber, char *afpPathString)
	{
	int 	cCode;
	long  AFPLastSeenID;
	AFPFILEINFO AFPScanInfo;
	LONG	files=0L;
	LONG	bytes=0L;

	printf("%-12s %-33s %10s %10s\n",
		"DOSname",
		"MACname",
		"DataFork",
		"ResFork"
		);
	printf("-----------------------------------------------------------------------------\n");

	/*------------------------------------------------------------------------
	** Scan for first file.
	*/
	AFPLastSeenID = (-1L);
	cCode=AFPScanFileInformation(
		/* I-	fileServerID	*/	fileServerID,
		/*	I-	volumeNum		*/	volumeNumber,
		/*	I-	AFPEntryID		*/	AFP_ENTRYID_VOLROOT,
		/*	I-	AFPLastSeenID	*/	&AFPLastSeenID,
		/*	I-	searchBitMap	*/	AFP_SEARCH_FILES|AFP_SEARCH_HIDDEN|AFP_SEARCH_SYSTEM|AFP_SEARCH_DIR,
		/*	I- requestBitMap	*/	AFP_REQUEST_ALL,
		/*	I-	AFPPathString	*/	afpPathString,
		/*	I-	structSize		*/	sizeof(AFPScanInfo),
		/*	-O	AFPScanInfo		*/ &AFPScanInfo
		);
	while(cCode == ESUCCESS)
		{
		/*---------------------------------------------------------------------
		**	Update stats.
		*/
		++files;
		bytes += AFPScanInfo.dataForkLength + AFPScanInfo.resourceForkLength;

		/*---------------------------------------------------------------------
		**	Print file information.
		*/
		printf("%-12.12s %-33.33s %10lu %10lu\n",
			AFPScanInfo.shortName,
			AFPScanInfo.longName,
			AFPScanInfo.dataForkLength,
			AFPScanInfo.resourceForkLength
			);

		/*---------------------------------------------------------------------
		**	Scan for next file.
		*/
		cCode=AFPScanFileInformation(
			/* I-	fileServerID	*/	fileServerID,
			/*	I-	volumeNum		*/	volumeNumber,
			/*	I-	AFPEntryID		*/	AFP_ENTRYID_VOLROOT,
			/*	I-	AFPLastSeenID	*/	&AFPLastSeenID,
			/*	I-	searchBitMap	*/	AFP_SEARCH_FILES|AFP_SEARCH_HIDDEN|AFP_SEARCH_SYSTEM|AFP_SEARCH_DIR,
			/*	I- requestBitMap	*/	AFP_REQUEST_ALL,
			/*	I-	AFPPathString	*/	afpPathString,
			/*	I-	structSize		*/	sizeof(AFPScanInfo),
			/*	-O	AFPScanInfo		*/ &AFPScanInfo
			);
		}
	switch(cCode)
		{
	   case ERR_FAILURE:	/* No more files found.  Normal exit. */
			break;

		default:
			printf("ERROR:  Unknown error returned by AFPScanFileInformation(): %d\n", cCode);
			goto END;
		}

	printf("%10lu file(s)  %10lu bytes used\n", files, bytes);

END:

	return(cCode);
	}

/****************************************************************************
** Print program's usage (help screen).
*/
void Usage(void)
	{
	printf("\n");
	printf("USAGE:     LOAD xafp2 \"{volumeName}:{path}\"\n");
	printf("\n");
	printf("                volumeName  Name of a volume which has an AFP name space.\n");
	printf("\n");
	printf("                path        Directory path to scan.\n");
	printf("\n");
	printf("NOTE:  Be aware that the MAC name space allows spaces in the path\n");
	printf("       and filename.  Therefore, the entire (arg[1]) must be in quotes\n");
	printf("       so that if it contains spaces, NetWare will not parse it into\n");
	printf("       multiple args.\n");
	printf("\n");
	return;
	}

/****************************************************************************
** Program start.
*/
void main(int argC, char *argV[])
	{
	int 	cCode;
	WORD	fileServerID		=	0;					/* Assume local file server.	*/
	
	char	*nwVol;
	int	volumeNumber;
	
	char  *nwPath;
	char	afpPath[255+1];

	/*------------------------------------------------------------------------
	**	Determine if AFP calls are supported on fileserver.
	*/
  	cCode=AFPSupported(fileServerID);
	if(cCode == FALSE)
		{
		printf("ERROR:  AFP system calls are not supported on this file server.\n");
		goto END;
		}

	/*------------------------------------------------------------------------
	**	Parse and validate command line args.
	*/
	if(argC != 2)										/* make sure we have an arg. */
		{
		Usage();
		goto END;
		}
	nwVol=argV[1];
	if((nwPath=strchr(nwVol, ':')) == NULL)	/* make sure we have a volume name. */
		{
		Usage();
		goto END;
		}
	*nwPath='\0';
	++nwPath;
	if(strlen(nwVol) > 15)							/* make sure volume name is 15 chars or less */
		{
		Usage();
		goto END;
		}

	/*------------------------------------------------------------------------
	**	Get the specified volume number.
	*/
	cCode=GetVolumeNumber(
		/*	I-	volumeName		*/ nwVol,
		/* -O	volumeNumber	*/	&volumeNumber
		);
	switch(cCode)
		{
		case ESUCCESS:
			break;

		case ERR_VOLUME_DOES_NOT_EXIST:
			printf("ERROR:  Volume %s: does not exist.\n", nwVol);
			goto END;

		default:
			printf("ERROR:  Unknown error returned by GetVolumeNumber(): %d\n", cCode);
			goto END;
		}

	/*------------------------------------------------------------------------
	**	Convert DOS style path to AFP path.
	*/
	cCode=StrDOStoAFP(afpPath, nwPath);
	switch(cCode)
		{
		case 0:
			break;

		case 1:
			printf("ERROR:  StrDOStoAFP() reports that path [%s] is too long.\n", nwPath);
			goto END;

		default:
			printf("ERROR:  Unknown error returned by STRDOStoAFP(): %d\n", cCode);
			goto END;
		}


	/*------------------------------------------------------------------------
	** Print DIR header and do the DIR.
	*/
	printf("\n");
	printf("\n Directory of %s:%s\n", nwVol, nwPath);
	printf("\n");

	AFP_Dir(fileServerID, volumeNumber, afpPath);

END:
	exit(0);
	}


