/*******************************************************************************
module name:  MKSTADIR
author:       Karl Boyken
date:         3/13/92
OS:           MS-DOS
compiler:     MicroSoft C 6.0
purpose:      MKSTADIR takes two or three arguments.  The first must be the
	      value of the Novell P_STATION identifier variable.  The second
	      must be the full path of the directory containing workstation
	      configuration directories.  The optional third is a switch that
	      indicates whether to exit with an errorlevel if the workstation
	      directory already exists.

	      MKSTADIR converts the 12-character P_STATION value into a DOS
	      directory structure, appends this structure to the parent 
	      workstation configuration directory, and tries to create the 
	      resulting directories.  It then tries to change the current 
	      directory for that drive to the lowest directory.

	      MKSTADIR returns 0 if it succeeds, and 1 if it does not.  It
	      fails if any of six conditions are encountered:

		1)  Less than two or more than three command-line 
		    arguments;

		2)  The first parameter is greater than 12 characters long;

		3)  The third parameter is on the command line and is not
		    Y, y, N, or n.

		4)  The third parameter is N or n and the directory already 
		    exists;

		5)  The directory creation fails for any reason other than 
		    a pre-existing directory; or

		6)  The directory cannot be changed to.
*******************************************************************************/


#include <errno.h>
#include <stdio.h>
#include <direct.h>

/*
author:    Karl Boyken
date:      1/7/92
requires:  str points to a block at least len + 1 bytes long
modifies:  str
effects:   pads str from the left with ch, to length len.  Returns str.
*/
char *strright(char *str, int len, char ch);

/*
author:    Karl Boyken
date:      12/10/91
modifies:  str
effects:   trims leading and trailing white space from str.  Returns str.
*/
char *strtrim(char *str);

int main(int argc, char *argv[]) {
  char p_station[81];  /* P_STATION value */
  char dir1_name[5];   /* High part of p_station */
  char dir2_name[10];  /* Low part of p_station */
  char dir_path[81];   /* Directory in which to create new directory */
  int  dir1_res = -1;  /* Result of parent directory creation */
  int  dir2_res = -1;  /* Result of child directory creation */

  /* Test number of command-line arguments (condition 1) */
  if ((argc < 3) || (argc > 4))
    return 1;
  
  /* Get and format P_STATION (condition 2) */
  strcpy(p_station, argv[1]);
  strtrim(p_station);
  if (strlen(p_station) > 12)
    return 1;
  strright(p_station, 12, '0');
  memmove(dir1_name, p_station, 4);
  dir1_name[4] = '\0';
  dir2_name[0] = '\\';
  memmove(dir2_name + 1, p_station + 4, 8);
  dir2_name[9] = '\0';
  
  /* Get parent directory path */
  strcpy(dir_path, argv[2]);
  if (dir_path[strlen(dir_path) - 1] != '\\')
    strcat(dir_path, "\\");

  /* Test the third command-line argument, if any (condition 3) */
  if (argc == 4) {
    if (strlen(argv[3]) > 1)
      return 1;
  }
  
  /* Try to create the directories */
  strcat(dir_path, dir1_name);
  dir1_res = mkdir(dir_path);
  strcat(dir_path, dir2_name);
  dir2_res = mkdir(dir_path);
  
  /* If the third parameter is N or n and the directory already exists,
     return (condition 4) */
  if (   ((argv[3][0] == 'N') || (argv[3][0] == 'n'))
      && (dir2_res == -1) 
      && (errno == EACCES))
    return 1;
  
  /* If the directory creation fails for a reason other than that it
     already exists, return (condition 5) */
  if ((dir2_res == -1) && (errno != EACCES))
    return 1;
  
  /* If we can't change directory, return 1 (condition 6) */
  if (chdir(dir_path))
    return 1;
  return 0;
}

/*
author:    Karl Boyken
date:      1/7/92
requires:  str points to a block at least len + 1 bytes long
modifies:  str
effects:   pads str from the left with ch, to length len.  Returns str.
*/
char *strright(char *str, int len, char ch) {
  int i, old_len;

  old_len = strlen(str);
  if (old_len >= len)
    return str;
  memmove(str + len - old_len, str, old_len + 1);
  memset(str, ch, len - old_len);
  return str;
}

/*
author:    Karl Boyken
date:      12/10/91
modifies:  str
effects:   trims leading and trailing white space from str.  Returns str.
*/
char *strtrim(char *str) {
  int i, len;

  i = strlen(str);
  while ((i > 0) && (str[i - 1] <= ' '))
    i -= 1;
  str[i] = '\0';
  len = i;
  i = 0;
  while ((i < len) && (str[i] <= ' '))
    i += 1;
  memmove(str, str + i, len - i + 1);
  return str;
}
