/*
    Module      : Netware Fake Root Calls
    File        : fakeroot.c

    Compiler    : Turbo C/C++

    Author      : Roger Vaughn
    Date        : 01-14-91

    Modified    : 01-28-91 - Fixed GetRelativeDepth to return -1 for invalid
                             drives.  Previously it returned 15, which was
                             interpreted as a fake root depth.
                  02-27-91 - Fixed GetRelativeDepth again.  It now returns
                             -1 for all possible error conditions.

    Source      : PDS Bullets, Vol 2, No. 5   June 1990

    Requirements: These calls require the Netware shells v3.01 or above.


    This source is written for Borland Turbo C/C++.  If you are using a
    different compiler, you will have to modify the source code as necessary.

*/


#include <dos.h>


#define MAPROOT     0xe905
#define DELETEROOT  0xe906
#define ROOTDEPTH   0xe907

#define NETWARE     0x21

/*-------------------------------------------------------------------------*/
/*                                                                         */
/*   MapFakeRoot                                                           */
/*                                                                         */
/*   This call allows you to map any drive as a fake root.  If the drive   */
/*   is not currently mapped, this will create the mapping and set the     */
/*   fake root.                                                            */
/*                                                                         */
/*   Parameters:                                                           */
/*       drive - the drive to map  (0 = current, 1 = A, 2 = B, ...)        */
/*       path  - the full path for the fake root                           */
/*                                                                         */
/*   Returns:                                                              */
/*       0 (0x00) - Success                                                */
/*       3 (0x03) - Invalid path                                           */
/*      15 (0x0f) - Invalid drive                                          */
/*      17 (0x11) - Not same device                                        */
/*                                                                         */
/*-------------------------------------------------------------------------*/

int MapFakeRoot(int drive, const char *path)
{
    asm push ds
    _AX = FP_SEG(path);
    _DS = _AX;
    _DX = FP_OFF(path);
    _BH = 0;
    _BL = drive;
    _AX = MAPROOT;
    geninterrupt(NETWARE);
    asm pop ds

    return(_AL);
}


/*-------------------------------------------------------------------------*/
/*                                                                         */
/*   DeleteFakeRoot                                                        */
/*                                                                         */
/*   This call deletes a fake root.  It does not delete the drive mapping. */
/*   If the drive is left in the subdirectory corresponding to the fake    */
/*   root.                                                                 */
/*                                                                         */
/*   Parameters:                                                           */
/*       drive - the drive to unmap  (0 = current, 1 = A, 2 = B, ...)      */
/*                                                                         */
/*   Returns:                                                              */
/*       none                                                              */
/*                                                                         */
/*-------------------------------------------------------------------------*/

void DeleteFakeRoot(int drive)
{
    _BH = 0;
    _BL = drive;
    _AX = DELETEROOT;
    geninterrupt(NETWARE);
}


/*-------------------------------------------------------------------------*/
/*                                                                         */
/*   GetRelativeDepth                                                      */
/*                                                                         */
/*   This call returns the number of current directories below the fake    */
/*   root for the specified drive.   This does not include the depth of    */
/*   the fake root itself.                                                 */
/*                                                                         */
/*   Parameters:                                                           */
/*       drive - the drive to map  (0 = current, 1 = A, 2 = B, ...)        */
/*                                                                         */
/*   Returns:                                                              */
/*      a number corresponding to the current directory depth.             */
/*      -1 indicates an error.   This can occur if the drive is not a      */
/*      fake root, if the drive is invalid, or if fake roots are not       */
/*      supported.                                                         */
/*                                                                         */
/*-------------------------------------------------------------------------*/

int GetRelativeDepth(int drive)
{
    int depth;

    _BH = 0;
    _BL = drive;
    _AX = ROOTDEPTH;
    geninterrupt(NETWARE);
    asm jc Error

    depth = _AL;

    if (depth == 0xff)
Error:
        return(-1);
    else
        return(depth);
}

