/**************************************************************************
** Copyright (c) 1996 Novell, Inc.  All Rights Reserved.
**
** THIS WORK IS SUBJECT TO U.S. AND INTERNATIONAL COPYRIGHT LAWS AND
** TREATIES.  USE AND REDISTRIBUTION OF THIS WORK IS SUBJECT TO THE
** LICENSE AGREEMENT ACCOMPANYING THE SOFTWARE DEVELOPMENT KIT (SDK)
** THAT CONTAINS THIS WORK.
** 
** Pursuant to the SDK License Agreement, Novell hereby grants to
** Developer a royalty-free, non-exclusive license to include the
** sample code SAPGENCB.C and derivative binaries in its product.
** Novell grants to Developer worldwide distribution rights to market,
** distribute or sell the sample code SAPGENCB.C and derivative
** binaries as a component of Developer's product(s).  Novell shall
** have no obligations to Developer or Developer's customers with
** respect to this code.
** 
** DISCLAIMER:
** 
** 	Novell, Inc. makes no representations or warranties with respect
** to the contents or use of this code, and specifically disclaims any
** express or implied warranties of merchantability or fitness for any
** particular purpose.  Further, Novell, Inc. reserves the right to revise
** this publication and to make changes to its content, at any time,
** without obligation to notify any person or entity of such revisions or
** changes.
**
** 	Further, Novell, Inc. makes no representations or warranties with
** respect to any software, and specifically disclaims any express or
** implied warranties of merchantability or fitness for any particular
** purpose.  Further, Novell, Inc. reserves the right to make changes to
** any and all parts of the software, at any time, without obligation to
** notify any person or entity of such changes.
**
***************************************************************************
**  File:   FSMON.C
**
**  Desc:   Main part of the FS Hooks (File System Monitor) sample
**
**	This file is primarily showing the (preparation for the) use of
**	NWAddFSMonitorHook() and NWRemoveFSMonitorHook(), plus it contains
**	the stub routines for all the hooks.
**
**  Programmers:
**
**      Ini Who                     Firm
**      -----------------------------------------------------------------------
**      JB  Jan Beulich	            Novell Developer Support
**
**  History:
**
**      When        Who What
**      -----------------------------------------------------------------------
**      06Jun96     jb  First code.
*/

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <signal.h>

#include <nwadv.h>
#include <nwdir.h>
#include <nwfshook.h>
#include <nwthread.h>

#include "fsmon.h"

struct displayRecord	*listHead = NULL;
int			initialized = N_FALSE;
int			mainThreadID, mainThreadGroupID;

static void	terminate(int);

#define HookFunc(name,id)						\
LONG hookPre##name##(##name##CallBackStruct*pCBS) {			\
	return hookPreHandler(FSHOOK_PRE_##id##,			\
		pCBS,							\
		sizeof(##name##CallBackStruct));			\
}									\
void hookPost##name##(##name##CallBackStruct*pCBS, LONG ccode) {	\
	hookPostHandler(FSHOOK_POST_##id##, pCBS, ccode);		\
}									\
struct HookEntry hookEntry##name##[2] =	{				\
	{FSHOOK_PRE_##id##,	(void*)hookPre##name##},		\
	{FSHOOK_POST_##id##,	(void*)hookPost##name##}		\
};

#define HookFuncGen(name,id)						\
LONG hookPreGen##name##(Generic##name##CBStruct*pCBS) {			\
	return hookPreHandler(FSHOOK_PRE_GEN_##id##,			\
		pCBS,							\
		sizeof(Generic##name##CBStruct));			\
}									\
void hookPostGen##name##(Generic##name##CBStruct*pCBS, LONG ccode) {	\
	hookPostHandler(FSHOOK_POST_GEN_##id##, pCBS, ccode);		\
}									\
struct HookEntry hookEntryGen##name##[2] =	{			\
	{FSHOOK_PRE_GEN_##id##,	(void*)hookPreGen##name##},		\
	{FSHOOK_POST_GEN_##id##,(void*)hookPostGen##name##}		\
};

#define hookTableStart hookEntryCreateFile
HookFunc(CreateFile,CREATEFILE)
HookFunc(CreateAndOpen,CREATE_OPENFILE)
HookFuncGen(OpenCreate,OPEN_CREATE)
HookFunc(OpenFile,OPENFILE)
HookFunc(CloseFile,CLOSEFILE)
HookFunc(EraseFile,ERASEFILE)
HookFuncGen(EraseFile,ERASEFILE)
HookFunc(RenameMoveEntry,RENAME_OR_MOVE)
HookFuncGen(Rename,RENAME)
HookFunc(CreateDir,CREATEDIR)
HookFunc(DeleteDir,DELETEDIR)
HookFunc(ModifyDirEntry,MODIFY_DIRENTRY)
HookFuncGen(ModifyDOSInfo,MODIFY_DOS_INFO)
HookFuncGen(ModifyNSInfo,MODIFY_NS_INFO)
HookFunc(RenameNSEntry,RENAME_NS_ENTRY)
HookFunc(SalvageDeleted,SALVAGE_DELETED)
HookFuncGen(SalvageDeleted,SALVAGE_DELETED)
HookFunc(PurgeDeleted,PURGE_DELETED)
HookFuncGen(PurgeDeleted,PURGE_DELETED)
static struct HookEntry hookTableEnd = {-1, NULL};
#undef HookFunc
#undef HookFuncGen

static const char	msgConn[]		= " connection %lu";
static const char	msgTask[]		= ", task %lu";
static const char	msgHandle[]		= ", handle %08lX";
static const char	msgAction[]		= ", action %02X";
static const char	msgSequence[]		= ", sequence %08lX";
static const char	msgAttributes[]		= "attributes";
static const char	msgAccessRights[]	= "access rights";
static const char	msgAccessMask[]		= "access mask";
static const char	msgAttrMatch[]		= "attribute match mask";
static const char	msgFlags[]		= "flags";
static const char	msgOperation[]		= "operation";
static const char	strCreateFile[]		= "Create";
static const char	strCreateAndOpenFile[]	= "Open/Create";
static const char	strOpenFile[]		= "Open";
static const char	strCloseFile[]		= "Close";
static const char	strEraseFile[]		= "Delete";
static const char	strCreateDir[]		= "Create directory";
static const char	strDeleteDir[]		= "Delete directory";
static const char	strRenameOrMove[]	= "Rename/Move";
static const char	strModifyDirEntry[]	= "Modify entry";
static const char	strSalvage[]		= "Salvage";
static const char	strPurge[]		= "Purge";
static const char	strRenameNSEntry[]	= "Modify namespace entry";
static const char	strModifyNSInfo[]	= "Modify namespace info";
static const char	strModifyDOSInfo[]	= "Modify DOS info";
static const char	strUnknown[]		= "Unknown operation";

static const struct idToString {
	LONG			id;
	const char		*string;
} idToStringTable[] = {
	{FSHOOK_PRE_CREATEFILE,		strCreateFile},
	{FSHOOK_PRE_CREATE_OPENFILE,	strCreateAndOpenFile},
	{FSHOOK_PRE_GEN_OPEN_CREATE,	strCreateAndOpenFile},
	{FSHOOK_PRE_OPENFILE,		strOpenFile},
	{FSHOOK_PRE_CLOSEFILE,		strCloseFile},
	{FSHOOK_PRE_ERASEFILE,		strEraseFile},
	{FSHOOK_PRE_GEN_ERASEFILE,	strEraseFile},
	{FSHOOK_PRE_CREATEDIR,		strCreateDir},
	{FSHOOK_PRE_DELETEDIR,		strDeleteDir},
	{FSHOOK_PRE_RENAME_OR_MOVE,	strRenameOrMove},
	{FSHOOK_PRE_GEN_RENAME,		strRenameOrMove},
	{FSHOOK_PRE_MODIFY_DIRENTRY,	strModifyDirEntry},
	{FSHOOK_PRE_SALVAGE_DELETED,	strSalvage},
	{FSHOOK_PRE_GEN_SALVAGE_DELETED,strSalvage},
	{FSHOOK_PRE_PURGE_DELETED,	strPurge},
	{FSHOOK_PRE_GEN_PURGE_DELETED,	strPurge},
	{FSHOOK_PRE_RENAME_NS_ENTRY,	strRenameNSEntry},
	{FSHOOK_PRE_GEN_MODIFY_NS_INFO,	strModifyNSInfo},
	{FSHOOK_PRE_GEN_MODIFY_DOS_INFO,strModifyDOSInfo},
	{(LONG)-1,			strUnknown}
};

const char strSetThreadContextSpecifier[] = "SetThreadContextSpecifier";

int main(/*int argc, char**argv*/) {
	struct HookEntry	*pHook;
	int 			(*pSetThreadContextSpecifier)(int, int);

	mainThreadID = GetThreadID();
	mainThreadGroupID = GetThreadGroupID();
	pSetThreadContextSpecifier = (int(*)(int,int))ImportSymbol(GetNLMHandle(), (char*)strSetThreadContextSpecifier);
	if(pSetThreadContextSpecifier != NULL) {
		pSetThreadContextSpecifier(mainThreadID, NO_CONTEXT);
		UnimportSymbol(GetNLMHandle(), (char*)strSetThreadContextSpecifier);
	}
	signal(SIGTERM, terminate);

	for(pHook = hookTableStart; pHook < &hookTableEnd; pHook++)
		NWAddFSMonitorHook(pHook->id, pHook->handler.func, &pHook->handler.handle);

	// now start operation (after having ignored log file accesses due
	// to the hooks being installed)
	initialized = N_TRUE;

	while(listHead != LIST_KILL) {
		struct displayRecord	*curRecord, *prevRecord = NULL;
		char			displayed = 0;

		// get semaphore
		curRecord = listHead;
		while(curRecord != NULL && curRecord != LIST_KILL) {
			if(curRecord->reference == NULL)
				break;
			curRecord = (prevRecord = curRecord)->link;
		}
		if(curRecord != NULL && curRecord != LIST_KILL)
			if(prevRecord == NULL)
				listHead = curRecord->link;
			else
				prevRecord->link = curRecord->link;
		// release semaphore

		if(curRecord != NULL && curRecord != LIST_KILL) {
			const struct idToString	*xlat = idToStringTable;
			while(xlat->id != (LONG)-1 && xlat->id != curRecord->id)
				xlat++;
			printf("%s ", xlat->string);
			switch(curRecord->id) {

#define PutName(name,member)						\
	showFileName(((##name##CallBackStruct*)curRecord->info)->volume,\
		curRecord->info + sizeof(##name##CallBackStruct),	\
		((##name##CallBackStruct*)curRecord->info)->##member##ComponentCount,\
		((##name##CallBackStruct*)curRecord->info)->nameSpace)
#define PutNameGen(name,member)						\
	showFileName(((Generic##name##CBStruct*)curRecord->info)->volume,\
		curRecord->info + sizeof(Generic##name##CBStruct),	\
		((Generic##name##CBStruct*)curRecord->info)->##member##ComponentCount,\
		((Generic##name##CBStruct*)curRecord->info)->nameSpace)
#define PutConn(name)							\
	printf(msgConn,							\
		((##name##CallBackStruct*)curRecord->info)->connection)
#define PutConnGen(name)						\
	printf(msgConn,							\
		((Generic##name##CBStruct*)curRecord->info)->connection)
#define PutTask(name)							\
	printf(msgTask,							\
		((##name##CallBackStruct*)curRecord->info)->task)
#define PutTaskGen(name)						\
	printf(msgTask,							\
		((Generic##name##CBStruct*)curRecord->info)->task)
#define PutHandle(name)							\
	if(curRecord->ccode == 0)					\
		printf(msgHandle,					\
			(LONG)((##name##CallBackStruct*)curRecord->info)->fileHandle)
#define PutSeqGen(name)							\
	printf(msgSequence,						\
		((Generic##name##CBStruct*)curRecord->info)->sequence)

				case FSHOOK_PRE_CREATEFILE:
#define pCBS ((CreateFileCallBackStruct*)curRecord->info)
					PutName(CreateFile, path);
					printf(" %s = %08lX, %s = %08lX\n",
						msgAttributes,
						pCBS->createAttributeBits,
						msgFlags,
						pCBS->createFlagBits);
					PutConn(CreateFile);
					PutTask(CreateFile);
					PutHandle(CreateFile);
					break;
#undef pCBS
				case FSHOOK_PRE_CREATE_OPENFILE:
#define pCBS ((CreateAndOpenCallBackStruct*)curRecord->info)
					PutName(CreateAndOpen, path);
					printf(" %s = %08lX, %s = %08lX, %s = %08lX\n",
						msgAttributes,
						pCBS->createAttributeBits,
						msgFlags,
						pCBS->createFlagBits,
						msgAccessRights,
						pCBS->requestedAccessRights);
					PutConn(CreateAndOpen);
					PutTask(CreateAndOpen);
					PutHandle(CreateAndOpen);
					break;
#undef pCBS
				case FSHOOK_PRE_GEN_OPEN_CREATE:
#define pCBS ((GenericOpenCreateCBStruct*)curRecord->info)
					PutNameGen(OpenCreate, path);
					printf(" %s = %08lX, %s = %08lX, %s = %08lX\n",
						msgAttributes,
						pCBS->createAttributes,
						msgFlags,
						pCBS->openCreateFlags,
						msgAccessRights,
						pCBS->requestedAccessRights);
					PutConnGen(OpenCreate);
					PutTaskGen(OpenCreate);
					if(curRecord->ccode == 0) {
						printf(msgHandle,
							(LONG)pCBS->fileHandle);
						printf(msgAction,
							(BYTE)pCBS->openCreateAction);
					}
					break;
#undef pCBS
				case FSHOOK_PRE_OPENFILE:
#define pCBS ((OpenFileCallBackStruct*)curRecord->info)
					PutName(OpenFile, path);
					printf(" %s = %08lX, %s = %08lX\n",
						msgAttrMatch,
						pCBS->attributeMatchBits,
						msgAccessRights,
						pCBS->requestedAccessRights);
					PutConn(OpenFile);
					PutTask(OpenFile);
					PutHandle(OpenFile);
					break;
#undef pCBS
				case FSHOOK_PRE_CLOSEFILE:
					printf("%08lX\n",
						((CloseFileCallBackStruct*)curRecord->info)->fileHandle);
					PutConn(CloseFile);
					PutTask(CloseFile);
					break;
				case FSHOOK_PRE_ERASEFILE:
					PutName(EraseFile, path);
					printf(" %s = %08lX\n",
						msgAttrMatch,
						((EraseFileCallBackStruct*)curRecord->info)->attributeMatchBits);
					PutConn(EraseFile);
					PutTask(EraseFile);
					break;
				case FSHOOK_PRE_GEN_ERASEFILE:
					PutNameGen(EraseFile, path);
					printf(" %s = %08lX\n",
						msgAttrMatch,
						((GenericEraseFileCBStruct*)curRecord->info)->searchAttributes);
					PutConnGen(EraseFile);
					PutTaskGen(EraseFile);
					break;
				case FSHOOK_PRE_RENAME_OR_MOVE:
#define pCBS ((RenameMoveEntryCallBackStruct*)curRecord->info)
					PutName(RenameMoveEntry, path);
					printf(" to ");
					showFileName(pCBS->volume,
						(char*)(pCBS + 1)
						+ getPathLength(0, 0,
							(void*)(pCBS + 1),
							pCBS->pathComponentCount),
						pCBS->originalNewCount,
						pCBS->nameSpace);
					printf(" %s = %08lX\n",
						msgAttrMatch,
						pCBS->attributeMatchBits);
					PutConn(RenameMoveEntry);
					PutTask(RenameMoveEntry);
					break;
#undef pCBS
				case FSHOOK_PRE_GEN_RENAME:
#define pCBS ((GenericRenameCBStruct*)curRecord->info)
#define volume srcVolume
#define dirBase srcDirBase
					PutNameGen(Rename, srcPath);
#undef dirBase
#undef volume
					printf(" to ");
					showFileName(pCBS->dstVolume,
						(char*)(pCBS + 1)
						+ getPathLength(0, 0,
							(void*)(pCBS + 1),
							pCBS->srcPathComponentCount),
						pCBS->dstPathComponentCount,
						pCBS->nameSpace);
					printf(" %s = %08lX\n",
						msgAttrMatch,
						pCBS->searchAttributes);
					PutConnGen(Rename);
					PutTaskGen(Rename);
					break;
#undef pCBS
				case FSHOOK_PRE_CREATEDIR:
					PutName(CreateDir, path);
					printf(" %s = %08lX\n",
						msgAccessMask,
						((CreateDirCallBackStruct*)curRecord->info)->directoryAccessMask);
					PutConn(CreateDir);
					break;
				case FSHOOK_PRE_DELETEDIR:
					PutName(DeleteDir, path);
					PutConn(DeleteDir);
					break;
				case FSHOOK_PRE_MODIFY_DIRENTRY: {
#define pCBS ((ModifyDirEntryCallBackStruct*)curRecord->info)
					struct ModifyStructure*modifyInfo;

					PutName(ModifyDirEntry, path);
					modifyInfo = (void*)((char*)(pCBS + 1) +
						getPathLength(0, 0,
						(void*)(pCBS + 1),
						pCBS->pathComponentCount));
					if(pCBS->modifyBits & MFileAttributesBit)
						printf(" Change attributes to %08lX with mask %08lX\n",
							modifyInfo->MFileAttributes,
							modifyInfo->MFileAttributesMask);
					if(pCBS->modifyBits & ~MFileAttributesBit) {
						printf(" Change");
						if(pCBS->modifyBits & MModifyNameBit)
							printf(" name;");
						if(pCBS->modifyBits & (
						    MCreateDateBit |
						    MCreateTimeBit))
							printf(" creation time;");
						if(pCBS->modifyBits & MOwnerIDBit)
							printf(" owner;");
						if(pCBS->modifyBits &
						   (MLastArchivedDateBit |
						    MLastArchivedTimeBit))
							printf(" archive time;");
						if(pCBS->modifyBits & MLastArchivedIDBit)
							printf(" archiver;");
						if(pCBS->modifyBits &
						   (MLastUpdatedDateBit |
						    MLastUpdatedTimeBit |
						    MLastUpdatedInSecondsBit))
							printf(" update time;");
						if(pCBS->modifyBits & MLastUpdatedIDBit)
							printf(" updater;");
						if(pCBS->modifyBits & MLastAccessedDateBit)
							printf(" access time;");
						if(pCBS->modifyBits & MInheritanceRestrictionMaskBit)
							printf(" inheritance filter;");
						if(pCBS->modifyBits & MMaximumSpaceBit)
							printf(" maximum space;");
						printf("\b\n");
					}
					printf(" %s = %08lX\n",
						msgAttrMatch,
						pCBS->attributeMatchBits);
					PutConn(ModifyDirEntry);
					PutTask(ModifyDirEntry);
					break;
#undef pCBS
				}
				case FSHOOK_PRE_GEN_MODIFY_DOS_INFO:
#define pCBS ((GenericModifyDOSInfoCBStruct*)curRecord->info)
					PutNameGen(ModifyDOSInfo, path);
					printf(" %s = %08lX, %s = %08lX\n",
						msgAttrMatch,
						pCBS->searchAttributes,
						msgOperation,
						pCBS->modifyMask);
					PutConnGen(ModifyDOSInfo);
					PutTaskGen(ModifyDOSInfo);
					break;
#undef pCBS
				case FSHOOK_PRE_GEN_MODIFY_NS_INFO:
					printf(" %s = %08lX\n",
						msgOperation,
						((GenericModifyNSInfoCBStruct*)curRecord->info)->modifyMask);
					PutConnGen(ModifyNSInfo);
					PutTaskGen(ModifyNSInfo);
					break;
				case FSHOOK_PRE_RENAME_NS_ENTRY:
					PutName(RenameNSEntry, path);
					PutConn(RenameNSEntry);
					PutTask(RenameNSEntry);
					break;
				case FSHOOK_PRE_SALVAGE_DELETED: {
#define pCBS ((SalvageDeletedCallBackStruct*)curRecord->info)
					LONG*pComponentCount = (LONG*)(curRecord->info + sizeof(SalvageDeletedCallBackStruct));

					showFileName(pCBS->volume,
						(char*)(pCBS + 1) + 2*sizeof(LONG),
						pComponentCount[0],
						pCBS->nameSpace);
					showFileName(pCBS->volume,
						(char*)(pCBS + 1) + 2*sizeof(LONG) +
						getPathLength(0, 0,
							(char*)(pCBS + 1) + 2*sizeof(LONG),
							pComponentCount[0]),
						pComponentCount[1],
						pCBS->nameSpace);
					PutConn(SalvageDeleted);
					break;
#undef pCBS
				}
				case FSHOOK_PRE_GEN_SALVAGE_DELETED: {
#define pCBS ((GenericSalvageDeletedCBStruct*)curRecord->info)
					LONG*pComponentCount = (LONG*)(pCBS + 1);

					showFileName(pCBS->volume,
						(char*)(pCBS + 1) + sizeof(LONG),
						*pComponentCount,
						pCBS->nameSpace);
					PutConnGen(SalvageDeleted);
					PutSeqGen(SalvageDeleted);
					break;
#undef pCBS
				}
				case FSHOOK_PRE_PURGE_DELETED: {
#define pCBS ((PurgeDeletedCallBackStruct*)curRecord->info)
					LONG*pComponentCount = (LONG*)(pCBS + 1);

					showFileName(pCBS->volume,
						(char*)(pCBS + 1) + 2*sizeof(LONG),
						pComponentCount[0],
						pCBS->nameSpace);
					showFileName(pCBS->volume,
						(char*)(pCBS + 1) + 2*sizeof(LONG) +
						getPathLength(0, 0,
							(char*)(pCBS + 1) + 2*sizeof(LONG),
							pComponentCount[0]),
						pComponentCount[1],
						pCBS->nameSpace);
					PutConn(PurgeDeleted);
					break;
#undef pCBS
				}
				case FSHOOK_PRE_GEN_PURGE_DELETED: {
#define pCBS ((GenericPurgeDeletedCBStruct*)curRecord->info)
					LONG*pComponentCount = (LONG*)(pCBS + 1);

					showFileName(pCBS->volume,
						(char*)(pCBS + 1) + sizeof(LONG),
						*pComponentCount,
						pCBS->nameSpace);
					PutConnGen(PurgeDeleted);
					PutSeqGen(PurgeDeleted);
					break;
#undef pCBS
				}
				default:
					break;
			}
#undef PutSeqGen
#undef PutHandle
#undef PutTaskGen
#undef PutTask
#undef PutConnGen
#undef PutConn
#undef PutNameGen
#undef PutName
			printf(", result = %08lX\n", curRecord->ccode);
			free(curRecord);
			displayed = 1;
		}
		if(!displayed)
			SuspendThread(mainThreadID);
	}

	// ignore log file accesses due to unhooking our hooks
	initialized = N_FALSE;

	for(pHook = hookTableStart; pHook < &hookTableEnd; pHook++)
		NWRemoveFSMonitorHook(pHook->id, pHook->handler.handle);

	listHead = NULL;
	return 0;
}

void terminate(int type) {
	struct displayRecord	*curRecord, *prevRecord = NULL;
	int			orgThreadGroupID = SetThreadGroupID(mainThreadGroupID);

	if(type == SIGTERM) {
		// get semaphore
		curRecord = listHead;
		while(curRecord != NULL) {
			curRecord->ccode = -1;
			curRecord->reference = NULL;
			curRecord = (prevRecord = curRecord)->link;
		}
		if(prevRecord == NULL)
			listHead = LIST_KILL;
		else
			prevRecord->link = LIST_KILL;
		// release semaphore
		ResumeThread(mainThreadID);
		while(listHead != NULL)
			ThreadSwitchWithDelay();
	}

	SetThreadGroupID(orgThreadGroupID);
}
