/********************************************************************
*
* TIMEIT.C.....Application Timer Library
*
* Author:      John P. Repko
* Last Update: 12/19/94
* Compiler:    WatCom C, version 9
* Fox Version: FoxPro 2.6
*
* Description: This library provides three functions to enable you
*              to add a series of up to four timed functions to your
*              application.
*
*              Since the Library is a true Windows Timer, it will
*              work regardless of the process the
*              user is performing (e.g.: menus, reads, etc...) or
*              even if your App is not the active Windows App.
*
*********************************************************************
*
* Function:    TimerOn(proc1,proctime1 [,proc2,proctime2...])
*
* Description: This function activates the timer.  Up to four
*              procedures can be included with different frequencies
*              if desired.
*
* Parameters:  proc1     - <char> - Required.  This is the name of
*                                   the procedure to "DO".
*              proctime1 - <int > - Required.  The frequency, in
*                                   seconds, that the procedure is
*                                   called.
*
* Returns:     Nothing.
*              
*********************************************************************
*
* Function:    TimerDead()
*
* Description: This function deactivates the timers without
*              unloading the library.  I can't think of why you would
*              want to use it, but it was useful for debugging.
*
* Parameters:  None.
*
* Returns:     Nothing.
*
*********************************************************************
*
* Function:    TimerShow()
*
* Description: This function identifies the procedures and frequencies
*              currently in place.
*
* Parameters:  None.
*
* Returns:     <char> string which lists the timers in effect.
*
*********************************************************************
*
* Sample Use:  *--Set up Timed Procedures--*
*              SET LIBRARY TO timeit ADDITIVE
*              =TimerOn("GETMSG",5,"SENDMSG",5,"CHECKUP",60,"OKL",30)
*              .
*              .
*              ? TimerShow()
*              .
*              .
*              PROCEDURE GETMSG
*              ...do something
*              RETURN
*              
*              PROCEDURE SENDMSG
*              ...do something
*              RETURN
*              
*              PROCEDURE CHECKUP
*              ...do something
*              RETURN
*              
*              PROCEDURE OKL
*              ...do something
*              RETURN
*
*********************************************************************/

/*--Include Files--*/
#include <windows.h>
#include <pro_ext.h>
#include <stdlib.h>
#include <string.h>

/*--Function Declarations--*/
void FAR CleanKill( void );
void FAR GoTimer( void );
void FAR TimerOn(ParamBlk FAR *parm);
void FAR TimerDead(ParamBlk FAR *parm);
void FAR TimerDead2(ParamBlk FAR *parm);
void FAR GetSetParms(ParamBlk FAR *parm);
void FAR NullTerminate(Value FAR *cVal);
void FAR TimerShow(ParamBlk FAR *parm) ;
void _export CALLBACK MyTimerProc(HWND foxwnd, WORD msg, WORD wParam, DWORD dwParam);

/*--Variable Declarations--*/
static unsigned int TimerFreq=1;
static HWND         hWndMain;
static unsigned int TimerID;
static BOOL         timer_on;
static HINSTANCE    hinst;
static TIMERPROC    lpfnMyTimerProc;
static char         proc1[18],proc2[18],proc3[18],proc4[18];
static unsigned int pTime1,pTime2,pTime3,pTime4;
static DWORD        pold1,pold2,pold3,pold4;

/********************************************************************
*
* FUNCTION:    MyTimerProc( ... )
*
* Type:        Internal Library Routine
*
* Description: The Timer Procedure that controls everything.
*
*********************************************************************/
void _export CALLBACK MyTimerProc(HWND foxwnd, WORD msg, WORD wParam, DWORD dwParam)
{
  DWORD cur_time;


  /*--Is it time?--*/

  if (pold1 > 0)
  {
     cur_time = GetTickCount();
     if ( ((cur_time - pold1)/1000) >= pTime1 )
     {
       CleanKill();
       pold1 = cur_time ;
       _Execute(proc1);
     }
  }

  if (pold2 > 0)
  {
     cur_time = GetTickCount();
     if ( ((cur_time - pold2)/1000) >= pTime2 )
     {
       CleanKill();
       pold2 = cur_time ;
       _Execute(proc2);
     }
  }

  if (pold3 > 0)
  {
     cur_time = GetTickCount();
     if ( ((cur_time - pold3)/1000) >= pTime3 )
     {
       CleanKill();
       pold3 = cur_time ;
       _Execute(proc3);
     }
  }

  if (pold4 > 0)
  {
     cur_time = GetTickCount();
     if ( ((cur_time - pold4)/1000) >= pTime4 )
     {
       CleanKill();
       pold4 = cur_time ;
       _Execute(proc4);
     }
  }

  /*--Re-Establish Timer--*/
  if (!timer_on)
  {
     GoTimer();
  }

}

/********************************************************************
*
* FUNCTION:    TimerShow(ParamBlk FAR *parm)
*
* Type:        Library Routine
*
* Description: Display Timers in Effect
*
*********************************************************************/
void FAR TimerShow(ParamBlk FAR *parm)
{
  char display[250],numchar[20];
  
  /*--Set Default--*/
  _StrCpy(display,"");
  
  /*--Build Display String--*/
  if (timer_on)
  {
     if (_StrLen(proc1) > 0)
     {
        itoa(pTime1,numchar,10);
        _StrCpy(display,proc1);
        strcat(display,"...every ");
        strcat(display,numchar);
        strcat(display," seconds\n");
     }
     if (_StrLen(proc2) > 0)
     {
        itoa(pTime2,numchar,10);
        strcat(display,proc2);
        strcat(display,"...every ");
        strcat(display,numchar);
        strcat(display," seconds\n");
     }
     if (_StrLen(proc3) > 0)
     {
        itoa(pTime3,numchar,10);
        strcat(display,proc3);
        strcat(display,"...every ");
        strcat(display,numchar);
        strcat(display," seconds\n");
     }
     if (_StrLen(proc4) > 0)
     {
        itoa(pTime4,numchar,10);
        strcat(display,proc4);
        strcat(display,"...every ");
        strcat(display,numchar);
        strcat(display," seconds\n");
     }
  }

  /*--Stuff it In--*/
  _RetChar(display);
}

/********************************************************************
*
* FUNCTION:    GetSetParms(ParamBlk FAR *parm)
*
* Type:        Internal Library Routine
*
* Description: Reassignment of passed parameters to static library
*              variables.
*
*********************************************************************/
void FAR GetSetParms(ParamBlk FAR *parm)
{
  WHANDLE wh;
  int     ok,tfreq,tdur;

  /* Set Local Values */
  ok = FALSE;
  timer_on  = FALSE;
  pold1 = 0;
  pold2 = 0;
  pold3 = 0;
  pold4 = 0;

  wh        = _WMainWindow();
  hWndMain  = _WhToHwnd(wh);

  /* Set Timed Procedures */
  if (parm->p[0].val.ev_type == 'C')
  {
     NullTerminate(&parm->p[0].val);
     _StrCpy(proc1,"DO ");
     strcat(proc1,_HandToPtr(parm->p[0].val.ev_handle));

     if (parm->p[1].val.ev_type == 'I')
     {
        pTime1 = parm->p[1].val.ev_long ;
        pTime1 = ( (pTime1 <= 0) ? 1 : pTime1 );
     }
     else
     {
        pTime1 = 1 ;
     }
     ok = TRUE;
     pold1 = GetTickCount();
  }
  else
  {
     _StrCpy(proc1,"");
     pTime1 = 0;
  }

  if (parm->p[2].val.ev_type == 'C')
  {
     NullTerminate(&parm->p[2].val);
     _StrCpy(proc2,"DO ");
     strcat(proc2,_HandToPtr(parm->p[2].val.ev_handle));

     if (parm->p[3].val.ev_type == 'I')
     {
        pTime2 = parm->p[3].val.ev_long ;
        pTime2 = ( (pTime2 <= 0) ? 1 : pTime2 );
     }
     else
     {
        pTime2 = 1 ;
     }
     ok = TRUE;
     pold2 = GetTickCount();
  }
  else
  {
     _StrCpy(proc2,"");
  }

  if (parm->p[4].val.ev_type == 'C')
  {
     NullTerminate(&parm->p[4].val);
     _StrCpy(proc3,"DO ");
     strcat(proc3,_HandToPtr(parm->p[4].val.ev_handle));

     if (parm->p[5].val.ev_type == 'I')
     {
        pTime3 = parm->p[5].val.ev_long ;
        pTime3 = ( (pTime3 <= 0) ? 1 : pTime3 );
     }
     else
     {
        pTime3 = 1 ;
     }
     ok = TRUE;
     pold3 = GetTickCount();
  }
  else
  {
     _StrCpy(proc3,"");
  }

  if (parm->p[6].val.ev_type == 'C')
  {
     NullTerminate(&parm->p[6].val);
     _StrCpy(proc4,"DO ");
     strcat(proc4,_HandToPtr(parm->p[6].val.ev_handle));

     if (parm->p[7].val.ev_type == 'I')
     {
        pTime4 = parm->p[7].val.ev_long ;
        pTime4 = ( (pTime4 <= 0) ? 1 : pTime4 );
     }
     else
     {
        pTime4 = 1 ;
     }
     ok = TRUE;
     pold4 = GetTickCount();
  }
  else
  {
     _StrCpy(proc4,"");
  }

  /* Establish Timer */
  if (ok)
  {
     GoTimer();
  }
}


/********************************************************************
*
* FUNCTION:    NullTerminate(Value FAR *cVal)
*
* Type:        Internal Library Routine
*
* Description: Null terminates character strings passed from FoxPro.
*
*********************************************************************/
void FAR NullTerminate(Value FAR *cVal)
{
  if (!_SetHandSize(cVal->ev_handle,cVal->ev_length+1))
  {
    _Error(182);
  }
  ((char FAR *) _HandToPtr(cVal->ev_handle))[cVal->ev_length]='\0';
}


/********************************************************************
*
* FUNCTION:    TimerOn(ParamBlk FAR *parm)
*
* Type:        Library Routine
*
* Description: Entry point to the whole shebang...
*
*********************************************************************/
void FAR TimerOn(ParamBlk FAR *parm)
{
  if (timer_on)
  {
    CleanKill();
  }
  GetSetParms(parm);
}


/********************************************************************
*
* FUNCTION:    TimerDead2(ParamBlk FAR *parm)
*
* Type:        Internal Library Routine
*
* Description: When the library is unloaded we must release the
*              Timer and Free the Proc Instance we Created.
*
*********************************************************************/
void FAR TimerDead2(ParamBlk FAR *parm)
{
  CleanKill();
}


/********************************************************************
*
* FUNCTION:    TimerDead(ParamBlk FAR *parm)
*
* Type:        Library Routine
*
* Description: When the library is unloaded we must release the
*              Timer and Free the Proc Instance we Created.
*
*********************************************************************/
void FAR TimerDead(ParamBlk FAR *parm)
{
  CleanKill();
}


/********************************************************************
*
* FUNCTION:    CleanKill( void )
*
* Type:        Internal Library Routine
*
* Description: Clean up the mess you created...
*
*********************************************************************/
void FAR CleanKill( void )
{
  if (timer_on)
  {
    timer_on = FALSE;

    KillTimer(hWndMain,TimerID);
    FreeProcInstance(lpfnMyTimerProc);
  }
}


/********************************************************************
*
* FUNCTION:    GoTimer( void )
*
* Type:        Internal Library Routine
*
* Description: Starts the Timer
*
*********************************************************************/
void FAR GoTimer( void )
{
  timer_on  = TRUE;
  hinst = GetWindowWord(hWndMain, GWW_HINSTANCE);
  lpfnMyTimerProc = (TIMERPROC) MakeProcInstance(MyTimerProc, hinst);
  TimerID = SetTimer(hWndMain,1,(TimerFreq * 1000),lpfnMyTimerProc);
}


/*--FoxPro Needs to know what you got!--*/
FoxInfo myFoxInfo[] =
{
  {"TIMERON",    (FPFI) TimerOn,    8,           ".?,.?,.?,.?,.?,.?,.?,.?"},
  {"TIMERDEAD",  (FPFI) TimerDead,  0,            ""       },
  {"TIMERSHOW",  (FPFI) TimerShow,  0,            ""       },
  {"TIMERDEAD2", (FPFI) TimerDead2, CALLONUNLOAD, ""       }
};

FoxTable _FoxTable =
{
	(FoxTable FAR *) 0, sizeof(myFoxInfo)/sizeof(FoxInfo), myFoxInfo
};

/*--EOF--*/
