/**************************************************************************
	Name: Printer Functions
	Date: 1/30/95
	by  : Dennis King
		  CIS - 75254,3153
	
	Libraries:
		FoxPro 2.6a
		
	Description:
		This library contains functions necessary for manipulating
		the printer.  It contains 2 functions as described below.
	
	Functions:
		FoxPro called:
			(Getpntsize)
			GetPointSize - Returns the largest point size a string
						   can be to fit into a width. This is a 
						   Best fit program.
			PrintSel - Configures a printer from a binary DEVMOD file.
			 
		Other:
			NullTerminate - Null Terminates a sring from FoxPro.
			Bestfit - Best fits a string in a width.                   
			InitFont - Iniatializes the LOGFONT structure.
			PrintStruct - Prints the structure for Bestfit (debug tool)
			

**************************************************************************/      
#include <windows.h>
#include <string.h>
#include <stdlib.h>
#include <pro_ext.h>
#include <print.h>

/* Magic number used for converting the pointsize 
   with respect to the DPI of the device.  */
#define MAGICNUM 72
#define FOXELUNIT 10000

/* Global Structures and variables */
char msg[100];            // Output message buffer 
HWND hWnd;                // Handle of the calling window 

/* Structures used for Bestfit */
struct DEVICEINFO
	{
		char FAR *lpsName;
		char FAR *lpsDriverName;
		char FAR *lpsPort;
	} DeviceInfo;

struct FOXVARS
	{
		unsigned int nMaxSize;
		unsigned int nMinSize;
		unsigned int nLengthtoFit;          
		int nFontStyle;
		char FAR *lpsFontFace;
		char FAR *lpsStringtoFit;
	} FoxVars;
		
/*****************************************/
/**        Function Prototypes          **/
/*****************************************/
void NullTerminate( Value FAR *);  
int Bestfit( HDC, struct FOXVARS * );
void PrintStructure( struct DEVICEINFO *, struct FOXVARS * );
void InitFont( LOGFONT *, struct FOXVARS *, int );
																		

/********************* Begin GetPointSize Functions *********************/

/*************************************************************************

	Name:  GetPointSize (Getpntsize)
	Date:  1/24/95
	Author: Dennis King
	
	Libraries:
		FoxPro 2.6a
	
	Description:
		This program will get a DC to the passed in printer.
		With the DC it will bestfit a string and return the
		point size back to foxpro.

	Revision History:
		1/26/95 - Changed length to be unsigned int. Also changed 
				  division of foxels to be unsigned long.                                                          
	Parameters:
		lpsStringtoFit - The string to best fit.
		nMinSize - The minimum point size.
		nMaxSize - The maximum point size.
		nLengthtoFit - Length the string needs to fit in. (in Foxels)
					   Foxels are 10000 units per inch.
		lpsFontFace - Name of the font.
		nFontStyle - Style of the font. This is a numeric (0-N,1-B,2-I,3-BI)
		lpsName - Device name.
		lpsDriverName - Driver name for the Device.
		lpsPort - Device port.
		Device information can be found in the win.ini file.

**************************************************************************/

/*****************************************/
/**     Main GetPointSize Procedure     **/
/*****************************************/
FAR GetPointSize( ParamBlk FAR *parm )
{               
	/* Variables passed in from FoxPro */
	struct DEVICEINFO Device;  /* Structure containing device info */
	struct FOXVARS Fox;        /* Structure containing string and font info */
	
	/* Other vars */
	HDC PrintDC;
	int nBestPointSize;
	int ret;                  /* Return value */

	/* Get the foxpro data and intialize the structures */
	NullTerminate( &parm->p[0].val );
	NullTerminate( &parm->p[4].val );
	NullTerminate( &parm->p[6].val );
	NullTerminate( &parm->p[7].val );
	NullTerminate( &parm->p[8].val );
	
	Fox.lpsStringtoFit = _HandToPtr(parm->p[0].val.ev_handle);
	Fox.nMinSize = (unsigned int)parm->p[1].val.ev_long;
	Fox.nMaxSize = (unsigned int)parm->p[2].val.ev_long;
	Fox.nLengthtoFit = (unsigned int)parm->p[3].val.ev_long;
	Fox.lpsFontFace = _HandToPtr(parm->p[4].val.ev_handle);
	Fox.nFontStyle = (int)parm->p[5].val.ev_long;
	Device.lpsName = _HandToPtr(parm->p[6].val.ev_handle);
	Device.lpsDriverName = _HandToPtr(parm->p[7].val.ev_handle);
	Device.lpsPort = _HandToPtr(parm->p[8].val.ev_handle);

	nBestPointSize = Fox.nMinSize;
	
	/* Get the handle to the calling procedure */
	hWnd = GetFocus();
																   
	/* Create a DC to the device */
	PrintDC = CreateDC(Device.lpsDriverName,Device.lpsName,Device.lpsPort,NULL);
	if (PrintDC != NULL)                                       
	{                                                                  
		/* Get the bestfit for the input */ 
		nBestPointSize = Bestfit(PrintDC,&Fox);
		DeleteDC(PrintDC);   
	}
	else                             
	{
		ret=MessageBox(hWnd,"Error creating DC to printer.","Best Fit",MB_OK);
	}                                                                  

	/* Return the point size to FoxPro */
	_RetInt(nBestPointSize,10);
} // End Get Point Size
									  

/**********************************************************************
	Name: Bestfit
	Date: 1/25/95
	By  : Dennis King
	Description:
		This function will find the largest point size that will fit
		into the desired length.
**********************************************************************/
int Bestfit(HDC hdc, struct FOXVARS *Fox)
{         
	int         charCnt;
	int         nOldMapMode;
	LOGFONT     lf;
	HFONT       hfont, hfontOld,tfont;
	int         nDeviceDPI;                      
	DWORD       dwExtent;
	unsigned int nPixelLength;
	
	/* Get device DPI */
	nDeviceDPI = GetDeviceCaps(hdc, LOGPIXELSY);    
	charCnt=strlen(Fox->lpsStringtoFit);
	
	/* Convert length to fit to pixels per inch. Foxels are 10000/inch */
	nPixelLength = (unsigned int)(((unsigned long)Fox->nLengthtoFit * (unsigned long)nDeviceDPI) / FOXELUNIT);
	
	/* Initialize the font structure lf */
	memset( &lf, 0, sizeof(lf) );
	InitFont( &lf, Fox, nDeviceDPI );
	
	/* Create the font */
	hfont = CreateFontIndirect(&lf);
	if (hfont != NULL)
	{
		/* Inialize and save environment */           
		nOldMapMode = SetMapMode(hdc,MM_TEXT);
		hfontOld = SelectObject(hdc, hfont);
				
		/* Get width of printed text */
		dwExtent = LOWORD(GetTextExtent(hdc,Fox->lpsStringtoFit,charCnt));
							   
		/* Do while width is longer than the output length and we have not
		   gone smaller than the minimum point size. */
		while ((dwExtent > nPixelLength) && (Fox->nMaxSize > Fox->nMinSize))
		{                                            
			/* Width was to large so decrease point size of the font */
			Fox->nMaxSize--;                                           
			
			/* This line is magic. It converts the point size with respect to
			   the DPI of the device. I have no idea where MAGICNUM comes from. */
			lf.lfHeight = -MulDiv(Fox->nMaxSize,nDeviceDPI,MAGICNUM);
			
			hfont = CreateFontIndirect(&lf);
			if (hfont != NULL)
			{           
				tfont = SelectObject(hdc, hfont);
				DeleteObject(tfont);
				dwExtent = (int)LOWORD(GetTextExtent(hdc,Fox->lpsStringtoFit,charCnt));
			} /* End if */
		
		} /* End while */
						 
		/* Reset things back to normal */
		SetMapMode(hdc,nOldMapMode);     
		SelectObject(hdc, hfontOld);
		DeleteObject(hfont);
	} /* end create font if */

	return Fox->nMaxSize;   

} // end function Bestfit

	
/**********************************************************************
	Name: Initialize Font
	Date: 1/25/95
	by  : Dennis King
	Description:
		This program will initialize the LOGFONT structure based on
		what was passed in from FoxPro.
***********************************************************************/
void InitFont( LOGFONT *lf, struct FOXVARS *f, int nDeviceDPI )
{
	/* This line is magic. It converts the point size with respect to
	   the DPI of the device. I have no idea where MAGICNUM comes from. */
	lf->lfHeight = -MulDiv(f->nMaxSize,nDeviceDPI,MAGICNUM);

	lf->lfWidth = 0;
	strcpy(lf->lfFaceName, f->lpsFontFace);
	
	if ( f->nFontStyle == 0)     
	{   /* Normal */
		lf->lfWeight = FW_NORMAL;
		lf->lfItalic = 0;
	}
	else if ( f->nFontStyle == 1)
	{   /* Bold */
		lf->lfWeight = FW_BOLD;
		lf->lfItalic = 0;
	}
	else if ( f->nFontStyle == 2)
	{   /* Italic */
		lf->lfWeight = FW_NORMAL;
		lf->lfItalic = 1;
	}
	else if ( f->nFontStyle == 3)
	{   /* Bold Italic */
		lf->lfWeight = FW_BOLD;
		lf->lfItalic = 1;
	} 
} // End InitFont function

/**********************************************************************
	Name: Print Structure
	Description:
		This function will print the foxpro and device structure to a
		messagebox. This is used for debugging.
**********************************************************************/
void PrintStructure( struct DEVICEINFO *d, struct FOXVARS *f )
{   
	int ret;  // Return value 

	/* Print device info */
	sprintf(msg,"Name = %s\nDriver = %s\nPort = %s",d->lpsName,d->lpsDriverName,d->lpsPort);
	ret=MessageBox(hWnd,msg,"Device Info",MB_OK);   
	
	/* Fox Info */
	sprintf(msg,"String = %s\nMin=%d, Max=%d, Len=%u\nFont = %s Style = %d",
	f->lpsStringtoFit, f->nMinSize, f->nMaxSize, f->nLengthtoFit, f->lpsFontFace, f->nFontStyle);
	ret=MessageBox(hWnd,msg,"Fox Info",MB_OK);   
} // End PrintStructure
									
/*********************** End GetPointSize Functions *********************/

/*********************** Begin Set Printer Functions ********************/
										  

/*************************************************************************

	Name:  Set Printer
	Date:  1/30/95
	Author: Dennis King
	
	Libraries:
		FoxPro 2.6a
	
    PARAMETERS:
    	PrinterName - Name of the printer to configure.
    	DriverName - Name of the driver for the printer
    	PrinterPort - Port printer is on.
    	InputFile - Input DEVMOD binary file. see below.
    	SaveFile - File name for the previos settings. This
    			   can be null if you don't want them saved.
        
        You can find PrinterName,DriverName,PrinterPort in the win.ini
        file.
    
	Description:
		This program will configure the printer based on
		the information in the InputFile. The Input file is a binary
		file that is the DEVMOD structure.  This file can be found
		in the TAG2 memo field of the report of the first record.
			 sample: USE rpt.frx
					 GO TOP
					 COPY MEMO tag2 TO "devmod.bin"
					 =SetPrnt("HP LaserJet III","HP3.drv","LPT1:","devmod.bin","SaveFile.bin")
                     USE
                     REPORT FORM rpt.frx TO PRINTER
    				 =SetPrnt("HP LaserJet III","HP3.drv","LPT1:","SaveFile.bin","")
                 
	Revision History:
		
**************************************************************************/

/*****************************************/
/**     Main Set Printer Procedure      **/
/*****************************************/
FAR SetPrnt( ParamBlk FAR *parm )
{               
	/* Variables passed in from FoxPro */
	struct DEVICEINFO Device;    // Structure containing device info
	char FAR *lpsInputPrintFile; // string for Printer file 
	char FAR *lpsSavePrintFile;  // string for the old Printer file
		
	/* Other vars */
	int ret;                  // Return value 
	HINSTANCE hinstPrntDev;   // Instance of the printer module
	LPFNDEVMODE lpDeviceMode; // Pointer to the ExtDeviceMode function
	LPDEVMODE UpdateDev;      // Pointer to the printer structure
	int SizeNeeded;           // Size of the printer structure
	OFSTRUCT ofFile;          // Open file structure
	HFILE hFile;              // Pointer to an open file
	
	
	hWnd = GetFocus();                                  
	
	/* Get FoxPro variables */
	NullTerminate( &parm->p[0].val );
	NullTerminate( &parm->p[1].val );
	NullTerminate( &parm->p[2].val );
	NullTerminate( &parm->p[3].val );
	/* Check to see if they want to save the old printer config */
	if ( ((char FAR *)_HandToPtr(parm->p[4].val.ev_handle))[0] != NULL)
		NullTerminate( &parm->p[4].val );

	Device.lpsName = _HandToPtr(parm->p[0].val.ev_handle);
	Device.lpsDriverName = _HandToPtr(parm->p[1].val.ev_handle);
	Device.lpsPort = _HandToPtr(parm->p[2].val.ev_handle);
	lpsInputPrintFile = _HandToPtr(parm->p[3].val.ev_handle);
	lpsSavePrintFile = _HandToPtr(parm->p[4].val.ev_handle);

	/* Load the device driver in memory */
	hinstPrntDev = LoadLibrary(Device.lpsDriverName);
			 
	/* If the library was loaded continue */
	if ( hinstPrntDev > 32 )
	{                                     
		/* Find out the address of the ExtDeviceMode procedure */
		lpDeviceMode = (LPFNDEVMODE) GetProcAddress(hinstPrntDev,"ExtDeviceMode");
		if ( lpDeviceMode != NULL )  
		{   
			/* Call ExtDeviceMode and get the size of the structure */
			SizeNeeded = (* lpDeviceMode) (hWnd,hinstPrntDev,UpdateDev,Device.lpsName,Device.lpsPort,NULL,NULL,0);
			
			/* Allocate the memory */
			UpdateDev = ( DEVMODE *) malloc(SizeNeeded);
			
			/* If everything is OK set the printer defaults */
			if ( UpdateDev != NULL )           
			{   
				/* Initialize the structure */
				memset( UpdateDev, 0, SizeNeeded );
				ret = (* lpDeviceMode) (hWnd,hinstPrntDev,UpdateDev,Device.lpsName,Device.lpsPort,NULL,NULL,DM_OUT_BUFFER);
				
				/* Save the previous printer settings */
				if (lpsSavePrintFile[0] != NULL)
				{                                  
					hFile = OpenFile( lpsSavePrintFile,&ofFile, OF_CREATE);
					ret = _lwrite(hFile, UpdateDev, SizeNeeded);
					_lclose(hFile);
				}
				
				/* Initialize structure with the new printer settings */
				hFile = OpenFile( lpsInputPrintFile,&ofFile, OF_READWRITE);
				ret = _lread(hFile, UpdateDev, SizeNeeded);
				_lclose(hFile);
				
				/* Update the printer with the new settings */
				ret = (* lpDeviceMode) (hWnd,hinstPrntDev,NULL,Device.lpsName,Device.lpsPort,UpdateDev,NULL,DM_IN_BUFFER|DM_OUT_DEFAULT);
			
				if ( ret != IDOK )                           
					ret=MessageBox(hWnd,"Error setting printer","Set Printer",MB_OK); 
				free( UpdateDev);
			} 
			else
			{
				ret = MessageBox(NULL,"Not enough memory for printer structure.","Set Printer",MB_OK);
			} // End if memory is OK              
		} 
		else
		{
			ret=MessageBox(NULL,"Error getting module handle","Set Printer",MB_OK);
		} // End if printer has ExtDeviceMode function
		FreeLibrary(hinstPrntDev);
	}
	else
	{
		sprintf(msg,"Error: %d loading %s",hinstPrntDev,Device.lpsDriverName);
		ret=MessageBox(NULL,msg,"Set Printer",MB_OK);   
	} // End if able load module 
} // End main program SetPrnt
		
/********************** End Set Printer Functions *********************/


/************************* Begin General Functions *********************/

/**********************************************************************
	Name: NullTerminate
	Description:
		 This procedure null terminates the current handle.
**********************************************************************/
void NullTerminate( Value FAR *cVal)
{

	if ( !_SetHandSize(cVal->ev_handle, cVal->ev_length + 1))
	{
		_Error(182); /* Insufficient memory */
	}
	((char FAR *) _HandToPtr(cVal->ev_handle))[cVal->ev_length] = '\0';
}


/*********************************************************************
			FoxPro lck code
**********************************************************************/
FoxInfo myFoxInfo[] = {
		{"GETPNTSIZE",GetPointSize,9,"C,I,I,I,C,I,C,C,C"},
		{"SETPRNT",SetPrnt,5,"C,C,C,C,C"},
};

FoxTable _FoxTable = {
		(FoxTable FAR *) 0, sizeof(myFoxInfo) / sizeof(FoxInfo), myFoxInfo
};
