/****************************************************************
*  NUMTOSTR.C                                                   *
*                                                               *
*  By        : Leonard Zerman                                   *
*  Notice    : Placed into the public domain by Leonard Zerman. *
*  Date      : 09/01/93                                         *
*  Version   : 1.00                                             *
*  Revision  : 0                                                *
*  Usage     :                                                  *
*  Operation :                                                  *
*  Compile   : CL /c /ALw /FPc /Gs NUMTOSTR.C                   *
*  Notes     : Only been tested with MS Visual C/C++.           *
*  Notes     : Only been tested under MS Windows.               *
****************************************************************/


/****************************************************************

CONTENTS:

*-----------------------* MAIN FUNCTIONS *----------------------*

  NumericToString       Converts a double floating point number 
                        into a numeric string representation.

****************************************************************/

/*----------------------* PRAGMAS *----------------------------*/

#pragma warning (disable : 4001 4100)

/*----------------------* C INCLUDES *-------------------------*/

#include <windows.h>
#include <math.h>

/*----------------------* LOCAL INCLUDES *---------------------*/

#include "numtostr.h"

/*----------------------* LOCAL DEFINES *----------------------*/

/* The number of zeros in PRECISION_NUMBER determains the      */
/* number of decimal places that are in the result. If you     */
/* change the number of zeros, you will also need to change    */
/* the .%07 in the wsprintf() function.                        */

#define PRECISION_NUMBER     (10000000L)
#define LOG_OF_2             (0.301029995)


/****************************************************************

FUNCTION NAME: NumericToString

PROTOTYPE:
  LPSTR WINAPI NumericToString(double dblFloat) 

PARAMETERS:
  double dblFloat
  Double floating point numer to convert.

RETURNS:
  The NumericToString function returns a pointer to the resultant
  string. The string is returned in a buffer, so later invocation
  will overwrite the previous contents. So be sure to save the
  results to a local buffer before calling the function again.

ERROR:
  None.

DESCRIPTION:
  The NumericToString function converts a double floating point
  number into a numeric string representation.

COMMENTS:
  This function was written because there is no support in the
  FoxPro Library Construction Kit to do the conversion. The other 
  normal was of doing this conversion also fail under FoxPro.

EXAMPLE:
  N/A.

MODIFICATION HISTORY:
  L. A. Zerman  09/01/93  Original version.

****************************************************************/

LPSTR WINAPI NumericToString(double dblFloat) 
{
   static char szNbrBuffer[35];

   int nExponent;
   int nPowerOf10;

   double dblInt;
   double dblFraction;
   double dblMantissa;

   unsigned long ulFraction;

   /* Get the fractional part and integer part of the number. */ 
   dblFraction = modf(dblFloat, &dblInt);
   nPowerOf10  = 1;

   /* If exponent is too big or too small, then show the */
   /* float in scientific notation.                      */

   if (dblFloat && (fabs(dblFloat) > 1E10 || 
                    fabs(dblFloat) < 1E-10))
   {
      double dblTempNumber;

      /* Convert the exponent from base 2 to base 10 */
      dblMantissa   = frexp(dblFloat, &nExponent);
      nPowerOf10    = (int)(LOG_OF_2 * nExponent);
      dblTempNumber = dblMantissa * 
                      pow(2.0,  (double)nExponent) / 
                      pow(10.0, (double)nPowerOf10);

      if (dblFloat < 0.0)
          dblTempNumber = -dblTempNumber;

      dblFraction = modf(dblTempNumber, &dblInt);
   }

   dblFraction = fabs(dblFraction) * PRECISION_NUMBER;
   ulFraction  = (unsigned long)dblFraction;

   /* Round the numer up, if the fraction is >= 0.5. */
   if (dblFraction - (double)ulFraction >= 0.5)
      ulFraction++;

   /* Use scientific notation? Display the exponent? */
   if (dblFloat && (fabs(dblFloat) > 1E10 || 
                    fabs(dblFloat) < 1E-10))
   {
      (void) wsprintf(szNbrBuffer, "%ld.%07luE%d", (long)dblInt, 
                      ulFraction, nPowerOf10);
   }
   else
   {
      (void) wsprintf(szNbrBuffer, "%ld.%07lu", (long)dblInt, 
                      ulFraction);
   }

   return (szNbrBuffer);
}

/*-----------------------* EOF NUMTOSTR.C *--------------------*/
