/* Opens a DBF file and gives some information about the structure */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <io.h>
#pragma pack(1)

/* Some dBASE constants */
#define VERSION "1.0b"
#define MAXFIELDS 255
#define FDASIZE 32
#define DBF_VERSION_MASK 7
#define DB4_MEMO_MASK 8
#define ANY_MEMO_MASK 128

/* A function to check for a specified extension in a file name string
   and add it if the extension wasn't found */
void FullName( char *fileName, char *fileExt );

/* Define header structure without including field descriptor array,
   see page E-2 of the dBASE IV Language Reference */
struct HEADER {
   unsigned char first_byte;
   unsigned char last_upd_yr;
   unsigned char last_upd_month;
   unsigned char last_upd_day;
   unsigned long num_recs;
   unsigned int head_bytes;
   unsigned int rec_bytes;
   unsigned char reserved_1[2];
   unsigned char incomplete;
   unsigned char is_encrypted;
   unsigned char reserved_for_lan[12];
   unsigned char has_mdx;
   unsigned char reserved_2[3];
} in_header ;

/* Define structure for array of field descriptors,
   see page E-3 of the dBASE IV Language Reference */
struct FDA {
   unsigned char field_name[11];
   unsigned char field_type; /* C D F L M N */
   unsigned char reserved1[4];
   unsigned char field_length; /* in binary */
   unsigned char field_decimal; /* in binary */
   unsigned char reserved2[2];
   unsigned char work_area_id;
   unsigned char reserved3[10];
   unsigned char has_tag; /* 01H if has tag in production MDX, 00H if not */
} fd_array[ MAXFIELDS + 1 ] ; /* MAXFIELDS + 1 so that we don't have to deal
                                 with array element 0 */

/* String for input file name */
char input_file[256];

void main( int argc, char *argv[] )
{
   unsigned int input_handle, x, field_counter;

   /* Make sure there was at least one command line argument besides
      program name */
   if( argc < 2 ) {
      printf( "\nDBRead Version %s by Martin R. Leon\n\n"\
         "Missing command line argument.\n"\
         "USAGE:\tDBREAD <DBF file name>\n", VERSION );
      exit(0);
   }

   strcpy( input_file, argv[1] );
   FullName( input_file, "DBF" );

   /* Open file for read-only in binary mode and make sure there are
      no errors openning it */
   if((input_handle = open(input_file, O_RDONLY | O_BINARY, S_IREAD)) == -1)
   {
      printf( "Unable to open file %s\n", input_file );
      exit( 0 );
   }

   /* Read record header to in_header and check for error */
   if( read( input_handle, &in_header, sizeof(in_header) ) == -1 ) {
      printf( "\nError reading header from file!\n" );
      if( close( input_handle ) != 0 )
         printf( "\nError closing file!\n" );
      else
         printf( "\nFile closed!\n" );
      exit(0);
   }

   /* Read in field descriptor array.  Read first field descriptor array
      then go into for loop until 0DH is read into the field name. */
   read( input_handle, &fd_array[1], FDASIZE );
   for( field_counter = 1; field_counter <= MAXFIELDS; field_counter++ )
   {
      if( fd_array[ field_counter ].field_name[ 0 ] == 0x0D )
      {
         field_counter--;
         break;
      }
      read( input_handle, &fd_array[field_counter + 1], FDASIZE );
   }

   /* Print out header information */
   printf( "\nDatabase name: %s \tDBF version: %d\n"\
      "Any Memo's ?  %c\t\t"\
      "dBASE IV Memo ?  %c\t"\
      "Last Update:  %d/%d/%d\n"\
      "Number of records:  %ld\t"\
      "Encrypted ?  %c\t\t"\
      "MDX ?  %c\n",
      input_file,
      in_header.first_byte & DBF_VERSION_MASK,
      ((in_header.first_byte & ANY_MEMO_MASK) ? 'Y' : 'N' ),
      ((in_header.first_byte & DB4_MEMO_MASK) ? 'Y' : 'N' ),
      in_header.last_upd_month,
      in_header.last_upd_day,
      in_header.last_upd_yr,
      in_header.num_recs,
      ((int)in_header.is_encrypted) ? 'Y' : 'N',
      ((int)in_header.has_mdx) ? 'Y' : 'N' );

   /* If encrypted, don't bother trying to read the rest of the structure */
   if( in_header.is_encrypted )
      return;

   /* Print out field information */
   printf( "\nField name\tType\tLen\tDec\tMDX\n\n" );

   for( x = 1; x <= field_counter; x++ )
      printf( "%-10s\t%c\t%d\t%d\t%c\n",
         fd_array[ x ].field_name,
         fd_array[ x ].field_type,
         fd_array[ x ].field_length,
         fd_array[ x ].field_decimal,
         ((fd_array[ x ].has_tag) ? 'Y' : 'N') );

   /* Close file */
   if( close( input_handle ) != 0 )
      printf( "\nError closing file!\n" );

   exit( 0 );
}


/* Adds an extension to a string containing a file name if the specified
   extension isn't already in the string.  String must be larger than the
   combination of the existing string and the extension to be added.
   All done through pointer to original string. */

void FullName( char *fileName, char *fileExt )
{
   /* Set ptr to last character before terminator in fiel name */
   char *ptr = fileName + ( strlen( fileName ) - 1 );

   /* Check for extension already in name */
   if( strstr( fileName, fileExt ) != NULL )
      return;

   /* Extension wasn't found, back up ptr to first non-alpha character,
      stop if we got to beginning of string */
   while( isalpha( (int)*ptr ) )
   {
      if( --ptr == fileName )
         break;
   }

   /* If we backed up to a slash, add ".DBF" to the end of the file name
      and return */
   if( *ptr == '\\' )
   {
      fileName = strcat( fileName, ".DBF" );
      return;
   }

   /* If we backed up to a period, return without adding extension */
   if( *ptr == '.' )
      return;

   /* Add extension, since it wasn't found */
   strcat( fileName, ".DBF" );
   return;
}


