/*
 *	Programmer:	Timothy E. Hoff, thoff@falcon.depaul.edu
 *	File name:	dbinfo.c release 1.0
 *	Written:	April 10, 1994
 *	Modification log:
 *
 *	This program provides a quick way to get information about
 *	the structure of a FoxPro dbf file including record count,
 *	last modification date, and record structure.
 *
 *	This program is free software which may be modified and/or
 *	redistributed under the terms of the GNU General Public
 *	License as published by the Free Software Foundation.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "dbf.h"

#define MAX_FILENAME 255

int main(int argc, char *argv[])
{	
	FILE *fp;						/*	Pointer to dbf file	*/
	DBF_HEADER *header;				
	char dbfname[255] = "";			/*	Path and filename	*/
	FIELD *fieldlist[MAX_FIELDS];
	int i;

	
	/*	Initialize fields list	*/
	for (i=0; i<MAX_FIELDS; i++)
		fieldlist[i] = NULL;

		
	/*	Allocate memory to hold header information	*/
	
	if ((header = (DBF_HEADER *)malloc(sizeof(DBF_HEADER))) == NULL) {
		fprintf(stderr, "Out of memory\n");
		return 0;
	}


	/*	Get dbf filename	*/

	if (argc == 1) {
		getfilename(dbfname);
		dosext(dbfname);
	} else {
		strcpy(dbfname, *++argv);
		dosext(dbfname);
	}


	/*	Open DBF file for reading	*/

	if ((fp = fopen(dbfname, "rb")) == NULL) {
		fprintf(stderr, "ERROR:  Could not open %s\n", dbfname);
		
		free(header);
		return EXIT_FAILURE;
	}


	/*	Read database header	*/
	
	if (! readDBF(header, fp)) {
		fprintf(stderr, "ERROR:  Could not read DBF header.\n");
		fprintf(stderr, "Header is corrupt or this is not a DBF.\n");
		
		fclose(fp);
		free(header);
		return EXIT_FAILURE;
	}
	

	/*	Read field subrecords	*/
	
	if (! readREC(fieldlist, header->rec_size, fp)) {
		fprintf(stderr, "ERROR:  Invalid field records.\n");

		fclose(fp);
		free(header);

		for (i=0; i<MAX_FIELDS; i++)
			free(fieldlist[i]);
		return EXIT_FAILURE;
	}
	
	
	/*	Print DBF header & field subrecord information	*/
	
	printhdr(header);
	printrec(fieldlist);


	/*	Close file and release memory	*/
	
	free(header);
	fclose(fp);

	return EXIT_SUCCESS;
}


/* * * * * * * * *
 *	FUNCTIONS	 *
 * * * * * * * * */
 
void getfilename(char *file)
{
	/*	Prompt the user for the name of a file and store it in file	*/

	fprintf(stdout, "Enter the name of a DBF file:  ");
	fscanf(stdin, "%s", file);

	return;
}


void dosext(char *file)
{
	/*
	 *	Check the string pointer to by file for an extension.
	 *	If there is not an extension, append the extension .dbf
	 *	to the string.  Return.
	 */

	if (strchr(file, '.') == NULL)
		strcat(file, ".dbf");

	return;
}


int readDBF(DBF_HEADER *header, FILE *fp)
{
	/*
	 *	Return into header the first 32-bytes from fp.
	 *	Test the type flag to make sure that this is a DBF
	 *	file.  If the file is valid return EXIT_SUCCESS,
	 *	otherwise return EXIT_FAILURE.
	 */

	if (fread(header, sizeof(DBF_HEADER), 1, fp)) {
		if (header->type == 3 ||			/*	'\x03'	*/
			header->type == 131 ||			/*	'\x83'	*/
			header->type == 245 ||			/*	'\xF5'	*/
			header->type == 139)			/*	'\x8B'	*/
			return 1;
		else
			return 0;
	} else {
		fprintf(stderr, "FILE ERROR:  Could not read from %d\n", fp);
		return 0;
	}

}


int	readREC(FIELD *fieldlist[], X_INT rec_size, FILE *fp)
{
	/*
	 *	Read field subrecords into field list until the sum
	 *	of their sizes equals rec_size.  Read the next byte
	 *	from fp and check that it is the header record
	 *	terminator ('\x0D').  If the field subrecords were
	 *	read successfully return 1, otherwise return 0.
	 */

	int length;					/*	The size in bytes of one record	*/
	int i = 0;					/*	Position in fieldlist	*/
	FIELD *node;
	char terminator;			/*	Header record terminator	*/

	length = xtoint(rec_size);
	length -= 1;				/*	Subtract the delete flag from the length	*/

	while (length > 0) {
		if ((node = (FIELD *)malloc(sizeof(FIELD))) == NULL) {
			fprintf(stderr, "Out of memory\n");
			return 0;
		} else {
			if (! fread(node, sizeof(FIELD), 1, fp)) {
				fprintf(stderr, "ERROR:  Could not read from %d\n", fp);
				return 0;
			}

			fieldlist[i++] = node;
			length -= node->length;
		}
	}

	if ((terminator = fgetc(fp)) == '\x0D')
		return 1;
	else
		return 0;
}

	 
void printhdr(DBF_HEADER *header)
{
	/*	Print basic information from the DBF header	*/

	fprintf(stdout, "\n\tNumber of data records:\t%ld\n", xtolong(header->records));
	fprintf(stdout, "\tLength of one record:\t%d\n", xtoint(header->rec_size));
	fprintf(stdout, "\tDate of last update:\t%d/%d/%d\n",
			header->last_update.month, header->last_update.day, header->last_update.year);
	fprintf(stdout, "\n");
	
	return;
}


void printrec(FIELD *fieldlist[])
{
	/*	Print field subrecords	*/

	int i;

	fprintf(stdout, "\tField\tField Name\tType\tWidth\tDec\n");

	for (i=0; i<MAX_FIELDS; i++) {
		if (fieldlist[i] == NULL)
			break;
		else
			fprintf(stdout, "\t%d\t%-10s\t%c\t%d\t%d\n", (i+1),
				fieldlist[i]->field, fieldlist[i]->type,
				fieldlist[i]->length, fieldlist[i]->precision);
	}

	return;
}

	
/* * * * * * * * * * * * * * * * * * * * *
 *	HEXADECIMAL CONVERSION FUNCTIONS	 *
 * * * * * * * * * * * * * * * * * * * * */

long xtolong(X_LONG hexval)
{
	/*
	 *	Take a value in a 4-byte hex format and convert it
	 *	to a long.  Return the value as a long.
	 */
	 
	long value = 0;

	value += (hexval.l_endian);
	value += (hexval.s_endian * 256L);
	value += (hexval.m_endian * 65536L);
	value += (hexval.b_endian * 16777216L);

	return value;
}


int xtoint(X_INT hexval)
{
	/*
	 *	Take a value in a 2-byte hex format and convert it
	 *	to an int.  Return the value as an int.
	 */

	int value = 0;

	value += (hexval.l_endian);
	value += (hexval.b_endian * 256);

	return value;
}


X_LONG *longtox(long value, X_LONG *hexval)
{
	/*
	 *	Convert value from type long into a 4-byte hex format
	 *	pointed to by hexval.  Return a pointer to the
	 *	formatted value.
	 */

	int quotient;

	quotient = value / 16777216L;
	hexval->b_endian = quotient;
	value -= (hexval->b_endian * 16777216L);

	quotient = value / 65536L;
	hexval->m_endian = quotient;
	value -= (hexval->m_endian * 65536L);

	quotient = value / 256L;
	hexval->s_endian = quotient;
	value -= (hexval->s_endian * 256L);

	hexval->l_endian = value;
	
	return hexval;
}


X_INT *inttox(int value, X_INT *hexval)
{
	/*
	 *	Convert value from type int into a 2-byte hex format
	 *	pointed to by hexval.  Return a pointer to the
	 *	formatted value.
	 */

	int quotient;

	quotient = (value / 256);

	hexval->b_endian = quotient;
	value -= (hexval->b_endian * 256);

	hexval->l_endian = value;

	return hexval;
}
