; 
; Program Name - IPRS
;  Script Name - Utility3.sc
; Library Name - Global.lib
;
; This script contains processes that are used by the Built-in Calculator:
;
;        Calculator!Open
;        Calculator!SetCharKeys
;        Calculator!SetOpKeys
;        Calculator!Event
;        Calculator!Escape
;        Calculator!OK
;        Calculator!Menu
;        Calculator!Op
;        Calculator!Clear
;        Calculator!Percent
;        Calculator!Sqrt
;        Calculator!Round
;        Calculator!SaveMem
;        Calculator!GetMem
;        Calculator!Factorial
;        Calculator!PlusMinus
;        Calculator!Exponent
;        Calculator!Multiply
;        Calculator!Divide
;        Calculator!InfixOperator
;        Calculator!PrepStackForOp
;        Calculator!DoInfixOperator
;        Calculator!Add
;        Calculator!Subtract
;        Calculator!AddSubtract
;        Calculator!DoAddSubtract
;        Calculator!Equals
;        Calculator!Subtotal
;        Calculator!DoSubTotal
;        Calculator!ResolveStack
;        Calculator!NumKey
;        Calculator!Decimal
;        Calculator!RefreshTape
; 

; libname = "Libs\\global"
libname = "Calc"

;============================================================================
; Control trigger procs
;============================================================================

;----------------------------------------------------------------------------
; Called once at start of session, ready to take any legal character...
;----------------------------------------------------------------------------

Proc Calculator!Open()
   Calculator!SetCharKeys()
   Return true
EndProc ; Calculator!Open

WRITELIB libname Calculator!Open
RELEASE PROCS    Calculator!Open

;----------------------------------------------------------------------------
; Adds keycode list of all legal calculator keys. Supports the
; Calculator_WAITING state.
;----------------------------------------------------------------------------

Proc Calculator!SetCharKeys()
   NewDialogSpec
   Key
     Calculator_CharEventKey[1],
     Calculator_CharEventKey[2],
     Calculator_CharEventKey[3],
     Calculator_CharEventKey[4],
     Calculator_CharEventKey[5],
     Calculator_CharEventKey[6],
     Calculator_CharEventKey[7],
     Calculator_CharEventKey[8],
     Calculator_CharEventKey[9],
     Calculator_CharEventKey[10],
     Calculator_CharEventKey[11],
     Calculator_CharEventKey[12],
     Calculator_CharEventKey[13],
     Calculator_CharEventKey[14],
     Calculator_CharEventKey[15],
     Calculator_CharEventKey[16],
     Calculator_CharEventKey[17],
     Calculator_CharEventKey[18],
     Calculator_CharEventKey[19],
     Calculator_CharEventKey[20],
     Calculator_CharEventKey[21],
     Calculator_CharEventKey[22],
     Calculator_CharEventKey[23],
     Calculator_CharEventKey[24],
     Calculator_CharEventKey[25],
     Calculator_CharEventKey[26],
     Calculator_CharEventKey[27],
     Calculator_CharEventKey[28],
     Calculator_CharEventKey[29],
     Calculator_CharEventKey[30],
     Calculator_CharEventKey[31],
     Calculator_CharEventKey[32],
     Calculator_CharEventKey[33]
EndProc ; Calculator!SetCharKeys

WRITELIB libname Calculator!SetCharKeys
RELEASE PROCS    Calculator!SetCharKeys

;----------------------------------------------------------------------------
; Adds keycode list of legal operator type calculator keys. Supports the
; Calculator_CALC state.
;----------------------------------------------------------------------------

Proc Calculator!SetOpKeys()
   NewDialogSpec
   Key
     Calculator_OpEventKey[1],
     Calculator_OpEventKey[2],
     Calculator_OpEventKey[3],
     Calculator_OpEventKey[4],
     Calculator_OpEventKey[5],
     Calculator_OpEventKey[6],
     Calculator_OpEventKey[7],
     Calculator_OpEventKey[8],
     Calculator_OpEventKey[9],
     Calculator_OpEventKey[10],
     Calculator_OpEventKey[11],
     Calculator_OpEventKey[12],
     Calculator_OpEventKey[13],
     Calculator_OpEventKey[14],
     Calculator_OpEventKey[15]
EndProc ; Calculator!SetOpKeys

WRITELIB libname Calculator!SetOpKeys
RELEASE PROCS    Calculator!SetOpKeys

;----------------------------------------------------------------------------
; Traps for EVENT type messages. In particular traps for legal
; keys. Once trapped, they are classified and dispatched, followed
; by processing of the operator or function.
; Numchar and Editchar types cause the CellStatus to be set to CALC
; (something new has happened.), with the associated more limited eventlist.
; This leads to some performance improvement, as well as greater refinement
; in calculator logic.
;----------------------------------------------------------------------------

Proc Calculator!Event()
   Private
      OK
   Switch
      Case EventBag["Type"] = "KEY":
         Calculator_Keycode = EventBag["Keycode"]
         IF Calculator_KeyCode = 27 THEN                    ; SG
            OK = False                                      ; so Esc key works
            CANCELDIALOG                                    ;
         ELSE                                               ;
            Calculator_KeycodeAttribute =
            Calculator_KeycodeType[Calculator_Keycode]
            If (Calculator_KeyCodeAttribute = Calculator_OPCHAR) or
               (Calculator_KeyCodeAttribute = Calculator_FUNCCHAR) Then
               If TagValue <> "Cell" and Calculator_KeyCode = Asc("Enter") Then
                  OK = True ; Allow the keystroke
               Else
                  Calculator.Value = ControlValue("Cell")
                  ExecProc Calculator_ResponseProc[Calculator_Keycode]
                  OK = False
               Endif
            Else ; NUMCHAR or EDITCHAR
               Calculator_CellStatus = Calculator_CALC
               Calculator!SetOpKeys()
               OK = True
            Endif
        ENDIF
   EndSwitch
   Return OK
EndProc ; Calculator!Event

WRITELIB libname Calculator!Event
RELEASE PROCS    Calculator!Event

;----------------------------------- OPTIONS ----------------------------------

Proc Calculator!Escape()
   Calculator_Value = ControlValue("Cell")
;  CancelDialog                              ; per Henrik
   ; Calculator!Clear()
   Return True
EndProc ; Calculator!Escape

WRITELIB libname Calculator!Escape
RELEASE PROCS    Calculator!Escape

;----------------------------------------------------------------------------

Proc Calculator!OK()
   If Calculator_StackPtr > 0 Then
      Calculator!Equals()
   Endif
;  AcceptDialog                              ; per Henrik
   Return true
EndProc ; Calculator!OK

WRITELIB libname Calculator!OK
RELEASE PROCS    Calculator!OK

;----------------------------------------------------------------------------

Proc Calculator!Menu()
   Private
      OK,
      Command,
      TapeChoice

   OK = True
   If Calculator_TapeIsOn Then
      TapeChoice = " ~T~ape Toggle"
   Else
      TapeChoice = "  ~T~ape Toggle"
   Endif

   ; I added Reset and Print options, and changed the View option

   ShowPopup "Options" Centered
      TapeChoice:" Toggle tape on and off":"TapeOn",
      "  ~V~iew Tape":" Full view of the \"paper\" tape.":"ViewTape",
      "  ~R~eSet Tape":" Clear existing data from the \"paper\" tape.":"Reset",
      "  ~P~rint Tape":" Print the contents of the \"paper\" tape.":"Print",
      Separator,
      "  ~E~xplain":" A help screen which explains the use of the calculator."
                   :"Explain"
   EndMenu
   To Command
   If Retval Then
      Switch
         Case Command = "TapeOn":
            Calculator_TapeIsOn = Not Calculator_TapeIsOn
            Calculator!TapeModeLine()
         Case Command = "ViewTape":
              IF (DYNARRAYSIZE(Calculator_TapeList) > 0) THEN
                 Calculator!ViewTape()
              ELSE
                 Dia_OK(TRUE,"Message","No Paper Tape to View","")
              ENDIF

         Case Command = "Reset" :                                    ; SG
              MESSAGE "Clearing paper tape.."
              SLEEP 1000
              FOREACH element IN Calculator_TapeList
                 RELEASE VARS Calculator_TapeList[element]
              ENDFOREACH
              Calculator_TapeCtr = 1
              MESSAGE ""

         Case Command = "Print" :                                    ; SG
              IF (DYNARRAYSIZE(Calculator_TapeList) > 0) THEN
                 IF PRINTERSTATUS() THEN
                    Calculator!PrintTape()
                 ELSE
                    Dia_OK(TRUE,"Error","Printer Not Ready","")
                 ENDIF
              ELSE
                 Dia_OK(TRUE,"Message","No Paper Tape to Print","")
              ENDIF

         Case Command = "Explain":
            Calculator!ViewExplain()
      EndSwitch
   Else
      OK = false
   Endif
   SelectControl "Cell"
   Return OK
EndProc ; Calculator!Menu

WRITELIB libname Calculator!Menu
RELEASE PROCS    Calculator!Menu

;----------------------------------------------------------------------------

Proc Calculator!Op()
   Private
      OK,
      Command
   ShowPopup "Operators" Centered
      "~A~dd":"The operator [+]":"+",
      "S~u~btract":"The operator [-]":"-",
      "~M~ultiply":"The operator [*]":"*",
      "~D~ivide":"The operator [/]":"/",
      "~E~xponent":"The operator [^]":"^",
      Separator,
      "~P~ercent":"The function [%]":"%",
      "~F~actorial":"The function [!]":"!",
      "~R~everse sign":"The function [Alt-R]":"R",
      "S~q~uare root":"The square root function [Alt-Q]":"",
      "Rou~n~d,2":"Round to 2 decimal places [Alt-N]":"N",
      Separator,
      "Sa~v~e memory":"Save the current cell value to memory [>]":">",
      "~G~et memory":"Retrieve the current cell value from memory [<]":"<",
      Separator,
      "~S~ubtotal":"Subtotal operator [Alt-S]":"S",
      "~T~otal":"Total operator [=] or [Enter]":"=",
      Separator,
      "C~l~ear":"Clear operator [Alt-L]":"C"
   EndMenu
   To Command
   If Retval Then
      ExecProc Calculator_ResponseProc[Asc(Command)]
      OK = true
   Else
      OK = false
   Endif
   SelectControl "Cell"
   Return OK
EndProc ; Calculator!Op

WRITELIB libname Calculator!Op
RELEASE PROCS    Calculator!Op

;----------------------------------------------------------------------------
; Clear abandons the current calculation, and resets everything as though it
; had just been initialized. Stack items are abandoned (garbage) rather than
; re-initialized or deleted.
;----------------------------------------------------------------------------

Proc Calculator!Clear()
   Private
      OK
   OK = True
   Calculator_DecimalFlag = False
   Calculator!KeyPadModeLine()
   Calculator_CellStatus = Calculator_WAITING
   Calculator!SetCharKeys()
   Calculator.Value = 0
   Calculator!RefreshTape(BlankNum(),"Clear")
   Calculator_CellAttribute = Calculator_FLOATING
   Calculator_CellLastValue = Calculator.Value
   Calculator_CellLastAttribute = Calculator_CellAttribute
   Calculator_StackPtr = 0
   ResyncControl "Cell"
   SelectControl "Cell"
   Return OK
EndProc ; Calculator!Clear

WRITELIB libname Calculator!Clear
RELEASE PROCS    Calculator!Clear

;------------------------------ FUNCTIONS -----------------------------------

Proc Calculator!Percent()
   Private
      OK
   If Not IsBlank(Calculator.Value) Then
      OK = True
      Calculator_LastOperator = Calculator_CurrentOperator
      Calculator_CurrentOperator = "%"
      Calculator!RefreshTape(Calculator.Value,"% =")
      Calculator.Value = Calculator.Value/100
      Calculator!RefreshTape(Calculator.Value,":")
      Calculator_CellLastValue = Calculator.Value
      Calculator_CellLastAttribute = Calculator_CellAttribute
      Calculator_CellStatus = Calculator_CALC
      Calculator!SetOpKeys()
      ResyncControl "Cell"
   Else
      OK = false
      Beep
      Message "Cannot take percent of blank value."
      Sleep 2000
      Message ""
   Endif
   SelectControl "Cell"
   Return OK
EndProc ; Calculator!Percent

WRITELIB libname Calculator!Percent
RELEASE PROCS    Calculator!Percent

;----------------------------------------------------------------------------

Proc Calculator!Sqrt()
   Private
      OK
   If Not IsBlank(Calculator.Value) And
      Not (Calculator.Value < 0) Then
      OK = True
      Calculator_LastOperator = Calculator_CurrentOperator
      Calculator_CurrentOperator = ""
      Calculator!RefreshTape(Calculator.Value," =")
      Calculator.Value = Sqrt(Calculator.Value)
      Calculator!RefreshTape(Calculator.Value,":")
      Calculator_CellLastValue = Calculator.Value
      Calculator_CellLastAttribute = Calculator_CellAttribute
      Calculator_CellStatus = Calculator_CALC
      Calculator!SetOpKeys()
      ResyncControl "Cell"
   Else
      OK = false
      Beep
      Message "Cannot take square root () of blank or negative value."
      Sleep 2000
      Message ""
   Endif
   SelectControl "Cell"
   Return OK
EndProc ; Calculator!Sqrt

WRITELIB libname Calculator!Sqrt
RELEASE PROCS    Calculator!Sqrt

;----------------------------------------------------------------------------

Proc Calculator!Round()
   Private
      OK
   If Not IsBlank(Calculator.Value) Then
      OK = True
      Calculator_LastOperator = Calculator_CurrentOperator
      Calculator_CurrentOperator = "N"
      Calculator!RefreshTape(Calculator.Value,"rnd,2 =")
      Calculator.Value = Round(Calculator.Value,2)
      Calculator!RefreshTape(Calculator.Value,":")
      Calculator_CellLastValue = Calculator.Value
      Calculator_CellLastAttribute = Calculator_CellAttribute
      Calculator_CellStatus = Calculator_CALC
      Calculator!SetOpKeys()
      ResyncControl "Cell"
   Else
      OK = false
      Beep
      Message "Cannot round blank value."
      Sleep 2000
      Message ""
   Endif
   SelectControl "Cell"
   Return OK
EndProc ; Calculator!Round

WRITELIB libname Calculator!Round
RELEASE PROCS    Calculator!Round

;----------------------------------------------------------------------------

Proc Calculator!SaveMem()
   Private
      OK
   OK = True
   Calculator_MemoryValue = Calculator.Value
   Calculator_MemoryAttribute = Calculator_CellAttribute
   Calculator!RefreshTape(Calculator.Value,">")
   SelectControl "Cell"
   Return OK
EndProc ; Calculator!SaveMem

WRITELIB libname Calculator!SaveMem
RELEASE PROCS    Calculator!SaveMem

;----------------------------------------------------------------------------

Proc Calculator!GetMem()
   Private
      OK
   OK = True
   Calculator_LastOperator = Calculator_CurrentOperator
   Calculator_CurrentOperator = "<"
   Calculator.Value = Calculator_MemoryValue
   Calculator_CellAttribute = Calculator_MemoryAttribute
   Calculator_CellLastValue = Calculator.Value
   Calculator_CellLastAttribute = Calculator_CellAttribute
   Calculator!RefreshTape(Calculator.Value,"<")
   Calculator_CellStatus = Calculator_CALC
   Calculator!SetOpKeys()
   ResyncControl "Cell"
   SelectControl "Cell"
   Return OK
EndProc ; Calculator!GetMem

WRITELIB libname Calculator!GetMem
RELEASE PROCS    Calculator!GetMem

;----------------------------------------------------------------------------

Proc Calculator!Factorial()
   Private
      OK,i
   If Not IsBlank(Calculator.Value) Then
      OK = True
      If Int(Calculator.Value) > 170 Then
         Beep
         Message "Cannot take factorial of > 170."
         Sleep 2000
         Message ""
      Else
         Calculator_LastOperator = Calculator_CurrentOperator
         Calculator_CurrentOperator = "!"
         Calculator!RefreshTape(Calculator.Value,"! =")
         Calculator.Value = Int(Calculator.Value)
         For i From Calculator.Value  To 2 Step - 1
            Calculator.Value = Calculator.Value * (i - 1)
         EndFor
         Calculator!RefreshTape(Calculator.Value,":")
         Calculator_CellLastValue = Calculator.Value
         Calculator_CellLastAttribute = Calculator_CellAttribute
         Calculator_DecimalFlag = False
         Calculator!KeyPadModeLine()
         Calculator_CellStatus = Calculator_CALC
         Calculator!SetOpKeys()
         ResyncControl "Cell"
      Endif
   Else
      OK = false
      Beep
      Message "Cannot take factorial of blank value."
      Sleep 2000
      Message ""
   Endif
   SelectControl "Cell"
   Return OK
EndProc ; Calculator!Factorial

WRITELIB libname Calculator!Factorial
RELEASE PROCS    Calculator!Factorial

;----------------------------------------------------------------------------

Proc Calculator!PlusMinus()
   Private
      OK
   If Not IsBlank(Calculator.Value) Then
      OK = True
      Calculator_LastOperator = Calculator_CurrentOperator
      Calculator_CurrentOperator = "R"
      Calculator!RefreshTape(Calculator.Value,"+/- =")
      Calculator.Value = -Calculator.Value
      Calculator!RefreshTape(Calculator.Value,":")
      Calculator_CellLastValue = Calculator.Value
      Calculator_CellLastAttribute = Calculator_CellAttribute
      Calculator_CellStatus = Calculator_CALC
      Calculator!SetOpKeys()
      ResyncControl "Cell"
   Else
      OK = false
      Beep
      Message "Cannot reverse sign of blank value."
      Sleep 2000
      Message ""
   Endif
   SelectControl "Cell"
   Return OK
EndProc ; Calculator!PlusMinus

WRITELIB libname Calculator!PlusMinus
RELEASE PROCS    Calculator!PlusMinus

;------------------------------ OPERATORS -----------------------------------

Proc Calculator!Exponent()
   Private
      OK
   OK = Calculator!InfixOperator("^")
   Return OK
EndProc ; Calculator!Exponent

WRITELIB libname Calculator!Exponent
RELEASE PROCS    Calculator!Exponent

;------------------------------ MULTIPLY/DIVIDE -----------------------------

Proc Calculator!Multiply()
   Private
      OK
   OK = Calculator!InfixOperator("*")
   Return OK
EndProc ; Calculator!Multiply

WRITELIB libname Calculator!Multiply
RELEASE PROCS    Calculator!Multiply

;----------------------------------------------------------------------------

Proc Calculator!Divide()
   Private
      OK
   OK = Calculator!InfixOperator("/")
   Return OK
EndProc ; Calculator!Divide

WRITELIB libname Calculator!Divide
RELEASE PROCS    Calculator!Divide

;----------------------------------------------------------------------------

Proc Calculator!InfixOperator(Operator)
   Private
      OK
   If isBlank(Calculator.Value) Then
      Beep
      Message "Cannot use [",Operator,"] with blank value."
      Sleep 2000
      Message ""
      OK = false
   Else
      Calculator!PrepStackForOp()
      Calculator.Value = Calculator_StackValue[Calculator_Stackptr]
      Calculator_CellAttribute = Calculator_StackAttribute[Calculator_Stackptr]
      Calculator_StackPtr = Calculator_StackPtr + 1
      Calculator_StackValue[Calculator_StackPtr] = Operator
      Calculator_StackAttribute[Calculator_Stackptr] = Calculator_OPERATOR

      ResyncControl "Cell"
      Calculator_DecimalFlag = False
      Calculator!KeyPadModeLine()
      Calculator_CellStatus = Calculator_WAITING
      Calculator!SetCharKeys()

      OK = True
   Endif
   SelectControl "Cell"
   Return OK
EndProc ; Calculator!InfixOperator

WRITELIB libname Calculator!InfixOperator
RELEASE PROCS    Calculator!InfixOperator

;----------------------------------------------------------------------------

Proc Calculator!PrepStackForOp()
   Calculator_LastOperator = Calculator_CurrentOperator
   Calculator_CurrentOperator = Operator
   If (Calculator_CellStatus = Calculator_WAITING) Then
      Calculator.Value = Calculator_CellLastValue
      Calculator_CellAttribute = Calculator_CellLastAttribute
   Else
      Calculator_CellLastValue = Calculator.Value
      Calculator_CellLastAttribute = Calculator_CellAttribute
   Endif
   Calculator_StackPtr = Calculator_StackPtr + 1
   Calculator_StackValue[Calculator_StackPtr] = Calculator.Value
   Calculator_StackAttribute[Calculator_StackPtr] = Calculator_CellAttribute
   OK = Calculator!DoInfixOperator() ; to resolve existing expressions
   If OK Then
      Calculator!RefreshTape(Calculator.Value,"=")
      Calculator!RefreshTape(Calculator_StackValue[Calculator_StackPtr],
         Operator)
      Calculator_CellLastValue = Calculator_StackValue[Calculator_StackPtr]
      Calculator_CellLastAttribute = Calculator_StackValue[Calculator_StackPtr]
   Else
      Calculator!RefreshTape(Calculator.Value,Operator)
   Endif
Endproc ; Calculator!PrepStackForOp

WRITELIB libname Calculator!PrepStackForOp
RELEASE PROCS    Calculator!PrepStackForOp

;----------------------------------------------------------------------------

Proc Calculator!DoInfixOperator()
   Private
      Operator,
      OK
   OK = false
   ;-------------------------------------------------------------------------
   ; Operation must be binary...
   ;-------------------------------------------------------------------------
   If (Calculator_StackPtr > 2) And
      (Calculator_StackAttribute[Calculator_StackPtr - 2] <>
      Calculator_OPERATOR) Then
      Operator = Calculator_StackValue[Calculator_StackPtr - 1]
      If Operator = "*" or Operator = "/" Or Operator = "^" Then
         OK = True
         ;-------------------------------------------------------------------
         ; Perform the operation. The resultant Stack attribute is not
         ; changed for now.
         ;-------------------------------------------------------------------
         Switch
            Case Operator = "*":
               Calculator_StackValue[Calculator_StackPtr - 2] =
               Calculator_StackValue[Calculator_StackPtr - 2] *
               Calculator_StackValue[Calculator_StackPtr]
            Case Operator = "/":
               Calculator_StackValue[Calculator_StackPtr - 2] =
               Calculator_StackValue[Calculator_StackPtr - 2] /
               Calculator_StackValue[Calculator_StackPtr]
            Case Operator = "^":
               Calculator_StackValue[Calculator_StackPtr - 2] =
               Pow(Calculator_StackValue[Calculator_StackPtr - 2],
                  Calculator_StackValue[Calculator_StackPtr])
               If Calculator_StackValue[Calculator_StackPtr - 2] =
                  "Error" Then
                  Calculator_StackValue[Calculator_StackPtr - 2] = 0
                  Calculator!RefreshTape(BlankNum(),"Error")
               Endif
         EndSwitch
         Calculator_StackPtr = Calculator_StackPtr - 2
      Endif
   Endif
   Return OK
EndProc ; Calculator!DoInfixOperator

WRITELIB libname Calculator!DoInfixOperator
RELEASE PROCS    Calculator!DoInfixOperator

;------------------------------- ADD/SUBTRACT -------------------------------

Proc Calculator!Add()
   Private
      OK
   OK = Calculator!AddSubtract("+")
   Return OK
EndProc ; Calculator!Add

WRITELIB libname Calculator!Add
RELEASE PROCS    Calculator!Add

;----------------------------------------------------------------------------

Proc Calculator!Subtract()
   Private
      OK
   OK = Calculator!AddSubtract("-")
   Return OK
EndProc ; Calculator!Subtract

WRITELIB libname Calculator!Subtract
RELEASE PROCS    Calculator!Subtract

;----------------------------------------------------------------------------

Proc Calculator!AddSubtract(Operator)
   Private
      OK
   If isBlank(Calculator.Value) Then
      Beep
      Message "Cannot use [",Operator,"] with blank value."
      Sleep 2000
      Message ""
      OK = false
   Else
      Calculator!PrepStackForOp()
      Calculator_StackPtr = Calculator_StackPtr + 1
      Calculator_StackValue[Calculator_StackPtr] = Operator
      Calculator_StackAttribute[Calculator_StackPtr] = Calculator_OPERATOR
      Calculator!DoAddSubtract()
      Calculator.Value = Calculator_StackValue[Calculator_Stackptr]
      Calculator_CellAttribute = Calculator_StackAttribute[Calculator_StackPtr]
      ResyncControl "Cell"
      Calculator_DecimalFlag = False
      Calculator!KeyPadModeLine()
      Calculator_CellStatus = Calculator_WAITING
      Calculator!SetCharKeys()

      OK = True
   Endif
   SelectControl "Cell"
   Return OK
EndProc ; Calculator!AddSubtract

WRITELIB libname Calculator!AddSubtract
RELEASE PROCS    Calculator!AddSubtract

;----------------------------------------------------------------------------

Proc Calculator!DoAddSubtract()
   Private
      Operator
   Operator = Calculator_StackValue[Calculator_StackPtr]
   If Operator = "+" or Operator = "-" Then
      If Calculator_StackPtr = 2 Or
         Calculator_StackAttribute[Calculator_StackPtr - 2] =
         Calculator_OPERATOR Then ; This is a unary Plus or minus
         If Operator = "-" Then
            Calculator_StackValue[Calculator_StackPtr - 1] =
            -Calculator_StackValue[Calculator_StackPtr - 1]
         Endif
         Calculator_StackPtr = Calculator_StackPtr - 1
      Else
         If Operator = "+" Then
            Calculator_StackValue[Calculator_StackPtr - 2] =
            Calculator_StackValue[Calculator_StackPtr - 2] +
            Calculator_StackValue[Calculator_Stackptr - 1]
         Else
            Calculator_StackValue[Calculator_StackPtr - 2] =
            Calculator_StackValue[Calculator_StackPtr - 2] -
            Calculator_StackValue[Calculator_Stackptr - 1]
         Endif
         Calculator_StackPtr = Calculator_StackPtr - 2
      Endif
   Endif
EndProc ; Calculator!DoAddSubtract

WRITELIB libname Calculator!DoAddSubtract
RELEASE PROCS    Calculator!DoAddSubtract

;------------------------------ TERMINATORS ---------------------------------

;----------------------------------------------------------------------------
; Equals deterministically terminates the current calculation. If there is a
; blank value in the CellValue, it is set to 0. Outstanding expressions on
; the stack are resolved, and the result of the calculation is saved to
; CellLastValue for the use of the next calculation.
;----------------------------------------------------------------------------

Proc Calculator!Equals()
   Private
      OK
   Calculator_LastOperator = Calculator_CurrentOperator
   Calculator_CurrentOperator = "="
   Calculator!ResolveStack()
   ;-------------------------------------------------------------------------
   ; At this point, the only possible unresolved expressions are subtotal
   ; adds, so we do them now...
   ;-------------------------------------------------------------------------
   Calculator!DoSubTotal()
   Calculator.Value = Calculator_StackValue[Calculator_StackPtr]
   Calculator_CellAttribute = Calculator_StackAttribute[Calculator_StackPtr]
   Calculator!RefreshTape(Calculator.Value,"Total")
   ;-------------------------------------------------------------------------
   ; Remember the result of this calculation to use as the result value of
   ; the next operation if a WAITING cell is operated upon...
   ;-------------------------------------------------------------------------
   Calculator_CellLastValue = Calculator.Value
   Calculator_CellLastAttribute = Calculator_CellAttribute
   Calculator_StackPtr = Calculator_StackPtr - 1
   If Calculator_StackPtr <> 0 Then
      Beep
      Debug ; Stack pointer should be 0 with equals
   Endif

   Calculator_DecimalFlag = False
   Calculator!KeyPadModeLine()
   Calculator_CellStatus = Calculator_WAITING
   Calculator!SetCharKeys()
   ResyncControl "Cell"
   SelectControl "Cell"
   OK = true
   Return OK
EndProc ; Calculator!Equals

WRITELIB libname Calculator!Equals
RELEASE PROCS    Calculator!Equals

;----------------------------------------------------------------------------
; Subtotal resolves outstanding expressions for the subtotal, leaves the sub-
; total on the stack, and tops the stack with an "S" operator, which can only
; be resolved by DoSubTotal() as called by Equals, the ultimate terminator.
;----------------------------------------------------------------------------

Proc Calculator!Subtotal()
   Calculator_LastOperator = Calculator_CurrentOperator
   Calculator_CurrentOperator = "S"
   Calculator!ResolveStack()
   Calculator.Value = Calculator_StackValue[Calculator_StackPtr]
   Calculator_CellAttribute = Calculator_StackAttribute[Calculator_StackPtr]
   Calculator!RefreshTape(Calculator.Value,"Subtotal")
   Calculator_CellLastValue = Calculator.Value
   Calculator_CellLastAttribute = Calculator_CellAttribute
   Calculator_StackPtr = Calculator_StackPtr + 1
   Calculator_StackValue[Calculator_StackPtr] = "S"
   Calculator_StackAttribute[Calculator_StackPtr] = Calculator_OPERATOR

   Calculator_DecimalFlag = False
   Calculator!KeyPadModeLine()
   Calculator_CellStatus = Calculator_WAITING
   Calculator!SetCharKeys()
   ResyncControl "Cell"
   SelectControl "Cell"
   OK = true
   Return OK
EndProc ; Calculator!SubTotal

WRITELIB libname Calculator!SubTotal
RELEASE PROCS    Calculator!SubTotal

;----------------------------------------------------------------------------

Proc Calculator!DoSubTotal()
   Private
      Operator
   While Calculator_StackPtr > 1
      Operator = Calculator_StackValue[Calculator_StackPtr - 1]
      If Operator = "S" Then
         Calculator_StackValue[Calculator_StackPtr - 2] =
         Calculator_StackValue[Calculator_StackPtr - 2] +
         Calculator_StackValue[Calculator_Stackptr]
         Calculator_StackPtr = Calculator_StackPtr - 2
      Else
         Beep
         Debug ; expecting Subtotal operator
      Endif
   EndWhile
EndProc ; Calculator!DoSubTotal

WRITELIB libname Calculator!DoSubTotal
RELEASE PROCS    Calculator!DoSubTotal

;----------------------------------------------------------------------------

Proc Calculator!ResolveStack()
   ;-------------------------------------------------------------------------
   ; Anticipate add operation to resolve unresolved expressions...
   ;-------------------------------------------------------------------------
   If IsBlank(Calculator.Value) Then
      Calculator.Value = 0
   Endif
   ;-------------------------------------------------------------------------
   ; if the current value of CellValue is carried forward from the results
   ; of the previous operation, then neutralize it.
   ;-------------------------------------------------------------------------
   If Calculator_CellStatus = Calculator_WAITING Then
      ;----------------------------------------------------------------------
      ; if the stackpointer is greater than 0 then there has been a previous
      ; operation, so neutralize the carried forward result...
      ;----------------------------------------------------------------------
      If Calculator_Stackptr > 0 Then
         ;-------------------------------------------------------------------
         ; There may be an operator on top of the stack...
         ;-------------------------------------------------------------------
         If (Calculator_StackAttribute[Calculator_StackPtr] =
            Calculator_OPERATOR) Then
            If (Calculator_CurrentOperator = "S") Or
               (Calculator_StackValue[Calculator_StackPtr] <> "S") Then
               Calculator_StackPtr = Calculator_StackPtr + 1
               Calculator_StackValue[Calculator_StackPtr] =
                  Calculator_CellLastValue
               Calculator_StackAttribute[Calculator_StackPtr] =
                  Calculator_CellLastAttribute
               Calculator.Value = Calculator_CellLastValue
               Calculator_CellAttribute = Calculator_CellLastAttribute
            Else ; current operator is "=" and top of stack = "S"
               Calculator_StackPtr = Calculator_StackPtr - 1
            Endif
         Endif
      Else
         Calculator.Value = Calculator_CellLastValue
         Calculator_CellAttribute = Calculator_CellLastAttribute
         Calculator_CellStatus = Calculator_CALC
      Endif
   Endif
   ;-------------------------------------------------------------------------
   ; The only WAITING state that can arrive here is a held over result from
   ; an [Enter] before any operators were used in this calculation. This
   ; causes the result of the previous calculation to be used as the result
   ; of the current calculation.
   ;-------------------------------------------------------------------------
   If Calculator_CellStatus = Calculator_CALC Then
      Calculator!Add()
   Else
      OK = Calculator!DoInfixOperator() ; anything unresolved
      If OK Then
         Calculator!RefreshTape(Calculator.Value,"=")
         Calculator!RefreshTape(Calculator_StackValue[Calculator_StackPtr],"+")
      Endif
      If Calculator_StackPtr > 0 Then
         If Calculator_StackAttribute[Calculator_StackPtr] <>
            Calculator_OPERATOR Then
            Calculator_StackPtr = Calculator_StackPtr + 1
            Calculator_StackValue[Calculator_StackPtr] = "+"
            Calculator_StackAttribute[Calculator_StackPtr] =
               Calculator_OPERATOR
         Endif
      Endif
      Calculator!DoAddSubtract()
   Endif
EndProc ; Calculator!ResolveStack

WRITELIB libname Calculator!ResolveStack
RELEASE PROCS    Calculator!ResolveStack

;--------------------------- NUMERIC KEYPAD ---------------------------------

;----------------------------------------------------------------------------
; It would be better to stuff the keyboard here, as this doesn't cope with ".",
; nor does it recognize the position of the cursor within the input string,
; but keystroke initiation within ShowDialog is not available in Paradox 4
;----------------------------------------------------------------------------

Proc Calculator!NumKey(Numeral)
   Private
      String,
      OriginalValue
   OriginalValue = Calculator.Value
   If Calculator_CellStatus = Calculator_WAITING Then
      Calculator.Value = BlankNum()
      Calculator_CellStatus = Calculator_CALC
      Calculator!SetOpKeys()
   Endif
   String = StrVal(Calculator.Value)
   If Calculator_DecimalFlag then
      OK = Match(String,"..\".\"..")
      If Not OK Then
         String = String + "."
      Endif
   Endif
   Calculator.Value = NumVal(String + Numeral)
   If Calculator.Value = "Error" Then
      Beep
      Message "Error"
      Sleep 2000
      Message ""
      Calculator.Value = OriginalValue
      OK = False
   Else
      OK = True
   Endif
   ResyncControl "Cell"
   SelectControl "Cell"
   Return OK
EndProc ; Calculator!NumKey

WRITELIB libname Calculator!NumKey
RELEASE PROCS    Calculator!NumKey

;----------------------------------------------------------------------------

Proc Calculator!Decimal()
   If Calculator_DecimalFlag Then
      Calculator_DecimalFlag = False
   Else
      Calculator_DecimalFlag = True
   Endif
   Calculator!KeyPadModeLine()
   RepaintDialog ; to repaint the keypad mode line
   SelectControl "Cell"
   Return true
EndProc ; Calculator!Decimal

WRITELIB libname Calculator!Decimal
RELEASE PROCS    Calculator!Decimal

;----------------------------------------------------------------------------
;                             PAPER TAPE
;----------------------------------------------------------------------------

Proc Calculator!RefreshTape(Amount,OperationName)
   Private
      TapeString,
      i
   TapeString = Format("W15.2,EC",Amount) + " " + FORMAT("w6",OperationName)
   For i From 2 to 5
      Calculator_Tape[i-1] = Calculator_Tape[i]
   EndFor
   Calculator_Tape[5] = "      " + TapeString

   ; I'm adding this as a new element in the DynArray instead of a File

   If Calculator_TapeIsOn Then

      Calculator_TapeList[Calculator_TapeCtr] =
                FORMAT("w3,AR",Calculator_TapeCtr) + TapeString

      Calculator_TapeCtr = Calculator_TapeCtr + 1

   Endif
   RepaintDialog
EndProc ; Calculator!RefreshTape

WRITELIB libname Calculator!RefreshTape
RELEASE PROCS    Calculator!RefreshTape

