; 
; Program Name - IPRS
;  Script Name - Utility4.sc
; Library Name - Global.lib
;
; This script contains processes that are used by the Built-in Calendar:
;
;        CalendarShell      - Main shell for the Built-in Calendar
;
;        Calendar.Constructor
;        Calendar.Destructor
;        Calendar.Dialog
;        Calendar!GeneratePickList
;        Calendar!CatchEvents
;        Calendar!MovetoToday
;        Calendar!ChangeMonth
;        Calendar!CatchMonthEvents
;        Calendar!ChangeYear
;        Calendar!CatchYearEvents
;        Calendar!ResyncDatePick
;        Calendar!GetDate
;        Calendar!GetHighDay
; 

; libname = "Libs\\global"
libname = "Calc"

; 
; CalendarShell
;
; - Main shell for the Built-in Calendar
;
;        Author - Steven Green
; Creation Date - 11/03/93
; Last Modified - 04/04/94
;   Modified by - Steven Green
;  Changes Made - add trap for Dialog ACCEPT
;
; This process calls the Built-in Calendar and, optionally, places the value
; returned by that process into the current field
;
; EXTERNAL VARIABLES MODIFIED
;        pass.calendar.var - flag variable in DialogProc
; 

PROC CalendarShell()
     PRIVATE Calendar.Date,
             wasvar

Calendar.Constructor()

Calendar.DisplayTime = False

IF Calendar.Dialog() THEN

   IF ((SYSMODE() = "Edit") OR (SYSMODE() = "CoEdit")) AND
      (IMAGETYPE() = "Display") THEN

      SWITCH

            ; Value is going back to a Dialog ACCEPT that uses a date

            CASE ISASSIGNED(pass.calendar.var) :

                 pass.calendar.var = Calendar.Date

            ; Value is going to a date field

            CASE FIELDTYPE() = "D" :

                 ; Save the original value from the field.

                 wasval = []

                 IF ISFIELDVIEW() THEN
                    ENTER
                 ENDIF

                 CTRLBACKSPACE

                 ; Put the new value in the field.

                 [] = Calendar.Date

                 ; Revert to the original value
                 ; if the new one is unacceptable.

                 IF NOT ISVALID() THEN
                    Dia_OK(TRUE,"Error","Invalid Value for this Field","")
                    [] = wasval
                 ENDIF

            ; Current field is an Alphanumeric field

            CASE (SUBSTR(FIELDTYPE(),1,1) = "A") :

                 ; Save the original value from the field.

                 wasval = []

                 IF ISFIELDVIEW() THEN
                    ENTER
                 ENDIF

                 ; Put the new value in the field.

                 [] = wasval + STRVAL(Calendar.Date)

                 ; Revert to the original value
                 ; if the new one is unacceptable.

                 IF NOT ISVALID() THEN
                    Dia_OK(TRUE,"Error","Invalid Value for this Field","")
                    [] = wasval
                 ENDIF


            ; Current field is a "live" Memo field

            CASE ISFIELDVIEW() AND (SUBSTR(FIELDTYPE(),1,1) = "M") :

                 EDITOR INSERT Calendar.Date

            ; Obviously, we are ignoring the various Number-type fields

      ENDSWITCH

   ELSE

      ; Value is going back to a Dialog ACCEPT that uses a date

      IF ISASSIGNED(pass.calendar.var) THEN
         pass.calendar.var = Calendar.Date
      ENDIF

   ENDIF

ENDIF

Calendar.Destructor()

ENDPROC

WRITELIB libname CalendarShell
RELEASE PROCS    CalendarShell

;==============================================================================
; (c) Copyright Elect Software International Inc., 1992, Toronto. Anyone can
; use this code for anything as long as it is not resold as a software
; development resource, as long as the copyright notice isn't removed, as
; long as changes are clearly marked as to authorship, and as long as users
; indemnify Elect from any liability.
; Comments welcome. Henrik Bechmann, CIS:72701,3717; Tel:416-534-8176.
;==============================================================================

;==============================================================================
;                             CALENDAR INTERFACE
;==============================================================================
; DESCRIPTION:
; The Calendar allows the user to select a day of the month, a month of the
; year, or a year from 100 to 9999 A.D. If the user selects a date from the
; calendar, then the Calendar.Dialog() returns true, otherwise it returns
; false. If Calendar.Dialog returns true, then pick up the date that the user
; selected from the property Calendar.Date. To override the default date
; (today()) set the property Calendar.Date *before* calling Calendar.Dialog().
;
; METHODS:
; Calendar.Constructor() ; called at start of session in which calendar will be
;                        ; used.
; Calendar.Destructor()  ; called after session in which calendar is used, to
;                        ; clean up.
; Calendar.Dialog()      ; Displays the calendar to the user.
;                        ; Returns true if a date was selected, otherwise
;                        ; returns false. If it returns true, then get the
;                        ; date selected from the property Calendar.Date.
;
; PROPERTIES: (alter these after constructor call and before dialog call)...
; Calendar.Date          ; the date. Overrides default if set before
;                        ; Calendar.Dialog(), and contains the selected date
;                        ; if Calendar.Dialog() returns true.
; Calendar.DisplayTime   ; If true, displays time (with annoying cursor
;                        ; flicker if the cursor is on the PickArray),
;                        ; otherwise doesn't display time
; Calendar.ColorBag[]    ; assign colors by Window SetColors convention
; Calendar.FrameShadow   ; The color for the dark sides of the raised frames
; Calendar.FrameHighlight; The color for the light sides of the raised frames
; Calendar.WindowBag[]   ; assign attributes by Window SetAttributes
;                        ; convention
;
; Analagous properties for the sub-dialogs...
; Calendar.YearColorBag[]
; Calendar.YearFrameShadow
; Calendar.YearFrameHighlight
; Calendar.YearWindowBag[]
; Calendar.MonthColorBag[]
; Calendar.MonthFrameShadow
; Calendar.MonthFrameHighlight
; Calendar.MonthWindowBag[]
;
; Calendar.IsActive ; (readonly) indicates if Calendar.Constructor has
;                   ; been run
;==============================================================================
;                          CALENDAR IMPLEMENTATION
;==============================================================================

Proc Calendar.Constructor()
   Private
      i,j,k

   Message "Initializing calendar"

   Dynarray Calendar_DOWNum[]
   Calendar_DOWNum["Sun"] = 1
   Calendar_DOWNum["Mon"] = 2
   Calendar_DOWNum["Tue"] = 3
   Calendar_DOWNum["Wed"] = 4
   Calendar_DOWNum["Thu"] = 5
   Calendar_DOWNum["Fri"] = 6
   Calendar_DOWNum["Sat"] = 7

   Array Calendar_MonthName[12]
   Calendar_MonthName[1]  = "Jan"
   Calendar_MonthName[2]  = "Feb"
   Calendar_MonthName[3]  = "Mar"
   Calendar_MonthName[4]  = "Apr"
   Calendar_MonthName[5]  = "May"
   Calendar_MonthName[6]  = "Jun"
   Calendar_MonthName[7]  = "Jul"
   Calendar_MonthName[8]  = "Aug"
   Calendar_MonthName[9]  = "Sep"
   Calendar_MonthName[10] = "Oct"
   Calendar_MonthName[11] = "Nov"
   Calendar_MonthName[12] = "Dec"

   Array Calendar_MonthFullName[12]
   Calendar_MonthFullName[1]  = "January"
   Calendar_MonthFullName[2]  = "February"
   Calendar_MonthFullName[3]  = "March"
   Calendar_MonthFullName[4]  = "April"
   Calendar_MonthFullName[5]  = "May"
   Calendar_MonthFullName[6]  = "June"
   Calendar_MonthFullName[7]  = "July"
   Calendar_MonthFullName[8]  = "August"
   Calendar_MonthFullName[9]  = "September"
   Calendar_MonthFullName[10] = "October"
   Calendar_MonthFullName[11] = "November"
   Calendar_MonthFullName[12] = "December"

   ;---------------------------------------------------------------------------
   ; A map is set up as follows: bracketed numbers ([]) represent
   ; Calendar_Picklist cells, unbracketed numbers represent
   ; Calendar_MapToPickList cells. Calendar_MapToPickList cells contain
   ; pointers to the Calendar_PickList cells. The current calendar month is
   ; mapped onto the Calendar_MapToPickList array with an offset, being the
   ; number of cells of the month that are blank. The first cell
   ; of the Calendar_MapToPickList array that points to a day value in picklist
   ; is the first Calendar_DOWNum of the month.
   ;
   ;  1[ 1]   2[ 7]   3[13]   4[19]   5[25]   6[31]   7[37]
   ;  8[ 2]   9[ 8]  10[14]  11[20]  12[26]  13[32]  14[38]
   ; 15[ 3]  16[ 9]  17[15]  18[21]  19[27]  20[33]  21[39]
   ; 22[ 4]  23[10]  24[16]  25[22]  26[28]  27[34]  28[40]
   ; 29[ 5]  30[11]  31[17]  32[23]  33[29]  34[35]  35[41]
   ; 36[ 6]  37[12]  38[18]  39[24]  40[30]  41[36]  42[42]
   ;                   Ŀ
   ; For June of 1992:         Calendar_Offset: 1Ŀ                     
   ; PickList[1]  =            MapToPickList[1]  = 1      MapToMap[1]  = 1  
   ; PickList[2]  = 7   MapToPickList[2]  = 7  MapToMap[2]  = 8
   ; PickList[3]  = 14     MapToPickList[3]  = 13     MapToMap[3]  = 15
   ; PickList[4]  = 21     MapToPickList[4]  = 19     MapToMap[4]  = 22
   ; PickList[5]  = 28     MapToPickList[5]  = 25     MapToMap[5]  = 29
   ; PickList[6]  =         MapToPickList[6]  = 31     MapToMap[6]  = 36
   ; PickList[7]  = 1  ٳ MapToPickList[7]  = 37     MapToMap[7]  = 2
   ; PickList[8]  = 8        MapToPickList[8]  = 2 MapToMap[8]  = 9
   ; PickList[9]  = 15       MapToPickList[9]  = 8       MapToMap[9]  = 16
   ; PickList[10] = 22       MapToPickList[10] = 14      MapToMap[10] = 23
   ; PickList[11] = 29       MapToPickList[11] = 20      MapToMap[11] = 30
   ; PickList[12] =          MapToPickList[12] = 26      MapToMap[12] = 37
   ; PickList[13] = 2 ٳ MapToPickList[13] = 32      MapToMap[13] = 3
   ; PickList[14] = 9         MapToPickList[14] = 38      MapToMap[14] = 10
   ; PickList[15] = 16        MapToPickList[15] = 3       MapToMap[15] = 17
   ; PickList[16] = 23        MapToPickList[16] = 9       MapToMap[16] = 24
   ; PickList[17] = 30        MapToPickList[17] = 15      MapToMap[17] = 31
   ; PickList[18] =           MapToPickList[18] = 21      MapToMap[18] = 38
   ; PickList[19] = 3ٳ MapToPickList[19] = 27      MapToMap[19] = 4
   ; PickList[20] = 10         MapToPickList[20] = 33      MapToMap[20] = 11
   ; PickList[21] = 17         MapToPickList[21] = 39      MapToMap[21] = 18
   ; PickList[22] = 24         MapToPickList[22] = 4       MapToMap[22] = 25
   ; PickList[23] =            MapToPickList[23] = 10      MapToMap[23] = 32
   ; PickList[24] =            MapToPickList[24] = 16      MapToMap[24] = 39
   ; PickList[25] = 4 MapToPickList[25] = 22      MapToMap[25] = 5
   ; PickList[26] = 11          MapToPickList[26] = 28      MapToMap[26] = 12
   ; PickList[27] = 18          MapToPickList[27] = 34      MapToMap[27] = 19
   ; PickList[28] = 25          MapToPickList[28] = 40      MapToMap[28] = 26
   ; PickList[29] =             MapToPickList[29] = 5       MapToMap[29] = 33
   ; PickList[30] =             MapToPickList[30] = 11      MapToMap[30] = 40
   ; PickList[31] = 5           MapToPickList[31] = 17      MapToMap[31] = 6
   ; PickList[32] = 12          MapToPickList[32] = 23      MapToMap[32] = 13
   ; PickList[33] = 19          MapToPickList[33] = 29      MapToMap[33] = 20
   ; PickList[34] = 26          MapToPickList[34] = 35      MapToMap[34] = 27
   ; PickList[35] =             MapToPickList[35] = 41      MapToMap[35] = 34
   ; PickList[36] =             MapToPickList[36] = 6       MapToMap[36] = 42
   ; PickList[37] = 6           MapToPickList[37] = 12      MapToMap[37] = 7
   ; PickList[38] = 13          MapToPickList[38] = 18      MapToMap[38] = 14
   ; PickList[39] = 20          MapToPickList[39] = 24      MapToMap[39] = 21
   ; PickList[40] = 27          MapToPickList[40] = 30      MapToMap[40] = 28
   ; PickList[41] =             MapToPickList[41] = 36      MapToMap[41] = 35
   ; PickList[42] =             MapToPickList[42] = 42      MapToMap[42] = 42
   ;
   ; Therefore:
   ; - To get the picklist number for the current day of the month, choose the
   ; pointer in Calendar_MapToPickList with an index of the day of the month
   ; plus the month offset.
   ; - To get the date value from the Calendar_Picklist index, simply choose
   ; the value pointed to in Calendar_PickList by the index.
   ;---------------------------------------------------------------------------
   Array Calendar_MapToPickList[42]
   k = 1
   For i From 0 to 6
      For j From  1 to 42 step 7
         Calendar_MapToPickList[i + j] = k
         k = k + 1
      EndFor
   EndFor
   ;---------------------------------------------------------------------------
   ; Generate an array that allows easy lookup of the implied index of
   ; Calendar_MapToPickList given an index of Calendar_PickList. This
   ; allows resynchronization from a pointer into Picklist that points to a
   ; blank, to a pointer that points to a date within range. (See
   ; Calendar!ResyncDatePick())
   ;---------------------------------------------------------------------------
   Array Calendar_MapToMap[42]
   For i From 1 to 42
      Calendar_MapToMap[Calendar_MapToPickList[i]] = i
   EndFor
   ;---------------------------------------------------------------------------
   ; Set defaults for properties...                                Colors by SG
   ;---------------------------------------------------------------------------
   Calendar.FrameHighlight = 127
   Calendar.FrameShadow = 112
   Dynarray Calendar.ColorBag[]
   Calendar.ColorBag[1]  = Calendar.FrameShadow ; active frame
   Calendar.ColorBag[2]  = 126                  ; dragging frame
   Calendar.ColorBag[5]  = 112      ; text
   Calendar.ColorBag[9]  = 32       ; push button label
   Calendar.ColorBag[10] = 47       ; default push button label
   Calendar.ColorBag[11] = 47       ; default push button label
   Calendar.ColorBag[13] = 46       ; push button hot key
   Calendar.ColorBag[14] = 112      ; pushbutton shadow
   Calendar.ColorBag[26] = 79       ; active picklist selection
   Calendar.ColorBag[27] = 79       ; inactive picklist selection
   Dynarray Calendar.WindowBag[]
   ;---------------------------------------------------------------------------
   ; The Calendar is initialized off screen (@500,500) to hide the
   ; initialization flicker from the user. OriginRow and Col are specified
   ; here to tell the calendar where to move to after the initialization is
   ; complete.
   ;---------------------------------------------------------------------------
   Calendar.WindowBag["OriginRow"] = 3
   Calendar.WindowBag["OriginCol"] = 5
   Calendar_WindowHandle = BlankNum()

   Calendar.YearFrameHighlight = 127
   Calendar.YearFrameShadow  = 112
   Dynarray Calendar.YearColorBag[]
   Calendar.YearColorBag[1]  = Calendar.YearFrameShadow ; active frame
   Calendar.YearColorBag[2]  = 126                      ; dragging frame
   Calendar.YearColorBag[6]  = 112      ; inactive label
   Calendar.YearColorBag[7]  = 112      ; active label
   Calendar.YearColorBag[9]  = 32       ; push button label
   Calendar.YearColorBag[10] = 47       ; default push button label
   Calendar.YearColorBag[11] = 47       ; default push button label
   Calendar.YearColorBag[13] = 46       ; push button hot key
   Calendar.YearColorBag[18] = 63       ; normal typein box
   Calendar.YearColorBag[19] = 48       ; selected typein box
   Calendar.YearColorBag[26] = 127      ; active picklist selection
   Dynarray Calendar.YearWindowBag[]
   Calendar.YearWindowBag["OriginRow"] = 8
   Calendar.YearWindowBag["OriginCol"] = 30
   Calendar_YearWindowHandle = BlankNum()

   Calendar.MonthFrameHighlight = 127
   Calendar.MonthFrameShadow = 112
   Dynarray Calendar.MonthColorBag[]
   Calendar.MonthColorBag[1]  = Calendar.MonthFrameShadow ; active frame
   Calendar.MonthColorBag[2]  = 126                       ; dragging frame
   Calendar.MonthColorBag[9]  = 32       ; push button label
   Calendar.MonthColorBag[10] = 47       ; default push button label
   Calendar.MonthColorBag[11] = 47       ; default push button label
   Calendar.MonthColorBag[13] = 46       ; push button hot key
   Calendar.MonthColorBag[25] = 112      ; Normal pick list item text
   Calendar.MonthColorBag[26] = 127      ; active picklist selection
   Calendar.MonthColorBag[27] = 126      ; inactive picklist selection
   Calendar.MonthColorBag[28] = 112      ; Column dividers
   Dynarray Calendar.MonthWindowBag[]
   Calendar.MonthWindowBag["OriginRow"] = 8
   Calendar.MonthWindowBag["OriginCol"] = 30
   Calendar_MonthWindowHandle = BlankNum()

   Calendar.IsActive = True
   Calendar.Date = today()
   Calendar.DisplayTime = False                                      ; SG
   Message "" ; clear message
EndProc ; Calendar.Constructor

WRITELIB libname Calendar.Constructor
RELEASE PROCS    Calendar.Constructor

;----------------------------------------------------------------------------

Proc Calendar.Destructor()
   Release Vars
      Calendar_DOWNum,
      Calendar_MonthName,
      Calendar_MonthFullName,
      Calendar_MapToPickList,
      Calendar_MapToMap,
      Calendar.FrameHighlight,
      Calendar.FrameShadow,
      Calendar.ColorBag,
      Calendar.WindowBag,
      Calendar_WindowHandle,
      Calendar.YearFrameHighlight,
      Calendar.YearFrameShadow,
      Calendar.YearColorBag,
      Calendar.YearWindowBag,
      Calendar_YearWindowHandle,
      Calendar.MonthFrameHighlight,
      Calendar.MonthFrameShadow,
      Calendar.MonthColorBag,
      Calendar.MonthWindowBag,
      Calendar_MonthWindowHandle,
      Calendar.IsActive,
      Calendar.Date,
      Calendar.DisplayTime
EndProc ; Calendar.Destructor

WRITELIB libname Calendar.Destructor
RELEASE PROCS    Calendar.Destructor

;----------------------------------------------------------------------------

Proc Calendar.Dialog()
   Private
      Calendar_Day,
      Calendar_Month,
      Calendar_Year,
      Calendar_PickList,
      Calendar_Offset,
      Calendar_HighDay,
      Calendar_Prompt,
      Calendar_TimeVar,
      DatePick,
      CalendarVar,
      ControlVar,
      i,
      OK
   Calendar_TimeVar = ""
   If IsBlank(Calendar.Date) Then
      Calendar.Date = today()
   Endif
   Calendar_Day = Day(Calendar.Date)
   Calendar_Month = Month(Calendar.Date)
   Calendar_Year = Year(Calendar.Date)
   Calendar!GeneratePickList()
   ShowDialog "ESI PD40B Calendar"
      Proc "Calendar!CatchEvents"
         Trigger "Open"
      @500,500 Height 19 Width 38
      ;------------------------------------------------------------------------
      ; Draw raised frames...
      ;------------------------------------------------------------------------
      Frame Single ; titles and header
         From 0,0 To 2,35
      Frame Single ; the calendar
         From 3,0 To 12,35
      PaintCanvas
         Attribute Calendar.FrameShadow
         0,0,16,35
      Style Attribute Calendar.ColorBag[5]
      @1,2 ?? Format("W33,AL",Calendar_Prompt)
      @1,26 ?? IIf(Calendar.DisplayTime,Time(),"")
      Style Attribute Calendar.FrameShadow
      @4,1 ?? "Sun Mon Tue Wed Thu Fri Sat "
      @5,1 ?? ""
      Style
      ;------------------------------------------------------------------------
      ; Highlight the titles and header frame...
      ;------------------------------------------------------------------------
      PaintCanvas
         Attribute Calendar.FrameHighlight
         0,0,0,34
      PaintCanvas
         Attribute Calendar.FrameHighlight
         0,0,2,0
      ;------------------------------------------------------------------------
      ; Highlight the calendar frame...
      ;------------------------------------------------------------------------
      PaintCanvas
         Attribute Calendar.FrameHighlight
         3,0,3,34
      PaintCanvas
         Attribute Calendar.FrameHighlight
         3,0,12,0
      ;------------------------------------------------------------------------
      ; Draw the dialog box elements...
      ;------------------------------------------------------------------------
      PickArray
         @6,1 Height 6 Width 34
         Columns 7
         Calendar_PickList
         Tag "Pick"
         To DatePick
      PushButton
         @13,1 Width 10
         "~M~onth"
         Value Calendar!ChangeMonth()
         Tag "Month"
         To CalendarVar
      PushButton
         @13,13 Width 10
         "~Y~ear"
         Value Calendar!ChangeYear()
         Tag "Year"
         To CalendarVar
      PushButton
         @13,25 Width 10
         "~T~oday"
         Value Calendar!MovetoToday()
         Tag "Today"
         To CalendarVar
      PushButton
         @15,8 Width 10
         "~O~K"
         OK
;        Default
         Value "OK"
         Tag "OK"
         To ControlVar
      PushButton
         @15,20 Width 10
         "~C~ancel"
         Cancel
         Default                       ; SG
         Value "Cancel"
         Tag "Cancel"
         To ControlVar
   EndDialog
   OK = Retval
   If OK Then
      Calendar.Date = Calendar!GetDate(Calendar_PickList[DatePick],
               Calendar_Month,Calendar_Year)
   Endif
   Return OK
EndProc ; Calendar.Dialog

WRITELIB libname Calendar.Dialog
RELEASE PROCS    Calendar.Dialog

;----------------------------------------------------------------------------
; Calendar!GeneratePickList() expects three global variables to be set:
; Calendar_Day, Calendar_Month, and Calendar_Year. If Calendar_Day is greater
; than the last day of the current month, it will be set to the highest day of
; the month.
; In any case DatePick is set to the index value of Calendar_Picklist that
; points to Calendar_Day
;----------------------------------------------------------------------------

Proc Calendar!GeneratePickList()
   Array Calendar_PickList[42]
   Calendar_HighDay = Calendar!GetHighDay(Calendar_Month,Calendar_Year)
   If Calendar_Day > Calendar_HighDay Then
      Calendar_Day = Calendar_HighDay
   Endif
   Calendar_Offset =
      Calendar_DOWNum[
         DOW(Calendar!GetDate(
            1,Calendar_Month,Calendar_Year))] - 1
   For i From 1 To Calendar_HighDay
      Calendar_PickList[Calendar_MapToPickList[i + Calendar_Offset]] = i
   EndFor
   For i From 1 To Calendar_Offset
      Calendar_PickList[Calendar_MapToPickList[i]] = BlankNum()
   EndFor
   For i From Calendar_HighDay + Calendar_Offset + 1 To 42
      Calendar_PickList[Calendar_MapToPickList[i]] = BlankNum()
   EndFor
   DatePick = Calendar_MapToPickList[Calendar_Day + Calendar_Offset]
   Calendar_Prompt = Calendar_MonthFullName[Calendar_Month] + ", " +
      StrVal(Calendar_Year)
EndProc ; Calendar!GeneratePickList

WRITELIB libname Calendar!GeneratePickList
RELEASE PROCS    Calendar!GeneratePickList

;----------------------------------------------------------------------------

Proc Calendar!CatchEvents(EventType,TagValue,EventValue,ElementValue)
   Private
      Key,
      OK
   Switch
      Case EventType = "ACCEPT":
         If Not IsBlank(Calendar_PickList[DatePick]) Then
            OK = True
         Else
            Beep
            OK = false
            SelectControl "Pick"
         Endif
      Case EventType = "EVENT":
         Switch
            Case EventValue["Type"] = "IDLE":
               If Calendar_TimeVar <> Time() Then
                  Calendar_TimeVar = Time()
                  SetCanvas Calendar_WindowHandle
                  @1,26
                  Style Attribute Calendar.ColorBag[5]
                  ?? Calendar_TimeVar
                  Style
               Endif
               OK = True
            Case EventValue["Type"] = "KEY":
               Key = EventValue["KeyCode"]
               Switch
                  Case Key = Asc("PgUp"):
                     If (Calendar_Month = 1) Then
                        If (Calendar_Year > 100) Then
                           Calendar_Month = 12
                           Calendar_Year = Calendar_Year - 1
                        Endif
                     Else
                        Calendar_Month = Calendar_Month - 1
                     Endif
                     Calendar!ResyncDatePick()
                     Calendar!GeneratePickList()
                  Case Key = Asc("PgDn"):
                     If (Calendar_Month = 12) Then
                        If (Calendar_Year < 9999) Then
                           Calendar_Month = 1
                           Calendar_Year = Calendar_Year + 1
                        Endif
                     Else
                        Calendar_Month = Calendar_Month + 1
                     Endif
                     Calendar!ResyncDatePick()
                     Calendar!GeneratePickList()
                  Case Key = Asc("CtrlPgUp"):
                     If (Calendar_Year > 100) Then
                        Calendar_Year = Calendar_Year - 1
                        Calendar!ResyncDatePick()
                        Calendar!GeneratePickList()
                     Else
                        Beep
                     Endif
                  Case Key = Asc("CtrlPgDn"):
                     If (Calendar_Year < 9999) Then
                        Calendar_Year = Calendar_Year + 1
                        Calendar!ResyncDatePick()
                        Calendar!GeneratePickList()
                     Else
                        Beep
                     Endif
                  Case Key = Asc("Home"):
                     DatePick = Calendar_MapToPickList[Calendar_Offset + 1]
                  Case Key = Asc("End"):
                     DatePick = Calendar_MapToPickList[Calendar_Offset + Calendar_HighDay]
               EndSwitch
               RefreshControl "Pick"
               OK = False
         EndSwitch
      Case EventType = "OPEN":
         If Calendar.DisplayTime Then
            NewDialogSpec
               Key "PgUp","PgDn","CtrlPgUp","CtrlPgDn","Home","End"
               Idle
               Trigger "Accept"
         Else
            NewDialogSpec
               Key "PgUp","PgDn","CtrlPgUp","CtrlPgDn","Home","End"
               Trigger "Accept"
         Endif
         Window Handle Dialog to Calendar_WindowHandle
         Window SetColors Calendar_WindowHandle From Calendar.ColorBag
         Window GetColors Calendar_WindowHandle To Calendar.ColorBag
         Window SetAttributes Calendar_WindowHandle From Calendar.WindowBag
         OK = True
   EndSwitch
   Return OK
EndProc ; Calendar!CatchEvents

WRITELIB libname Calendar!CatchEvents
RELEASE PROCS    Calendar!CatchEvents

;----------------------------------------------------------------------------

Proc Calendar!MovetoToday()
   Private
      Date
   Date = Today()
   Calendar_Day = Day(Date)
   Calendar_Month = Month(Date)
   Calendar_Year = Year(Date)
   Calendar!GeneratePickList()
   SelectControl "Pick"
   RefreshControl "Pick"
   Return True
EndProc ; Calendar!MovetoToday

WRITELIB libname Calendar!MovetoToday
RELEASE PROCS    Calendar!MovetoToday

;----------------------------------------------------------------------------

Proc Calendar!ChangeMonth()
   Private
      MonthPick,
      ControlVar
   MonthPick = Calendar_Month
   ShowDialog "Select month"
      Proc "Calendar!CatchMonthEvents"
         Trigger "Open"
      @500,500 Height 9 Width 28
      Frame Single
         From 0,0 To 4,25
      PaintCanvas
         Attribute Calendar.MonthFrameShadow
         0,0,4,25
      PaintCanvas
         Attribute Calendar.MonthFrameHighlight
         0,0,4,0
      PaintCanvas
         Attribute Calendar.MonthFrameHighlight
         0,0,0,24
      PickArray
         @1,1 Height 3 Width 23
         Columns 4
         Calendar_MonthName
         Tag "Pick"
         To MonthPick
      PushButton
         @5,2 Width 10
         "~O~K"
         OK
         Default
         Value "OK"
         Tag "OK"
         To ControlVar
      PushButton
         @5,14 Width 10
         "~C~ancel"
         Cancel
         Value "Cancel"
         Tag "Cancel"
         To ControlVar
   EndDialog
   If retval Then
      Calendar_Month = MonthPick
      Calendar!ResyncDatePick()
      Calendar!GeneratePickList()
      RefreshControl "Pick"
   Endif
   SelectControl "Pick"
   Return True
EndProc ; Calendar!ChangeMonth

WRITELIB libname Calendar!ChangeMonth
RELEASE PROCS    Calendar!ChangeMonth

;----------------------------------------------------------------------------

Proc Calendar!CatchMonthEvents(EventType,TagValue,EventValue,ElementValue)
   Private
      OK
   Switch
      Case EventType = "OPEN":
         Window Handle Dialog to Calendar_MonthWindowHandle
         Window SetColors Calendar_MonthWindowHandle From Calendar.MonthColorBag
         Window GetColors Calendar_MonthWindowHandle To Calendar.MonthColorBag
         Window SetAttributes Calendar_MonthWindowHandle From
            Calendar.MonthWindowBag
         OK = True
   EndSwitch
EndProc ; Calendar!CatchMonthEvents

WRITELIB libname Calendar!CatchMonthEvents
RELEASE PROCS    Calendar!CatchMonthEvents

;----------------------------------------------------------------------------

Proc Calendar!ChangeYear()
   Private
      ControlVar,
      InputYear
   RefreshControl "Pick"
   InputYear = Calendar_Year
   ShowDialog "Select year"
      Proc "Calendar!CatchYearEvents"
         Key "PgUp","PgDn"
         Trigger "Open"
      @500,500 Height 7 Width 27
      Frame Single
         From 0,0 To 2,24
      PaintCanvas
         Attribute Calendar.YearFrameShadow
         0,0,2,24
      PaintCanvas
         Attribute Calendar.YearFrameHighlight
         0,0,0,23
      PaintCanvas
         Attribute Calendar.YearFrameHighlight
         0,0,2,0
      Label @1,1 "~E~nter year: " For "Accept"
      Accept
         @1,13 Width 7
         "S" Min 100 Max 9999 Required
         Tag "Accept"
         To InputYear
      PushButton
         @3,2 Width 10
         "~O~K"
         OK
         Default
         Value "OK"
         Tag "OK"
         To ControlVar
      PushButton
         @3,14 Width 10
         "~C~ancel"
         Cancel
         Value "Cancel"
         Tag "Cancel"
         To ControlVar
   EndDialog
   If RetVal Then
      Calendar_Year = InputYear
      Calendar!ResyncDatePick()
      Calendar!GeneratePickList()
      RefreshControl "Pick"
   Endif
   SelectControl "Pick"
   Return True
EndProc ; Calendar!ChangeYear

WRITELIB libname Calendar!ChangeYear
RELEASE PROCS    Calendar!ChangeYear

;----------------------------------------------------------------------------

Proc Calendar!CatchYearEvents(EventType,TagValue,EventValue,ElementValue)
   Private
      OK
   Switch
      Case EventType = "EVENT":
         InputYear = ControlValue("Accept")
         Switch
            Case EventValue["Keycode"] = Asc("PgDn"):
               If InputYear < 9999 Then
                  InputYear = InputYear + 1
               Endif
            Case EventValue["Keycode"] = Asc("PgUp"):
               If InputYear > 100 Then
                  InputYear = InputYear - 1
               Endif
         EndSwitch
         ResyncControl "Accept"
         OK = False
      Case EventType = "OPEN":
         Window Handle Dialog to Calendar_YearWindowHandle
         Window SetColors Calendar_YearWindowHandle From Calendar.YearColorBag
         Window GetColors Calendar_YearWindowHandle To Calendar.YearColorBag
         Window SetAttributes Calendar_YearWindowHandle From
            Calendar.YearWindowBag
         OK = True
   EndSwitch
EndProc ; Calendar!CatchYearEvents

WRITELIB libname Calendar!CatchYearEvents
RELEASE PROCS    Calendar!CatchYearEvents

;----------------------------------------------------------------------------

Proc Calendar!ResyncDatePick()
   Private
      MapPointer
   If IsBlank(Calendar_PickList[DatePick]) Then
      MapPointer = Calendar_MapToMap[DatePick]
      If MapPointer > 15 Then
         DatePick = Calendar_MapToPickList[Calendar_Offset + Calendar_HighDay]
         Calendar_Day = Calendar_HighDay
      Else
         DatePick = Calendar_MapToPickList[Calendar_Offset + 1]
         Calendar_Day = 1
      Endif
   Else
      Calendar_Day = Calendar_PickList[DatePick]
   Endif
EndProc ; Calendar!ResyncDatePick

WRITELIB libname Calendar!ResyncDatePick
RELEASE PROCS    Calendar!ResyncDatePick

;----------------------------------------------------------------------------

Proc Calendar!GetDate(DayOfMonth,MonthOfYear,CalendarYear)
   Return DateVal(StrVal(MonthOfYear) + "/" +
                  StrVal(DayOfMonth) + "/" +
                  StrVal(CalendarYear))
EndProc ; Calendar!GetDate

WRITELIB libname Calendar!GetDate
RELEASE PROCS    Calendar!GetDate

;----------------------------------------------------------------------------

Proc Calendar!GetHighDay(MonthOfYear,CalendarYear)
   Private
      MonthOfYear
   If MonthOfYear = 12 Then
      Return 31 ; December
   Else
      MonthOfYear = MonthOfYear + 1
      Return Day(Calendar!GetDate(1,MonthOfYear,CalendarYear) - 1)
   Endif
EndProc ; Calendar!GetHighDay

WRITELIB libname Calendar!GetHighDay
RELEASE PROCS    Calendar!GetHighDay

RELEASE VARS libname

