program Yet_Another_Clock;

  (* YACLOCK.PAS - Yet Another Clock
   * By Richard R. Sands CIS 70274,103
   *
   * This clock is a "liberal" clock.  Since it seems that is is impossible
   * to keep all the clocks you have around syncronized, I decided that a
   * vague clock is as good as any other.  This clock will make sure you will
   * never be (too) late for a meeting!
   *
   * For you really up-tight folks, there is a "Real" clock option to
   * display day/date/time in "normal" format.
   *
   * This program places a [LibClock] entry into your WIN.INI file.  The
   * keywords it uses are:
   *      iFormat=    A value of 1 is liberal, a value of 0 is "real".
   *      iPosX  =    X coord of clock window + IniBase
   *      iPosY  =    Y coord of clock window + IniBase
   *
   * To load as a (non-animated) icon, place on the load= statement of your
   * WIN.INI or hold shift key while clicking on it.  To load as a caption,
   * just run it or place on the run= statement.
   *
   * Version 1.01P  08/27/91
   *)

USES
  WObjects, WinTypes, WinProcs, WinDos, Strings;

{$R YaClock.Res }

CONST
  AppName : pchar = 'LibClock';

  cm_LibClock  = 1001;
  cm_RealClock = 1002;
  cm_About     = 1003;

  LibFreq = 60000;  { Timer Frequency for "Liberal" Clock }
  RealFreq= 10000;  { Timer Frequency for "Real" Clock }
  IniBase = 1000;   { Offset added to Ini Integer Values }

  Hours : Array[1..24] of pchar =
           ('one',   'two',   'three',   'four',  'five',   'six',
            'seven', 'eight', 'nine',    'ten',   'eleven', 'noon',
            'one',   'two',   'three',   'four',  'five',   'six',
            'seven', 'eight', 'nine',    'ten',   'eleven', 'midnight');

  Days : array [0..6] of pchar = ('Sun','Mon','Tue','Wed','Thr','Fri','Sat');

TYPE
  pClockWindow = ^tClockWindow;
  tClockWindow = object(tWindow)
    Liberal : Boolean;
    constructor Init;
    destructor  Done; virtual;
    procedure   SetUpWindow; virtual;
    procedure   GetWindowClass(var WndClass: TWndClass); virtual;
    procedure   SetMenu;
    procedure   WMTimer(var Msg:tMessage); virtual wm_First + wm_Timer;
    procedure   WMSysCommand(var Msg:tMessage); virtual wm_First + wm_SysCommand;
  end;

  tClockApp = object(TApplication)
    procedure  InitMainWindow; virtual;
  end;

{ --------------------------------------------------------------------------- }
{ General Routines                                                            }
{ --------------------------------------------------------------------------- }
function LiberalTimeStr(buffer:pChar):pChar;
  Const twelve : pchar = 'twelve';
  var H,M,S,S100:Word;
      Fmt : Array[0..50] of char;
      p: pChar;
  begin
      GetTime(H,M,S,S100);

      if M > 32 then  { Check for the 'til values }
      begin
         inc(H);
         if H = 25 then H := 1
      end;

      P := Hours[h];
      if (H = 12) AND (M>0) then  { Check for noon or twelve }
        P := Twelve;

      case M of
        0     : StrCopy(Fmt, '%s straight up');
        1..3  : StrCopy(Fmt, 'A little after %s');
        4..7  : StrCopy(Fmt, 'Around 5 after %s');
        8..12 : StrCopy(Fmt, 'About 10 after %s');
        13..17: StrCopy(Fmt, 'A quarter after %s or so');
        18..22: StrCopy(Fmt, 'Around 20 after %s');
        23..27: StrCopy(Fmt, 'About 25 after %s');
        28..29: StrCopy(Fmt, 'Almost half past %s');
        30..32: StrCopy(Fmt, 'Half past %s');
        33..37: StrCopy(Fmt, 'About 25 til %s');
        38..42: StrCopy(Fmt, 'Around 20 til %s');
        43..47: StrCopy(Fmt, 'About a quarter til %s');
        48..52: StrCopy(Fmt, 'About 10 til %s');
        53..56: StrCopy(Fmt, 'Around 5 til %s');
        57..59: StrCopy(Fmt, 'Almost %s');
      end;
      wvsprintf(Buffer, Fmt, P);
      LiberalTimeStr := Buffer
  end;

{ --------------------------------------------------------------------------- }
function RealTimeStr(buffer:pChar; ForceTime:Boolean):pChar;

  { Returns a string with the actual day/date/time in it.  Returns
    #0 if the last minute value is the same as the current minute
    value.  ForceTime is used to override this. }

  const LastMin:Integer = -1; { Last Minute }
  var
      Hr,Min,Sec,S100,
      Yr,Mon,Day,DW:Word;
      Data : record
        DOW: pChar;
        Mon, Pad1, Day, Yr,
        Hr:Word; Pad2, Min:Word;
        A  :Char;
      end;
      P : PChar;
  begin
      { Get time }
      GetTime(Hr,Min,Sec,S100);
      if (NOT ForceTime) AND (Min = LastMin) then
      begin
         Buffer[0] := #0; { Return Nothing }
         EXIT
      end;
      LastMin := Min;
      Data.Hr := Hr;
      Data.Min := Min;
      Data.A := 'a';
      if Hr >= 13 then
      begin
        dec(Data.Hr, 12);
        Data.A := 'p'
      end;
      if Min < 10 then Data.Pad2 := ORD('0')
      else
        Data.Pad2 := 0;

      { Get Date }
      GetDate(Yr,Mon,Day,DW);
      Data.Dow := Days[DW];
      Data.Yr := Yr-1900;
      Data.Day := Day;
      Data.Mon := Mon;

      if Day < 10 then Data.Pad1 := ORD('0')
      else
        Data.Pad1 := 0;

      wvsprintf(Buffer, '%s  %d/%c%d/%d  %d:%c%d%cm', Data);
      RealTimeStr := Buffer
  end;

{ --------------------------------------------------------------------------- }
{ tClockWindow                                                                }
{ --------------------------------------------------------------------------- }
constructor tClockWindow.Init;
  var DC: hDC;
      OldFont, Font: hFont;
      Metrics: tTextMetric;
      H,W:Integer;
      Lib: Boolean;
      Line: Array[0..40] of char;
  begin
     { Read the WIN.INI file for style }
     Lib := GetProfileInt(AppName, 'iFormat', 1) = 1;
     if Lib then
       LiberalTimeStr(Line)
     else
       RealTimeStr(Line, TRUE);

     tWindow.Init(NIL, Line);

     Liberal := Lib;

     { Figure out the size of the characters for our window }
     DC := GetWindowDC(GetDeskTopWindow);
     Font := GetStockObject(SYSTEM_FONT);
     OldFont := SelectObject(DC, Font);
     GetTextMetrics(DC, Metrics);
     SelectObject(DC, OldFont);
     W := GetDeviceCaps(DC, HorzRes);
     H := GetDeviceCaps(DC, VertRes);
     ReleaseDC(GetDeskTopWindow, DC); { Release - Don't Delete }

      { Now set min/max window coords }
     Attr.H := Metrics.tmHeight + 2;
     Attr.W := Metrics.tmAveCharWidth * 30{chars};

     { Read the WIN.INI file for position }
     if GetProfileInt(AppName, 'iPosX', 0) > 0 then
     begin
         Attr.X := GetProfileInt(AppName, 'iPosX', W) - IniBase;
         Attr.Y := GetProfileInt(AppName, 'iPosY', H) - IniBase
     end
     else
     begin
        Attr.X := W - Attr.W;
        Attr.Y := H - Attr.H - 2
     end;

     { Disable the Max/Minimize Box on Caption }
     Attr.Style := Attr.Style AND
                   (NOT ws_MaximizeBox) AND
                   (NOT ws_MinimizeBox) AND
                   (NOT ws_ThickFrame)
  end;

{ -------------------------------------------------------------------------- }
Destructor tClockWindow.Done;
  var Buffer : array[0..20] of char;
      Rect   : tRect;
  begin
    { Release the timer since there is a finite number of applications that
      can actually get a timer handle. (I think it's 16). }
    KillTimer(hWindow, wm_Timer);

    { Now update the WIN.INI file with the current clock style and
      window position }

    if Liberal then StrCopy(Buffer, '1')
    else
      StrCopy(Buffer, '0');
    WriteProfileString(AppName, 'iFormat', Buffer);

    { Adding in IniBase allows negative values up to -IniBase }
    GetWindowRect(hWindow, Rect);
    Str(Rect.Left+IniBase, Buffer);
    WriteProfileString(AppName, 'iPosX', Buffer);
    Str(Rect.Top+IniBase, Buffer);
    WriteProfileString(AppName, 'iPosY', Buffer);
    tWindow.Done
  end;

{ --------------------------------------------------------------------------- }
procedure tClockWindow.SetMenu;
  var Menu: hMenu;
  begin
     Menu := GetSystemMenu(hWindow, FALSE);
     if Liberal then
     begin
        CheckMenuItem(Menu, cm_LibClock,  mf_Checked);
        CheckMenuItem(Menu, cm_RealClock, mf_UnChecked)
     end
     else
     begin
        CheckMenuItem(Menu, cm_LibClock,  mf_UnChecked);
        CheckMenuItem(Menu, cm_RealClock, mf_Checked)
     end
  end;

{ --------------------------------------------------------------------------- }
procedure tClockWindow.SetUpWindow;
  var Menu: hMenu;
  begin
     tWindow.SetUpWindow;
     SetTimer(hWindow, wm_Timer, LibFreq, nil);
     { Now add our menu items }
     Menu := GetSystemMenu(hWindow, FALSE);
     InsertMenu(Menu, 9,  mf_Separator, 0, NIL);
     InsertMenu(Menu, 10, mf_SysMenu+mf_String, cm_LibClock,  'Li&beral clock');
     InsertMenu(Menu, 11, mf_SysMenu+mf_String, cm_RealClock, '&Real clock');
     InsertMenu(Menu, 12, mf_SysMenu+mf_String, cm_About,    '&About clock...');
     SetMenu
  end;

{ --------------------------------------------------------------------------- }
Procedure tClockWindow.wmTimer;
  var Line: Array[0..40] of char;
  begin
     if Liberal then
       LiberalTimeStr(Line)
     else
       RealTimeStr(Line, FALSE);
     if Line[0] <> #0 then
       SetWindowText(hWindow, Line)
  end;

{ --------------------------------------------------------------------------- }
procedure tClockWindow.WMSysCommand(var Msg:tMessage);
  var About : PDialog;
      Line: Array[0..40] of char;
  begin
     if Msg.wParam = cm_About then
     begin
        New(About, Init(@Self, 'ABOUT'));
        Application^.ExecDialog(About);
     end
     else if Msg.wParam = cm_LibClock then
     begin
        Liberal := TRUE;
        KillTimer(hWindow, wm_Timer);
        SetTimer(hWindow,  wm_Timer, LibFreq, nil);
        SetWindowText(hWindow, LiberalTimeStr(Line));
        SetMenu
     end
     else if Msg.wParam = cm_RealClock then
     begin
        Liberal := FALSE;
        KillTimer(hWindow, wm_Timer);
        SetTimer(hWindow,  wm_Timer, RealFreq, nil);
        SetWindowText(hWindow, RealTimeStr(Line,TRUE));
        SetMenu
     end
     else
       DefWindowProc(hWindow, Msg.Message, Msg.wParam, Msg.lParam)
  end;

{ -------------------------------------------------------------------------- }
procedure tClockWindow.GetWindowClass(var WndClass: TWndClass);
  begin
    TWindow.GetWindowClass(WndClass);
    WndClass.hIcon := LoadIcon(hInstance, 'ICON')
  end;

{ --------------------------------------------------------------------------- }
{ tClockApp                                                                   }
{ --------------------------------------------------------------------------- }
procedure tClockApp.InitMainWindow;
  begin
     MainWindow := New(pClockWindow, Init)
  end;

{ --------------------------------------------------------------------------- }
var ClockApp: tClockApp;
begin
   ClockApp.Init(AppName);
   ClockApp.Run;
   ClockApp.Done
end.
