
{*******************************************************}
{                                                       }
{      Borland Pascal 7.0 Simple 3D Library for         }
{      WinG Rotating Cube Demo                          }
{                                                       }
{      Translated from C++ version by                   }
{      Mike Scott, CIS 100140,2420, 24th July 1994      }
{                                                       }
{*******************************************************}

unit Dumb3D ;

interface

uses Objects ;

{ TQuadruple }

type
  PQuadElements = ^TQuadElements ;
  TQuadElements = record
    case byte of
      0 : ( x, y, z, w : real ) ;
      1 : ( Values : array[ 0..3 ] of real ) ;
  end ;

  PQuadruple = ^TQuadruple ;
  TQuadruple = object( TObject )
    Elements : TQuadElements ;
    constructor Init( x, y, z, w : real ) ;
    constructor InitCopy( const AQuadruple : TQuadruple ) ;
  end ;


{ TMatrix4x4 }

type
  PMatrixElements = ^TMatrixElements ;
  TMatrixElements = array[ 0..3, 0..3 ] of real ;

  PMatrix4x4 = ^TMatrix4x4 ;
  TMatrix4x4 = object( TObject )
    Elements : TMatrixElements ;
    constructor Init ;
    constructor InitOnly ;

    function  ConcatenateXRotation( Degrees : real ) : PMatrix4x4 ;
    function  ConcatenateYRotation( Degrees : real ) : PMatrix4x4 ;
    function  ConcatenateZRotation( Degrees : real ) : PMatrix4x4 ;

    function  ConcatenateXTranslation( Distance : real ) : PMatrix4x4 ;
    function  ConcatenateYTranslation( Distance : real ) : PMatrix4x4 ;
    function  ConcatenateZTranslation( Distance : real ) : PMatrix4x4 ;

    procedure MultiplyByMatrix( const AMatrix : TMatrix4x4 ) ;
  end ;


{ TPoint4 }

type
  PPoint4 = ^TPoint4 ;
  TPoint4 = object( TQuadruple )
    constructor Init( x, y, z : real ) ;
    constructor InitZero ;
    procedure Homogenize ;
    procedure MultiplyByMatrix( const AMatrix : TMatrix4x4 ) ;
  end ;


{ TVector4 }

  PVector4 = ^TVector4 ;
  TVector4 = object( TQuadruple )
    constructor Init( x, y, z : real ) ;
    constructor InitZero ;
    function  Normalize : PVector4 ;
    procedure Negate ;
    procedure Subtract( const AQuadruple : TQuadruple ) ;
    procedure MultiplyByMatrix( const AMatrix : TMatrix4x4 ) ;
    function  DotProduct( const AVector4 : TVector4 ) : real ;
    procedure CrossProduct( const AVector4 : TVector4 ) ;
  end ;


{ TViewTransform }

type
  PViewTransform = ^TViewTransform ;
  TViewTransform = object( TMatrix4x4 )
    constructor Init( const Viewpoint     : TPoint4 ;
                      const ViewDirection : TVector4 ;
                      const Up            : TVector4 ) ;
  end ;


implementation


{ TQuadruple }

constructor TQuadruple.Init( x, y, z, w : real ) ;

begin
  Elements.x := x ;
  Elements.y := y ;
  Elements.z := z ;
  Elements.w := w ;
end ;


constructor TQuadruple.InitCopy( const AQuadruple : TQuadruple ) ;

begin
  Elements := AQuadruple.Elements ;
end ;


{ TPoint4 }

constructor TPoint4.Init( x, y, z : real ) ;

begin
  inherited Init( x, y, z, 1 ) ;
end ;


constructor TPoint4.InitZero ;

begin
  TQuadruple.Init( 0, 0, 0, 1 ) ;
end ;


procedure TPoint4.Homogenize ;

begin
  with Elements do begin
    x := x / w ;
    y := y / w ;
    z := z / w ;
  end ;
end ;


procedure TPoint4.MultiplyByMatrix( const AMatrix : TMatrix4x4 ) ;

var AResult : TQuadElements ;
    AValue  : real ;
    i, j    : integer ;

begin
  for i := 0 to 3 do begin
    AValue := 0 ;
    { unroll loop to speed it up a bit }
    AValue := AValue + AMatrix.Elements[ i, 0 ] * Elements.Values[ 0 ] ;
    AValue := AValue + AMatrix.Elements[ i, 1 ] * Elements.Values[ 1 ] ;
    AValue := AValue + AMatrix.Elements[ i, 2 ] * Elements.Values[ 2 ] ;
    AValue := AValue + AMatrix.Elements[ i, 3 ] * Elements.Values[ 3 ] ;
    AResult.Values[ i ] := AValue ;
  end ;
  Elements := AResult ;
end ;


{ TVector4 }

constructor TVector4.Init( x, y, z : real ) ;

begin
  TQuadruple.Init( x, y, z, 0 ) ;
end ;


constructor TVector4.InitZero ;

begin
  TQuadruple.Init( 0, 0, 0, 0 ) ;
end ;


function  TVector4.Normalize : PVector4 ;

var ALength : real ;

begin
  with Elements do begin
    ALength := Sqrt( x * x + y * y + z * z ) ;
    x := x / ALength ;
    y := y / ALength ;
    z := z / ALength ;
  end ;
end ;


procedure TVector4.Negate ;

begin
  with Elements do begin
    x := -x ;
    y := -y ;
    z := -z ;
    w := 0 ;
  end ;
end ;


procedure TVector4.Subtract( const AQuadruple : TQuadruple ) ;

begin
  with Elements do begin
    x := x - AQuadruple.Elements.x ;
    y := y - AQuadruple.Elements.y ;
    z := z - AQuadruple.Elements.z ;
    w := 0 ;
  end ;
end ;


procedure TVector4.MultiplyByMatrix( const AMatrix : TMatrix4x4 ) ;

var i, j : integer ;
    AValue : real ;
    AResult : TQuadElements ;

begin
  for i := 0 to 3 do begin
    AValue := 0 ;
    for j := 0 to 3 do
      AValue := AValue + Elements.Values[ j ] * AMatrix.Elements[ i, j ] ;
    AResult.Values[ i ] := AValue ;
  end ;
  Elements := AResult ;
end ;


function  TVector4.DotProduct( const AVector4 : TVector4 ) : real ;

begin
  with Elements, AVector4 do
    DotProduct := x * Elements.x + y * Elements.y + z * Elements.z ;
end ;


procedure TVector4.CrossProduct( const AVector4 : TVector4 ) ;

var x, y, z : real ;

begin
  x := Elements.y * AVector4.Elements.z -
       Elements.z * AVector4.Elements.y ;
  y := Elements.z * AVector4.Elements.x -
       Elements.x * AVector4.Elements.z ;
  z := Elements.x * AVector4.Elements.y -
       Elements.y * AVector4.Elements.x ;
  Elements.x := x ;
  Elements.y := y ;
  Elements.z := z ;
  Elements.w := 0 ;
end ;


{ TMatrix4x4 }

constructor TMatrix4x4.Init ;

begin
  FillChar( Elements, sizeof( Elements ), 0 ) ;
  Elements[ 0, 0 ] := 1.0 ;
  Elements[ 1, 1 ] := 1.0 ;
  Elements[ 2, 2 ] := 1.0 ;
  Elements[ 3, 3 ] := 1.0 ;
end ;


constructor TMatrix4x4.InitOnly ;

begin
  FillChar( Elements, sizeof( Elements ), 0 ) ;
end ;


function  TMatrix4x4.ConcatenateXRotation( Degrees : real ) : PMatrix4x4 ;

var Temp01, Temp11, Temp21, Temp31 : real ;
    Temp02, Temp12, Temp22, Temp32 : real ;
    Radians : real ;
    ASin, ACos : real ;

begin
  Radians := Degrees * ( PI * 2 / 360 ) ;
  ASin := Sin( Radians ) ;
  ACos := Cos( Radians ) ;

  Temp01 := Elements[ 0, 1 ] * ACos + Elements[ 0, 2 ] * ASin ;
  Temp11 := Elements[ 1, 1 ] * ACos + Elements[ 1, 2 ] * ASin ;
  Temp21 := Elements[ 2, 1 ] * ACos + Elements[ 2, 2 ] * ASin ;
  Temp31 := Elements[ 3, 1 ] * ACos + Elements[ 3, 2 ] * ASin ;

  Temp02 := Elements[ 0, 1 ] * -ASin + Elements[ 0, 2 ] * ACos ;
  Temp12 := Elements[ 1, 1 ] * -ASin + Elements[ 1, 2 ] * ACos ;
  Temp22 := Elements[ 2, 1 ] * -ASin + Elements[ 2, 2 ] * ACos ;
  Temp32 := Elements[ 3, 1 ] * -ASin + Elements[ 3, 2 ] * ACos ;

  Elements[ 0, 1 ] := Temp01 ;
  Elements[ 1, 1 ] := Temp11 ;
  Elements[ 2, 1 ] := Temp21 ;
  Elements[ 3, 1 ] := Temp31 ;
  Elements[ 0, 2 ] := Temp02 ;
  Elements[ 1, 2 ] := Temp12 ;
  Elements[ 2, 2 ] := Temp22 ;
  Elements[ 3, 2 ] := Temp32 ;
end ;


function  TMatrix4x4.ConcatenateYRotation( Degrees : real ) : PMatrix4x4 ;

var Temp00, Temp10, Temp20, Temp30 : real ;
    Temp02, Temp12, Temp22, Temp32 : real ;
    Radians : real ;
    ASin, ACos : real ;

begin
  Radians := Degrees * ( PI * 2 / 360 ) ;
  ASin := Sin( Radians ) ;
  ACos := Cos( Radians ) ;

  Temp00 := Elements[ 0, 0 ] * ACos + Elements[ 0, 2 ] * -ASin ;
  Temp10 := Elements[ 1, 0 ] * ACos + Elements[ 1, 2 ] * -ASin ;
  Temp20 := Elements[ 2, 0 ] * ACos + Elements[ 2, 2 ] * -ASin ;
  Temp30 := Elements[ 3, 0 ] * ACos + Elements[ 3, 2 ] * -ASin ;

  Temp02 := Elements[ 0, 0 ] * ASin + Elements[ 0, 2 ] * ACos ;
  Temp12 := Elements[ 1, 0 ] * ASin + Elements[ 1, 2 ] * ACos ;
  Temp22 := Elements[ 2, 0 ] * ASin + Elements[ 2, 2 ] * ACos ;
  Temp32 := Elements[ 3, 0 ] * ASin + Elements[ 3, 2 ] * ACos ;

  Elements[ 0, 0 ] := Temp00 ;
  Elements[ 1, 0 ] := Temp10 ;
  Elements[ 2, 0 ] := Temp20 ;
  Elements[ 3, 0 ] := Temp30 ;
  Elements[ 0, 2 ] := Temp02 ;
  Elements[ 1, 2 ] := Temp12 ;
  Elements[ 2, 2 ] := Temp22 ;
  Elements[ 3, 2 ] := Temp32 ;
end ;


function  TMatrix4x4.ConcatenateZRotation( Degrees : real ) : PMatrix4x4 ;

var Temp00, Temp10, Temp20, Temp30 : real ;
    Temp01, Temp11, Temp21, Temp31 : real ;
    Radians : real ;
    ASin, ACos : real ;

begin
  Radians := Degrees * ( PI * 2 / 360 ) ;
  ASin := Sin( Radians ) ;
  ACos := Cos( Radians ) ;

  Temp00 := Elements[ 0, 0 ] * ACos + Elements[ 0, 1 ] * ASin ;
  Temp10 := Elements[ 1, 0 ] * ACos + Elements[ 1, 1 ] * ASin ;
  Temp20 := Elements[ 2, 0 ] * ACos + Elements[ 2, 1 ] * ASin ;
  Temp30 := Elements[ 3, 0 ] * ACos + Elements[ 3, 1 ] * ASin ;

  Temp01 := Elements[ 0, 0 ] * -ASin + Elements[ 0, 1 ] * ACos ;
  Temp11 := Elements[ 1, 0 ] * -ASin + Elements[ 1, 1 ] * ACos ;
  Temp21 := Elements[ 2, 0 ] * -ASin + Elements[ 2, 1 ] * ACos ;
  Temp31 := Elements[ 3, 0 ] * -ASin + Elements[ 3, 1 ] * ACos ;

  Elements[ 0, 0 ] := Temp00 ;
  Elements[ 1, 0 ] := Temp10 ;
  Elements[ 2, 0 ] := Temp20 ;
  Elements[ 3, 0 ] := Temp30 ;
  Elements[ 0, 1 ] := Temp01 ;
  Elements[ 1, 1 ] := Temp11 ;
  Elements[ 2, 1 ] := Temp21 ;
  Elements[ 3, 1 ] := Temp31 ;
end ;


function  TMatrix4x4.ConcatenateXTranslation( Distance : real ) : PMatrix4x4 ;

begin
  Elements[ 0, 3 ] := Elements[ 0, 0 ] * Distance + Elements[ 0, 3 ] ;
  Elements[ 1, 3 ] := Elements[ 1, 0 ] * Distance + Elements[ 1, 3 ] ;
  Elements[ 2, 3 ] := Elements[ 2, 0 ] * Distance + Elements[ 2, 3 ] ;
  Elements[ 3, 3 ] := Elements[ 3, 0 ] * Distance + Elements[ 3, 3 ] ;
end ;


function  TMatrix4x4.ConcatenateYTranslation( Distance : real ) : PMatrix4x4 ;

begin
  Elements[ 0, 3 ] := Elements[ 0, 1 ] * Distance + Elements[ 0, 3 ] ;
  Elements[ 1, 3 ] := Elements[ 1, 1 ] * Distance + Elements[ 1, 3 ] ;
  Elements[ 2, 3 ] := Elements[ 2, 1 ] * Distance + Elements[ 2, 3 ] ;
  Elements[ 3, 3 ] := Elements[ 3, 1 ] * Distance + Elements[ 3, 3 ] ;
end ;


function  TMatrix4x4.ConcatenateZTranslation( Distance : real ) : PMatrix4x4 ;

begin
  Elements[ 0, 3 ] := Elements[ 0, 2 ] * Distance + Elements[ 0, 3 ] ;
  Elements[ 1, 3 ] := Elements[ 1, 2 ] * Distance + Elements[ 1, 3 ] ;
  Elements[ 2, 3 ] := Elements[ 2, 2 ] * Distance + Elements[ 2, 3 ] ;
  Elements[ 3, 3 ] := Elements[ 3, 2 ] * Distance + Elements[ 3, 3 ] ;
end ;


procedure TMatrix4x4.MultiplyByMatrix( const AMatrix : TMatrix4x4 ) ;

var i, j, k : integer ;
    AValue  : real ;
    AResult : TMatrixElements ;

begin
  for i := 0 to 3 do begin
    for j := 0 to 3 do begin
      AValue := 0.0 ;
      for k := 0 to 3 do
        AValue := AValue + Elements[ i, k ] * AMatrix.Elements[ k, j ] ;
      AResult[ i, j ] := AValue ;
    end ;
  end ;
  Elements := AResult ;
end ;


{ TViewTransform }

constructor TViewTransform.Init( const Viewpoint     : TPoint4 ;
                                 const ViewDirection : TVector4 ;
                                 const Up            : TVector4 ) ;

var Right    : TVector4 ;
    ReallyUp : TVector4 ;
    LookDownZ : TMatrix4x4 ;
    i         : integer ;

begin
  Inherited Init ;

  { translate the viewpoint to the origin }
  ConcatenateXTranslation( -ViewPoint.Elements.x ) ;
  ConcatenateYTranslation( -ViewPoint.Elements.y ) ;
  ConcatenateZTranslation( -ViewPoint.Elements.z ) ;

  { get view vectors set up }
  Right.InitCopy( ViewDirection ) ;
  Right.CrossProduct( Up ) ;
  Right.Negate ;
  Right.Elements.w := 0 ;  { this is what C++ will do in effect }

  ReallyUp.InitCopy( Right ) ;
  ReallyUp.CrossProduct( ViewDirection ) ;

  LookDownZ.Init ;
  for i := 0 to 2 do begin
    LookDownZ.Elements[ 0, i ] := Right.Elements.Values[ i ] ;
    LookDownZ.Elements[ 1, i ] := ReallyUp.Elements.Values[ i ] ;
    LookDownZ.Elements[ 2, i ] := ViewDirection.Elements.Values[ i ] ;
  end ;

  LookDownZ.MultiplyByMatrix( Self ) ;
  Elements := LookDownZ.Elements ;
end ;


end.
