UNIT Graphics;


(*-------------------------------------------------------------------------*)
                                INTERFACE
(*-------------------------------------------------------------------------*)


  USES
    Graph,     (* Graphics module.          *)
    Fonts,     (* Graphics fonts            *)
    Drivers;   (* Graphics adaptor drivers. *)


  CONST
    NoGraphicsError            = grOk;
    GeneralGraphicsError       = grError;
    GraphicsDriverNotInstalled = grNoInitGraph;
    GraphicsAdaptorNotDetected = grNotDetected;
    GraphicsDriverFileNotFound = grFileNotFound;
    InvalidGraphicsDriver      = grInvalidDriver;
    NotEnoughMemoryForGraphics = grNoLoadMem;
    NoMemoryForScanFill        = grNoScanMem;
    NoMemoryForFloodFill       = grNoFloodMem;
    GraphicsFontFileNotFound   = grFontNotFound;
    NotEnoughMemoryForFont     = grNoFontMem;
    InvalidGraphicsFontFile    = grInvalidFont;
    InvalidGraphicsFontNumber  = grInvalidFontNum;
    GraphicsModeIsInvalid      = grInvalidMode;
    GraphicsIOError            = grIOError;
    InvalidGraphicsDevice      = grInvalidDeviceNum;

    LeftJustify                = 0;
    CenterJustify              = 1;
    RightJustify               = 2;
    BottomJustify              = 0;
    TopJustify                 = 2;
    HorizontalText             = 0;
    VerticalText               = 1;
    NormalSizeText             = 1;
    LargestSizeText            = 10;


  TYPE
    TypeBitImage     = RECORD
                         ImagePtr    : Pointer;
                         SizeOfImage : WORD
                       END;


  VAR
    GraphicsAdaptor,
    GraphicsMode          : INTEGER;
    LastGraphicsErrorCode : INTEGER;


  PROCEDURE EnterGraphicsMode (    GraphicsAdaptorToTry,
                                   GraphicsModeToTry     : INTEGER);


  PROCEDURE ExitGraphicsMode;


  PROCEDURE SelectFont (    FontNumber,
                            Direction,
                            SizeFactor,
                            HorizontalJustification,
                            VerticalJustification    : INTEGER);


  PROCEDURE GetBitImage (VAR ImageHolder : TypeBitImage;
                             LeftColumn,
                             TopRow,
                             RightColumn,
                             BottomRow   : INTEGER);


  PROCEDURE PutBitImage (VAR ImageToShow : TypeBitImage;
                             LeftColumn,
                             TopRow,
                             BitMask     : INTEGER);


  PROCEDURE ReleaseBitImageMemory (VAR ImageToKill : TypeBitImage);


  PROCEDURE ResetBitImageFile (VAR BitImageFile : FILE;
                                   Filename     : STRING);


  PROCEDURE RewriteBitImageFile (VAR BitImageFile : FILE;
                                     Filename     : STRING);


  PROCEDURE WriteBitImageToFile (VAR TargetFile   : FILE;
                                 VAR ImageToWrite : TypeBitImage);


  PROCEDURE ReadBitImageFromFile (VAR SourceFile  : FILE;
                                  VAR ImageHolder : TypeBitImage);



(*-------------------------------------------------------------------------*)
                              IMPLEMENTATION
(*-------------------------------------------------------------------------*)


  PROCEDURE EnterGraphicsMode;

    VAR
      DriverExists : BOOLEAN;

    BEGIN
      CASE GraphicsAdaptorToTry OF
        Detect   : DriverExists := TRUE;
        CGA      : DriverExists := (RegisterBGIDriver (@CGADriverProc) >= 0);
        MCGA     : DriverExists := (RegisterBGIDriver (@EGAVGADriverProc) >= 0);
        EGA      : DriverExists := (RegisterBGIDriver (@EGAVGADriverProc) >= 0);
        EGA64    : DriverExists := (RegisterBGIDriver (@EGAVGADriverProc) >= 0);
        EGAMono  : DriverExists := (RegisterBGIDriver (@EGAVGADriverProc) >= 0);
        HercMono : DriverExists := (RegisterBGIDriver (@HercDriverProc) >= 0);
        ATT400   : DriverExists := (RegisterBGIDriver (@ATTDriverProc) >= 0);
        VGA      : DriverExists := (RegisterBGIDriver (@EGAVGADriverProc) >= 0);
        PC3270   : DriverExists := (RegisterBGIDriver (@PC3270DriverProc) >= 0)
      END;
      IF DriverExists
        THEN
          BEGIN
            InitGraph (GraphicsAdaptorToTry, GraphicsModeToTry, '');
            LastGraphicsErrorCode := GraphResult
          END
        ELSE
          LastGraphicsErrorCode := GraphicsAdaptorNotDetected
    END;


  PROCEDURE ExitGraphicsMode;

    BEGIN
      CloseGraph
    END;


  PROCEDURE SelectFont;

    VAR
      FontAvailable : BOOLEAN;

    BEGIN
      CASE FontNumber OF
        TriplexFont   : LastGraphicsErrorCode := RegisterBGIFont (@TriplexFontProc);
        SmallFont     : LastGraphicsErrorCode := RegisterBGIFont (@SmallFontProc);
        SansSerifFont : LastGraphicsErrorCode := RegisterBGIFont (@SansSerifFontProc);
        GothicFont    : LastGraphicsErrorCode := RegisterBGIFont (@GothicFontProc);
        DefaultFont   : LastGraphicsErrorCode := NoGraphicsError
      END;
      IF (LastGraphicsErrorCode >= 0)
        THEN
          BEGIN
            SetTextStyle (FontNumber, Direction, SizeFactor);
            LastGraphicsErrorCode := GraphResult;
            SetTextJustify (HorizontalJustification, VerticalJustification)
          END
    END;


  PROCEDURE GetBitImage;

    BEGIN
      WITH ImageHolder DO
        BEGIN
          SizeOfImage := ImageSize (LeftColumn, TopRow,
                                    RightColumn, BottomRow);
          GetMem (ImagePtr, SizeOfImage);
          GetImage (LeftColumn, TopRow, RightColumn, BottomRow, ImagePtr^)
        END
    END;


  PROCEDURE PutBitImage;

    BEGIN
      PutImage (LeftColumn, TopRow, ImageToShow.ImagePtr^, BitMask)
    END;


  PROCEDURE ReleaseBitImageMemory;

    BEGIN
      WITH ImageToKill DO
        BEGIN
          FreeMem (ImagePtr, SizeOfImage);
          SizeOfImage := 0
        END
    END;


  PROCEDURE ResetBitImageFile;

    BEGIN
      Assign (BitImageFile, Filename);
      reset (BitImageFile, 1)
    END;


  PROCEDURE RewriteBitImageFile;

    BEGIN
      Assign (BitImageFile, Filename);
      rewrite (BitImageFile, 1)
    END;


  PROCEDURE WriteBitImageToFile;

    BEGIN
      WITH ImageToWrite DO
        BEGIN
          BlockWrite (TargetFile, SizeOfImage, 2);
          BlockWrite (TargetFile, ImagePtr^, SizeOfImage)
        END
    END;


  PROCEDURE ReadBitImageFromFile;

    BEGIN
      WITH ImageHolder DO
        BEGIN
          BlockRead (SourceFile, SizeOfImage, 2);
          GetMem (ImagePtr, SizeOfImage);
          BlockRead (SourceFile, ImagePtr^, SizeOfImage)
        END
    END;


END. (* of Graphics unit. *)