
(*
 * MakeGen - Generate a custom MAKEFILE for a set of .C sources
 *
 * Author: S.H.Smith, 27-Aug-86
 *
 * This utility is meant for use with the Datalight "MAKE" utility,
 * and should work with other Unix compatible Make utilities.
 *
 *)


uses tools;

const
   whoami = 'MakeGen';
   version = 'v1.1 (SHS 13-Sep-86)';

type
   anystring = string[80];
   string60 = string[60];
   string4  = string[4];

var
   objects:       array[1..200] of string60;
   exts:          array[1..200] of string4;
   object_count:  integer;
   mainfile:      anystring;


procedure get_object_list;
var
   i,j:    integer;
begin
   object_count := 0;
   for i := 2 to paramcount do
   begin
      getfiles(paramstr(i), filetable, filecount);
      for j := 1 to filecount do
      begin
         object_count := object_count + 1;
         objects[object_count] := copy(filetable[j]^,1,pos('.',filetable[j]^)-1);
         exts[object_count] := copy(filetable[j]^,pos('.',filetable[j]^),4);
      end;
   end;
end;


procedure output_dependency_list(source: anystring);
var
   fd:   text;
   line: anystring;
   i:    integer;
begin
   assign(fd,source);

   {$i-} reset(fd); {$i+}

   if ioresult <> 0 then
      writeln('Missing source: ',source)
   else

   begin
      while not eof(fd) do
      begin
         readln(fd,line);
         if pos('{',line) <> 0 then
         begin
            close(fd);
            exit;
         end;

         if (copy(line,1,8) = '#include') or
            (copy(line,1,9) = '# include') then
         begin
            write(' ');

            i := 8;
            while not (line[i] in ['<','"']) do
               i := i + 1;

            if line[i] = '<' then
               write('\include\');

            i := i + 1;
            while not (line[i] in ['"','>']) do
            begin
               write(upcase(line[i]));
               i := i + 1;
            end;
         end;
      end;

      close(fd);
   end;
end;


procedure gen_script_link;   {make a linker script to do the linkage}
                             {this is used when there are lots of files}
var
   i:     integer;
   parts: integer;
   first: char;

begin
   writeln;
   writeln('# Build linker command file');
   writeln;
   writeln(mainfile,'.l$(MODEL): makefile');
   write('   echo \lib\c$(MODEL)+');

   first := ' ';
   for i := 1 to object_count-1 do
   begin
      if (i mod 5) = 0 then
      begin
         writeln(' ',first,'>',mainfile,'.l$(MODEL)');
         write('   echo ');
         first := '>';
      end;
      write(objects[i],'+');
   end;

   writeln(objects[object_count],' >>',mainfile,'.l$(MODEL)');
   writeln('   echo ',mainfile,' >>',mainfile,'.l$(MODEL)');
   writeln('   echo ',mainfile,' >>',mainfile,'.l$(MODEL)');
   writeln('   echo \lib\nl$(MODEL) >>',mainfile,'.l$(MODEL)');
   writeln;

   writeln;
   writeln('# Object file abbreviations');
   writeln;
   parts := 1;
   write('p',parts,' = ');

   for i := 1 to object_count do
   begin
      if (i mod 5) = 0 then
      begin
         writeln;
         parts := parts + 1;
         write('p',parts,' = ');
      end;
      write(objects[i],'.OBJ ');
   end;
   writeln;

   writeln;
   writeln;
   writeln('# Build main file');
   writeln;
   write(mainfile,'.EXE:');
   for i := 1 to parts do
      write(' $(p',i,')');
   writeln(' ',mainfile,'.l$(MODEL)');
   writeln('   link /map/li/exepack @',mainfile,'.l$(MODEL)');
   writeln;
end;


procedure gen_dlc_link;     {use the dlc command to link}
var                         {used when there are only a few files}
   i:     integer;
   parts: integer;

begin
   writeln;
   writeln('# Object file abbreviations');
   writeln;
   parts := 1;
   write('p',parts,' = ');

   for i := 1 to object_count do
   begin
      if (i mod 5) = 0 then
      begin
         writeln;
         parts := parts + 1;
         write('p',parts,' = ');
      end;
      write(objects[i],'.OBJ ');
   end;
   writeln;

   writeln;
   writeln;
   writeln('# Build main file');
   writeln;
   write(mainfile,'.EXE:');
   for i := 1 to parts do
      write(' $(p',i,')');
   writeln;
   write('   dlc');
   for i := 1 to parts do
      write(' $(p',i,')');
   writeln(' -o',mainfile,'/exepack -g -m$(MODEL)');
   writeln;
end;


procedure generate_makefile;
var
   i:       integer;

begin
   writeln;
   writeln('#');
   writeln('# Makefile for "',mainfile,'" generated by ',whoami,'  ',version);
   writeln('#');
   writeln;
   writeln;
   writeln('# Define default macros');
   writeln('# (over-ride these with "macro=value" on MAKE command line)');
   writeln;
   writeln('# memory model');
   writeln('MODEL =');
   writeln;
   writeln('# compiler options');
   writeln('CCOPT = -c -w -g');
   writeln;
   writeln;
   writeln('# Default object causes entire program to be built');
   writeln;
   writeln(mainfile,'.SYM: ',mainfile,'.EXE ',mainfile,'.MAP');
   writeln('   mapsym ',mainfile);
   writeln;
   writeln;
   writeln('# Compile each source file');
   writeln;

   for i := 1 to object_count do
   begin
      write(objects[i],'.OBJ: ',objects[i],exts[i]);
      output_dependency_list(objects[i]+exts[i]);
      writeln;
      writeln('   dlc $(CCOPT) -m$(MODEL) ',objects[i],exts[i]);
      writeln;
   end;


   if object_count > 15 then
      gen_script_link            {make a linker script when lots of files}
   else
      gen_dlc_link               {otherwise use a dlc command line}
end;


begin
   if paramcount < 2 then
   begin
      writeln;
      writeln(whoami,'  ',version);
      writeln;
      writeln('Usage:   ',whoami,' mainfile source-file-list >MAKEFILE');
      writeln;
      writeln('Example: ',whoami,' CC68 *.C >MAKEFILE');
      writeln;
      halt;
   end;

   mainfile := paramstr(1);
   get_object_list;
   generate_makefile;
end.

