{  SPACE.PAS - Star_Scroller v1.1

   This is a star scroller in Turbo Pascal 7.0, using VGA mode 13h
   (320 x 200 at 256 colors). It is based on a program by Grant Smith
   (denthor@goth.vironix.co.za).

   It has colored stars. The colors fade from black to bright if the stars
   move closer towards the viewer. The stars also move at different speeds,
   thus giving a real sense of depth and speed.

   It is freeware. You may alter it or incorporate in your own programs.
   I would like to receive some feedback if you find it useful.

   While running, use the up and down arrow keys to change the speed.
   Press Esc to leave the program.

   Frederik Slijkerman.  CompuServe: 100651,405
                         Internet:   100651.405@compuserve.com

   Revision history:

   Version 1.0     -    Initial release
                        March 31, 1996

   Version 1.1     -    Changed compiler directives.
                        Added check on Speed variable.
                        Added colored stars (instead of grey and white).
                        Added different speeds of stars.
                        April 9, 1996
}

program Star_Scroller;

{$A+,B-,G+}

uses
   Crt;

const
{ These constants may be modified. }
   NumStars = 300;            { Number of stars. }
   ZShl = 4;
   MinSpeed = -500;           { These constants define min and max speed. }
   MaxSpeed = 500;

{ This version has colored stars and stars moving at a different speed. If
  you want to turn off these features easily, use the following constants. }
   WhiteStars = False;
   SameSpeed = False;

{ These not. }
   ScreenWords = 32000;       { 320 * 200 div 2 }
   NumCol = 256;

{ There are 256 colors. I divided them into 8 ranges of 8 different colors,
  thus allowing 32 gradiations of each color, ranging from 0 to 63. With the
  following constants, the colors are defined. Black is specified to provide
  a black background and must have the first index. }
   NumRanges = 8;

type
   RGBType = record
      R, G, B : byte;         { Palette entry. }
   end;

const
   ColorRanges : array[1..NumRanges] of RGBType = (
      ( R : 00; G : 00; B : 00 ),    { Black }
      ( R : 63; G : 63; B : 63 ),    { White }
      ( R : 30; G : 30; B : 63 ),    { Blue }
      ( R : 30; G : 63; B : 30 ),    { Green }
      ( R : 63; G : 30; B : 30 ),    { Red }
      ( R : 63; G : 30; B : 10 ),    { Orange }
      ( R : 63; G : 63; B : 10 ),    { Yellow }
      ( R : 55; G : 10; B : 63 ));   { Purple }

type
   StarType = record
      X, Y, Z : integer;      { Information on each star. }
      Spd,                    { Speed of the star. }
      Color : byte;           { Coloring information. }
   end;
   PosType = record
      X, Y : integer;         { Information on each point to be plotted. }
   end;

var
   StarData : array[1..NumStars] of StarType;
   ClearData : array[1..NumStars] of PosType;
   PaletteData : array[0..NumCol - 1] of RGBType;

   Speed,
   XCenter, YCenter : integer;

procedure InitGraphics; assembler;
{ Initializes VGA mode 13h. }

asm
   mov   ax, 0013h
   int   10h
end;

procedure CloseGraphics; assembler;
{ Puts system back in text mode. }

asm
   mov   ax, 0003h
   int   10h
end;

procedure SetVGAPalette(var Palette); assembler;
{ Set entire VGA palette using BIOS. }

asm
   mov   ax, 1012h
   xor   bx, bx
   mov   cx, NumCol
   les   dx, Palette
   int   10h
end;

procedure PutPixel(X, Y, Color : word); assembler;
{ Put a pixel on the real screen. }

asm
   { Clip. }
   mov   ax, Y
   or    ax, ax
   jb    @Exit
   cmp   ax, 200
   jae   @Exit

   mov   cx, X
   or    cx, cx
   jb    @Exit
   cmp   cx, 320
   jae   @Exit

   mov   bx, 0A000h
   mov   es, bx

   mov   bx, 320
   mul   bx
   add   ax, cx
   mov   di, ax
   mov   ax, Color
   stosb
@Exit:
end;

procedure LimitI(var Value : integer; Min, Max : integer); assembler;

asm
   les   di, Value
   mov   cx, es:[di]
   mov   ax, Min
   cmp   ax, cx
   jg    @Exit
   mov   ax, Max
   cmp   ax, cx
   jl    @Exit
   mov   ax, cx
@Exit:
   mov   es:[di], ax
end;

function GetRandom : integer;
(* Returns a random number: [-74, 75] \ { 0 } *)

var
   A : integer;

begin
   repeat
      A := Random(150) - 75;
   until A <> 0;
   GetRandom := A;
end;

function GetRandomColor : byte;
{ Returns a random value in the range 2..8 }

begin
   if WhiteStars
      then GetRandomColor := 2
      else GetRandomColor := Random(7) + 2;
end;

function GetRandomSpeed : byte;
{ Returns a random value in the range 1..8 }

begin
   if SameSpeed
      then GetRandomSpeed := 4
      else GetRandomSpeed := Random(8) + 1;
end;

procedure InitStars;
{ Initializes star positions. }

var
   A : word;

begin
   Randomize;
   for A := 1 to NumStars do
   with StarData[A] do
   begin
      X := GetRandom;
      Y := GetRandom;
      Z := 1 shl ZShl + Random(256 shl ZShl);
      if WhiteStars
         then Color := 2
         else Color := GetRandomColor;
      Spd := GetRandomSpeed;
   end;
end;

procedure InitPalette;
{ Initializes the palette. }

procedure InitRange(Range, Index : word);
{ Initializes a specified range of colors. }

var
   A, B : word;

begin
   A := Index;
   B := 0;
   while B < 32 do
   begin
      PaletteData[A].R := (ColorRanges[Range].R * B) shr 5;
      PaletteData[A].G := (ColorRanges[Range].G * B) shr 5;
      PaletteData[A].B := (ColorRanges[Range].B * B) shr 5;
      Inc(A);
      Inc(B);
   end;
end;

var
   Range, PalIdx : word;

begin
   PalIdx := 0;
   for Range := 1 to NumRanges do
   begin
      InitRange(Range, PalIdx);
      Inc(PalIdx, 32);
   end;
   SetVGAPalette(PaletteData);
end;

procedure HandleStars;

var
   StarX, StarY : integer;
   A, StarCol : word;
   StarSpd : integer;

begin
   for A := 1 to NumStars do
   begin
  { Limit speed. }
      LimitI(Speed, MinSpeed, MaxSpeed);
  { Calculate new position. }
      StarSpd := (Speed + StarData[A].Spd * Speed) div 8;
      StarData[A].Z := StarData[A].Z - StarSpd -
                       ((StarSpd shl 4) div (StarData[A].Z shr ZShl));
      if StarSpd >= 0 then
      begin
         if StarData[A].Z < 1 shl ZShl then
         with StarData[A] do
         begin
            X := GetRandom;
            Y := GetRandom;
            Z := 256 shl ZShl;
            if WhiteStars
               then Color := 2
               else Color := GetRandomColor;
            Spd := GetRandomSpeed;
         end;
      end else
      begin
         if StarData[A].Z > 256 shl ZShl then
         with StarData[A] do
         begin
            X := GetRandom;
            Y := GetRandom;
            Z := 1 shl ZShl;
            if WhiteStars
               then Color := 2
               else Color := GetRandomColor;
            Spd := GetRandomSpeed;
         end;
      end;

      StarX := XCenter - ((StarData[A].X shl 8) div
          (StarData[A].Z shr ZShl));
      StarY := YCenter - ((StarData[A].Y shl 8) div
          (StarData[A].Z shr ZShl));

      { Clear star. }
      PutPixel(ClearData[A].X, ClearData[A].Y, 0);

      { Draw new star. }
      StarCol := (StarData[A].Color shl 5);
      PutPixel(StarX, StarY, StarCol - (StarData[A].Z shr (ZShl + 3)));

      ClearData[A].X := StarX;
      ClearData[A].Y := StarY;
   end;
end;

procedure ScrollStars;
{ This is the main procedure. }

var
   KeyCode : char;

begin
   XCenter := 160;
   YCenter := 100;
   Speed := 0;
   repeat
      HandleStars;
      if KeyPressed then
      begin
         KeyCode := ReadKey;
         if KeyCode = #0 then
         begin
            KeyCode := ReadKey;
            case KeyCode of
               #72 : Inc(Speed);
               #80 : Dec(Speed);
               else KeyCode := #0;
            end;
         end;
      end;
      Delay(10);
   until KeyCode = #27;
end;

begin
   InitStars;
   InitGraphics;
   InitPalette;
   ScrollStars;
   CloseGraphics;
end.
