unit MD5;

interface

type
  TMD5Context = record
    State: array[0..3] of LongInt;
    Count: array[0..1] of LongInt;
    case Integer of
      0: (BufChar: array[0..63] of Byte);
      1: (BufLong: array[0..15] of LongInt)
  end;

  TMD5Digest = array[0..15] of Char;

procedure MD5Init(var MD5Context: TMD5Context);
procedure MD5Update(var MD5Context: TMD5Context; const Data; Len: Word);
function  MD5Final(var MD5Context: TMD5Context): TMD5Digest;

implementation

procedure MD5Init(var MD5Context: TMD5Context);
{  Start MD5 accumulation.  Set bit count to 0 and State to mysterious  }
{  initialization constants.                                            }
begin
  FillChar(MD5Context, SizeOf(TMD5Context), #0);
  with MD5Context do begin
    State[0] := $67452301;
    State[1] := $EFCDAB89;
    State[2] := $98BADCFE;
    State[3] := $10325476
  end
end;

procedure MD5Transform(var Buf: array of LongInt; const Data: array of LongInt); forward;

procedure MD5Update(var MD5Context: TMD5Context; const Data; Len: Word);
{  Update context to reflect the concatenation of another buffer full  }
{  of bytes.                                                           }
type
  TByteArray = array[0..0] of Byte;
var
  Index: Word;
  t: LongInt;
begin
  { Update bitcount }
  with MD5Context do begin
    t := Count[0];
    Inc(Count[0], LongInt(Len) shl 3);
    if Count[0] < t then
      Inc(Count[1]);
    Inc(Count[1], Len shr 29); { Makes no sense for Len of type Word, will be 0 }
    t := (t shr 3) and $3F;

    Index := 0;
    { Handle any leading odd-sized chunks }
    if t <> 0 then begin
      Index := t;
      t := 64 - t;
      if Len < t then begin
        Move(Data, BufChar[Index], Len);
        Exit
      end;
      Move(Data, BufChar[Index], t);
      MD5Transform(State, BufLong);
      Dec(Len, t)
    end;

    { Process data in 64-byte chunks }
    while Len >= 64 do begin
      Move(TByteArray(Data)[Index], BufChar, 64);
      MD5Transform(State, BufLong);
      Inc(Index, 64);
      Dec(Len, 64)
    end;

    { Handle any remaining bytes of data. }
    Move(TByteArray(Data)[Index], BufChar, Len)
  end
end;

function  MD5Final(var MD5Context: TMD5Context): TMD5Digest;
var
  Cnt: Word;
  p: Byte;
begin
  with MD5Context do begin
    { Compute number of bytes mod 64 }
    Cnt := (Count[0] shr 3) and $3F;

    { Set the first char of padding to $80 }
    p := Cnt;
    BufChar[p] := $80;
    Inc(p);

    { Bytes of padding needed to make 64 bytes }
    Cnt := 64 - 1 - Cnt;

    { Pad out to 56 mod 64 }
    if Cnt < 8 then begin
      { Two lots of padding:  Pad the first block to 64 bytes }
      FillChar(BufChar[p], Cnt, #0);
      MD5Transform(State, BufLong);

      { Now fill the next block with 56 bytes }
      FillChar(BufChar, 56, #0)
    end else
      { Pad block to 56 bytes }
      FillChar(BufChar[p], Cnt - 8, #0);

    { Append length in bits and transform }
    BufLong[14] := Count[0];
    BufLong[15] := Count[1];
    MD5Transform(State, BufLong);

    Move(State, Result, 16)
  end;

  FillChar(MD5Context, SizeOf(TMD5Context), #0)
end;

procedure MD5STEP1(var w: LongInt; x, y, z, data: LongInt; s: Byte);
begin
  Inc(w, (z xor (x and (y xor z))) + data);
  w := (w shl s) or (w shr (32 - s));
  Inc(w, x)
end;

procedure MD5STEP2(var w: LongInt; x, y, z, data: LongInt; s: Byte);
begin
  Inc(w, (y xor (z and (x xor y))) + data);
  w := (w shl s) or (w shr (32 - s));
  Inc(w, x)
end;

procedure MD5STEP3(var w: LongInt; x, y, z, data: LongInt; s: Byte);
begin
  Inc(w, (x xor y xor z) + data);
  w := (w shl s) or (w shr (32 - s));
  Inc(w, x)
end;

procedure MD5STEP4(var w: LongInt; x, y, z, data: LongInt; s: Byte);
begin
  Inc(w, (y xor (x or not z)) + data);
  w := (w shl s) or (w shr (32 - s));
  Inc(w, x)
end;

procedure MD5Transform(var Buf: array of LongInt; const Data: array of LongInt);
var
  a, b, c, d: LongInt;
begin
  a := Buf[0];
  b := Buf[1];
  c := Buf[2];
  d := Buf[3];

  MD5STEP1(a, b, c, d, Data[ 0] + $d76aa478,  7);
  MD5STEP1(d, a, b, c, Data[ 1] + $e8c7b756, 12);
  MD5STEP1(c, d, a, b, Data[ 2] + $242070db, 17);
  MD5STEP1(b, c, d, a, Data[ 3] + $c1bdceee, 22);
  MD5STEP1(a, b, c, d, Data[ 4] + $f57c0faf,  7);
  MD5STEP1(d, a, b, c, Data[ 5] + $4787c62a, 12);
  MD5STEP1(c, d, a, b, Data[ 6] + $a8304613, 17);
  MD5STEP1(b, c, d, a, Data[ 7] + $fd469501, 22);
  MD5STEP1(a, b, c, d, Data[ 8] + $698098d8,  7);
  MD5STEP1(d, a, b, c, Data[ 9] + $8b44f7af, 12);
  MD5STEP1(c, d, a, b, Data[10] + $ffff5bb1, 17);
  MD5STEP1(b, c, d, a, Data[11] + $895cd7be, 22);
  MD5STEP1(a, b, c, d, Data[12] + $6b901122,  7);
  MD5STEP1(d, a, b, c, Data[13] + $fd987193, 12);
  MD5STEP1(c, d, a, b, Data[14] + $a679438e, 17);
  MD5STEP1(b, c, d, a, Data[15] + $49b40821, 22);

  MD5STEP2(a, b, c, d, Data[ 1] + $f61e2562,  5);
  MD5STEP2(d, a, b, c, Data[ 6] + $c040b340,  9);
  MD5STEP2(c, d, a, b, Data[11] + $265e5a51, 14);
  MD5STEP2(b, c, d, a, Data[ 0] + $e9b6c7aa, 20);
  MD5STEP2(a, b, c, d, Data[ 5] + $d62f105d,  5);
  MD5STEP2(d, a, b, c, Data[10] + $02441453,  9);
  MD5STEP2(c, d, a, b, Data[15] + $d8a1e681, 14);
  MD5STEP2(b, c, d, a, Data[ 4] + $e7d3fbc8, 20);
  MD5STEP2(a, b, c, d, Data[ 9] + $21e1cde6,  5);
  MD5STEP2(d, a, b, c, Data[14] + $c33707d6,  9);
  MD5STEP2(c, d, a, b, Data[ 3] + $f4d50d87, 14);
  MD5STEP2(b, c, d, a, Data[ 8] + $455a14ed, 20);
  MD5STEP2(a, b, c, d, Data[13] + $a9e3e905,  5);
  MD5STEP2(d, a, b, c, Data[ 2] + $fcefa3f8,  9);
  MD5STEP2(c, d, a, b, Data[ 7] + $676f02d9, 14);
  MD5STEP2(b, c, d, a, Data[12] + $8d2a4c8a, 20);

  MD5STEP3(a, b, c, d, Data[ 5] + $fffa3942,  4);
  MD5STEP3(d, a, b, c, Data[ 8] + $8771f681, 11);
  MD5STEP3(c, d, a, b, Data[11] + $6d9d6122, 16);
  MD5STEP3(b, c, d, a, Data[14] + $fde5380c, 23);
  MD5STEP3(a, b, c, d, Data[ 1] + $a4beea44,  4);
  MD5STEP3(d, a, b, c, Data[ 4] + $4bdecfa9, 11);
  MD5STEP3(c, d, a, b, Data[ 7] + $f6bb4b60, 16);
  MD5STEP3(b, c, d, a, Data[10] + $bebfbc70, 23);
  MD5STEP3(a, b, c, d, Data[13] + $289b7ec6,  4);
  MD5STEP3(d, a, b, c, Data[ 0] + $eaa127fa, 11);
  MD5STEP3(c, d, a, b, Data[ 3] + $d4ef3085, 16);
  MD5STEP3(b, c, d, a, Data[ 6] + $04881d05, 23);
  MD5STEP3(a, b, c, d, Data[ 9] + $d9d4d039,  4);
  MD5STEP3(d, a, b, c, Data[12] + $e6db99e5, 11);
  MD5STEP3(c, d, a, b, Data[15] + $1fa27cf8, 16);
  MD5STEP3(b, c, d, a, Data[ 2] + $c4ac5665, 23);

  MD5STEP4(a, b, c, d, Data[ 0] + $f4292244,  6);
  MD5STEP4(d, a, b, c, Data[ 7] + $432aff97, 10);
  MD5STEP4(c, d, a, b, Data[14] + $ab9423a7, 15);
  MD5STEP4(b, c, d, a, Data[ 5] + $fc93a039, 21);
  MD5STEP4(a, b, c, d, Data[12] + $655b59c3,  6);
  MD5STEP4(d, a, b, c, Data[ 3] + $8f0ccc92, 10);
  MD5STEP4(c, d, a, b, Data[10] + $ffeff47d, 15);
  MD5STEP4(b, c, d, a, Data[ 1] + $85845dd1, 21);
  MD5STEP4(a, b, c, d, Data[ 8] + $6fa87e4f,  6);
  MD5STEP4(d, a, b, c, Data[15] + $fe2ce6e0, 10);
  MD5STEP4(c, d, a, b, Data[ 6] + $a3014314, 15);
  MD5STEP4(b, c, d, a, Data[13] + $4e0811a1, 21);
  MD5STEP4(a, b, c, d, Data[ 4] + $f7537e82,  6);
  MD5STEP4(d, a, b, c, Data[11] + $bd3af235, 10);
  MD5STEP4(c, d, a, b, Data[ 2] + $2ad7d2bb, 15);
  MD5STEP4(b, c, d, a, Data[ 9] + $eb86d391, 21);

  Inc(Buf[0], a);
  Inc(Buf[1], b);
  Inc(Buf[2], c);
  Inc(Buf[3], d)
end;

end.
