{

OvrLock is a function that places a "lock" on the overlay buffer.  Any units
present in the overlay buffer when the lock is applied will remain fixed in
memory until the lock is removed.  While the lock is in effect the overlay
manager continues to operate normally using the balance of the overlay buffer
and the remaining overlaid units.

It is the programmer's responsibility to load the desired units into
the overlay buffer before calling OvrLock.  One way to accomplish this 
is by doing a OvrClearBuf and then a dummy call to each unit.  For example:

  OvrClearBuf;
  Unit1_dummy;
  Unit2_dummy;
  OvrLockResult := OvrLock;

OvrLock returns an integer result code. The following result codes are
defined:

0  Successful completion.
1  Overlay buffer is empty.  There are no units to lock.
2  Free space not at end of buffer.  OvrLock requires all units in the
   overlay buffer to be loaded contiguously at the beginning of the buffer.
3  Overlay buffer already locked.  Only one lock can be placed on the overlay
   buffer at a time.  If you want to lock more units into the buffer, first
   call OvrUnlock, load the additional unit(s), and then call OvrLock again.
4  Not enough free space left.  There must be enough free space left in the
   buffer for the largest overlaid unit that is not already loaded.


OvrUnlock removes the lock from the overlay buffer.  All units that were
locked remain in the buffer when OvrUnlock is called.  All other units are
cleared.  OvrUnlock returns an integer result code.  The following result codes
are defined:

0  Successful completion.
1  Overlay buffer is not locked.


Written by Ron Schuster (CIS 76666,2322).  Copyright (c) 1989.
All rights reserved.  May be distributed freely, but not for a profit.

Version 0.2, 12/05/89
--------------------
  Beta release.
}

{$R-,S-,I-,V-,F-,B-,O-}

unit OverLock;

interface

uses
  Overlay;

  function OvrLock : Integer;
  function OvrUnlock : Integer;

implementation

type
  DispatcherHeader = record
    ReturnInt : Word;
    ReturnOfs : Word;
    FileOfs : LongInt;
    CodeSize : Word;
    FixupSize : Word;
    EntryPts : Word;
    CodeListNext : Word;
    LoadSegment : Word;
    Reprieved : Word;
    LoadListNext : Word;
  end;

  var
    SaveOvrHeapOrg : Word;
    SaveOvrHeapSize : Word;
    SaveOvrLoadList : Word;

  function Max (A,B : Word) : Word;
  begin
    if A > B then
      Max := A
    else
      Max := B;
  end;

  function NextPara(Bytes : LongInt) : LongInt;
  begin
    NextPara := (Bytes+15) and $FFFFFFF0;
  end;

  function OvrLock : Integer;
  var
    P : Word;
  begin
    if OvrLoadList = 0 then
      OvrLock := 1  { Overlay buffer is empty }

    else if DispatcherHeader(Ptr(OvrLoadList,0)^).LoadSegment <> OvrHeapOrg then
      OvrLock := 2  { Free space not at end of buffer }

    else if SaveOvrHeapOrg <> 0 then
      OvrLock := 3  { Overlay buffer already locked }

    else begin
      { Find largest overlaid unit that is not already loaded }
      P := OvrCodeList;
      SaveOvrHeapSize := OvrHeapSize;
      OvrHeapSize := 0;
      while P <> 0 do
        with DispatcherHeader (Ptr(P+PrefixSeg+$10,0)^) do begin
          if LoadSegment = 0 then
            OvrHeapSize := Max (OvrHeapSize,
              (NextPara(CodeSize) + NextPara(FixupSize)) shr 4);
           P := CodeListNext;
        end;

      { Compare largest unit's requirement to available buffer free space }
      if OvrHeapEnd - OvrHeapPtr < OvrHeapSize then begin
        OvrHeapSize := SaveOvrHeapSize;
        OvrLock := 4;  { Not enough free space left }
      end

      else begin
        SaveOvrHeapOrg := OvrHeapOrg;    { Save values to }
        SaveOvrLoadList := OvrLoadList;  { restore later }

        OvrHeapOrg := OvrHeapPtr; { Set start of buffer to start of free space}
        OvrLoadList := 0;         { "Hide" loaded units from overlay manager }

        OvrLock := 0;  { Successful completion }
      end;
    end;
  end;

  function OvrUnlock : Integer;
  begin
    if SaveOvrHeapOrg = 0 then
      OvrUnlock := 1  { Overlay buffer not locked }

    else begin
      OvrClearBuf;  { Remove any units above lock point }

      OvrHeapOrg := SaveOvrHeapOrg;  { Restore original values }
      OvrLoadList := SaveOvrLoadList;
      OvrHeapSize := SaveOvrHeapSize;

      SaveOvrHeapOrg := 0;  { Indicate no lock in effect }
      OvrUnlock := 0;       { Successful completion }
    end;
  end;

begin
  SaveOvrHeapOrg := 0;  { Indicate no lock in effect }
end.

