unit BtnBar;

interface

uses
  Winprocs,
  Wintypes,
  Objects,
  OWindows,
  Strings,
  Win31,
  MLBTypes;

{$R BTNBAR.RES}

const
  tm_CalcParentClientRect  = wm_User + 120;
  tm_SizingEnd             = wm_User + 122;
  tm_NewColSize            = wm_User + 127;
  tm_FirstColSize          = wm_User + 128;
  coDarkGray               = $808080;
  DenyRepaint              = 0;
  AllowRepaint             = 1;
  BorderWidth              = 1;

type
  PTool = ^TTool;
  TTool = object(TObject)
    Parent: PWindowsObject;
    constructor Init(AParent: PWindowsObject);
    function    GetWidth: Integer; virtual;
    procedure   Check(State: Boolean); virtual;
    function    GetHeight: Integer; virtual;
    procedure   GetRect(var AR: TRect); virtual;
    function    GetPart: Real; virtual;
    procedure   Resize(APart: Real); virtual;
    function    HitTest(P: TPoint): Boolean; virtual;
    function    HitSize(P: TPoint): Boolean; virtual;
    procedure   Paint(DC, AMemDC: HDC; var PS: TPaintStruct); virtual;
    procedure   BeginCapture(P: TPoint); virtual;
    procedure   ContinueCapture(P: TPoint); virtual;
    function    EndCapture(SendTo: HWnd; P: TPoint): Boolean; virtual;
    procedure   BeginNCapture(P: TPoint); virtual;
    procedure   ContinueNCapture(P: TPoint); virtual;
    function    EndNCapture(SendTo: HWnd; P: TPoint): Boolean; virtual;
    procedure   BeginSCapture(P: TPoint); virtual;
    procedure   ContinueSCapture(P: TPoint); virtual;
    function    EndSCapture(SendTo: HWnd; P: TPoint): Boolean; virtual;
    function    HasCommand(Command: Word): Boolean; virtual;
    function    IsToolChecked: Boolean; virtual;
    function    GetCommand: Word; virtual;
    procedure   Enable(State: Boolean); virtual;
    procedure   SetOrigin(X, Y: Integer); virtual;
    procedure   CalculateWidth(BarWidth: Word; var XOfs: Integer); virtual;
  end;

  PButtonBar = ^TButtonBar;
  TButtonBar = object(TWindow)
    ButtonsCount: Integer;
    Buttons     : TCollection;
    Capture     : PTool;
    Sizing      : Boolean;
    constructor Init(AParent: PWindowsObject; AnItemList: PItemsList; ABarColor: TColorRef);
    destructor  Done; virtual;
    function    CreateTool(Num: Integer; Origin: TPoint; Command: Word;
                           BtnName: PChar; BtnPart: Real; AnAlign: Word; AColor: TColorRef): PTool;
    procedure   EnableTool(Command: Word; NewState: Boolean); virtual;
    procedure   CheckTool(Command: Word);
    function    GetHeight: Integer;
    function    GetClassName: PChar; virtual;
    procedure   GetWindowClass(var WC: TWndClass); virtual;
    procedure   GetToolPos(ToolID: Integer; var StartPos, EndPos: Integer); virtual;
    function    GetToolPart(ToolID: Integer): Real;
    function    GetSortOrder: Integer;
    procedure   Paint(DC: HDC; var PS: TPaintStruct); virtual;
    procedure   AMCalcParentClientRect(var Msg: TMessage); virtual wm_First + tm_CalcParentClientRect;
    procedure   ToolSizingEnd(var Msg: TMessage); virtual wm_First + tm_SizingEnd;
    procedure   WMLButtonDown(var Msg: TMessage); virtual wm_First + wm_LButtonDown;
    procedure   WMMouseMove(var Msg: TMessage); virtual wm_First + wm_MouseMove;
    procedure   WMLButtonUp(var Msg: TMessage); virtual wm_First + wm_LButtonUp;
  end;

  PBarButton = ^TBarButton;
  TBarButton = object(TTool)
    Caption     : PChar;
    Command     : Word;
    Part        : Real;
    Align       : Word;
    NCapturing,
    SCapturing,
    IsPressed,
    IsEnabled,
    IsChecked   : Boolean;
    R           : TRect;
    GlyphSize   : TPoint;
    CapDC,
    MemDC       : HDC;
    BarColor    : TColorRef;
    constructor Init(AParent: PWindowsObject; ACommand: Word; AName: PChar; APart: Real; AnAlign: Word;
                     AColor: TColorRef);
    destructor  Done; virtual;
    function    HasCommand(ACommand: Word): Boolean; virtual;
    function    IsToolChecked: Boolean; virtual;
    function    GetCommand: Word; virtual;
    procedure   Enable(State: Boolean); virtual;
    procedure   Check(State: Boolean); virtual;
    function    GetWidth: Integer; virtual;
    function    GetHeight: Integer; virtual;
    procedure   GetRect(var AR: TRect); virtual;
    function    GetPart: Real; virtual;
    procedure   Resize(APart: Real); virtual;
    procedure   SetOrigin(X, Y: Integer); virtual;
    function    HitTest(P: TPoint): Boolean; virtual;
    function    HitSize(P: TPoint): Boolean; virtual;
    procedure   CalculateWidth(BarWidth: Word; var XOfs: Integer); virtual;
    procedure   Paint(DC, AMemDC: HDC; var PS: TPaintStruct); virtual;
    procedure   PaintState(DC, AMemDC: HDC);
    procedure   BeginNCapture(P: TPoint); virtual;
    procedure   ContinueNCapture(P: TPoint); virtual;
    function    EndNCapture(SendTo: HWnd; P: TPoint): Boolean; virtual;
    procedure   BeginSCapture(P: TPoint); virtual;
    procedure   ContinueSCapture(P: TPoint); virtual;
    function    EndSCapture(SendTo: HWnd; P: TPoint): Boolean; virtual;
    procedure   PressIn;
    procedure   PressOut;
  end;

{ Unit wide resources }
var
  ButtonFont  : HFont;
  WhitePen,
  DarkGrayPen,
  BlackPen,
  DotPen      : HPen;
  GrayBrush,
  GrayingBrush: HBrush;
  SizCursor,
  ArrowCursor : HCursor;

implementation

function Max(A, B: Integer): Integer;
begin
  if A > B then
    Max := A
  else
    Max := B;
end;

{ ********** TTool *********** }

constructor TTool.Init(AParent: PWindowsObject);
begin
  Parent := AParent;
end;

function TTool.GetWidth: Integer;
begin
  GetWidth := 0;
end;

function TTool.GetHeight: Integer;
begin
  GetHeight := 0;
end;

procedure TTool.GetRect(var AR: TRect);
begin
end;

function TTool.GetPart: Real;
begin
end;

procedure TTool.Resize;
begin
end;

function TTool.HitTest(P: TPoint): Boolean;
begin
  HitTest := False;
end;

function TTool.HitSize(P: TPoint): Boolean;
begin
  HitSize := False;
end;

procedure TTool.Paint(DC, AMemDC: HDC; var PS: TPaintStruct);
begin
end;

procedure TTool.BeginCapture(P: TPoint);
begin
end;

procedure TTool.ContinueCapture(P: TPoint);
begin
end;
 
function TTool.EndCapture(SendTo: HWnd; P: TPoint): Boolean;
begin
end;

procedure TTool.BeginNCapture(P: TPoint);
begin
end;

procedure TTool.ContinueNCapture(P: TPoint);
begin
end;

function TTool.EndNCapture(SendTo: HWnd; P: TPoint): Boolean;
begin
end;

procedure TTool.BeginSCapture(P: TPoint);
begin
end;

procedure TTool.ContinueSCapture(P: TPoint);
begin
end;

function TTool.EndSCapture(SendTo: HWnd; P: TPoint): Boolean;
begin
end;

procedure TTool.Check(State: Boolean);
begin
end;

procedure TTool.Enable(State: Boolean);
begin
end;

procedure TTool.SetOrigin(X, Y: Integer);
begin
end;

function TTool.HasCommand(Command: Word): Boolean;
begin
  HasCommand := False;
end;

procedure TTool.CalculateWidth(BarWidth: Word; var Xofs: Integer);
begin
end;

function TTool.IsToolChecked: Boolean;
begin
  IsToolChecked := False;
end;

function TTool.GetCommand: Word;
begin
  GetCommand := 0;
end;

{ ********** TButtonBar ********** }

constructor TButtonBar.Init(AParent: PWindowsObject; AnItemList: pItemsList; ABarColor: TColorRef);
var
  X: Integer;
  Origin: TPoint;
  P: PTool;
begin
  inherited Init(AParent, nil);
  Attr.Style := ws_Child or ws_Visible or ws_Border;
  SetFlags(wb_MDIChild, False);
  DefaultProc := @DefWindowProc;
  Attr.X := -1;
  Attr.Y := -1;
  Attr.W := 20;
  Attr.H := 18;
  Capture := nil;
  Sizing := False;
  ButtonsCount := AnItemList^.ColNumber;
  Buttons.Init(ButtonsCount, 1);
  Origin.X := 0;
  Origin.Y := 0;
  for X := 1 to ButtonsCount do
    With AnItemList^.Items^[X] do
    begin
      P := CreateTool(X, Origin, ItemID, Caption, Part, Align, ABarColor);
      if P <> nil then
      begin
        Inc(Origin.X, 20);
        if AnItemList^.Items^[X].Sort = True then P^.Check(True);
        Buttons.Insert(P);
      end;
    end;
end;

destructor TButtonBar.Done;
begin
  inherited Done;
  Buttons.Done;
end;

function TButtonBar.CreateTool(Num: Integer; Origin: TPoint;
                               Command: Word; BtnName: PChar;
                               BtnPart: Real; AnAlign: Word; AColor: TColorRef): PTool;
begin
  CreateTool := New(PBarButton, Init(@Self, Command, BtnName, BtnPart, AnAlign, AColor));
end;

procedure TButtonBar.EnableTool(Command: Word; NewState: Boolean);
var
  P: PTool;

  function FoundIt(P: PTool): Boolean; far;
  begin
    FoundIt := P^.HasCommand(Command);
  end;

begin
  P := Buttons.FirstThat(@FoundIt);
  if P <> nil then
    P^.Enable(NewState);
end;

function TButtonBar.GetClassName: PChar;
begin
  GetClassName := 'MButtonBar';
end;

procedure TButtonBar.GetWindowClass(var WC: TWndClass);
begin
  TWindow.GetWindowClass(WC);
  WC.hbrBackground := GetStockObject(Null_Brush);
  WC.hCursor := 0;
end;

procedure TButtonBar.GetToolPos(ToolID: Integer; var StartPos, EndPos: Integer);
var
  P: PTool;
  R: TRect;

  function FoundIt(P: PTool): Boolean; far;
  begin
    FoundIt := P^.HasCommand(ToolID);
  end;

begin
  P := Buttons.FirstThat(@FoundIt);
  P^.GetRect(R);
  StartPos := R.left;
  EndPos := R.Right;
end;

function TButtonBar.GetHeight: Integer;
var
  P: PTool;
begin
  GetHeight := Attr.H;
end;

function TButtonBar.GetToolPart(ToolID: Integer): Real;
var
  P: PTool;

  function FoundIt(P: PTool): Boolean; far;
  begin
    FoundIt := P^.HasCommand(ToolID);
  end;

begin
  P := Buttons.FirstThat(@FoundIt);
  GetToolPart := P^.GetPart;
end;

function TButtonBar.GetSortOrder: Integer;
var
  P: PTool;

  function FoundIt(P: PTool): Boolean; far;
  begin
    FoundIt := P^.IsToolChecked;
  end;

begin
  P := Buttons.FirstThat(@FoundIt);
  GetSortOrder := P^.GetCommand;
end;

procedure TButtonBar.Paint(DC: HDC; var PS: TPaintStruct);
var
  MemDC: HDC;

  procedure PaintIt(Item: PTool); far;
  begin
    Item^.Paint(DC, MemDC, PS);
  end;

begin
  MemDC := CreateCompatibleDC(DC);
  Buttons.ForEach(@PaintIt);
  DeleteDC(MemDC);
end;

procedure TButtonBar.AMCalcParentClientRect(var Msg: TMessage);
var
  BB,                                { ButtonBar rect in screen coords }
  PC,                                { Parent client rect in screen coords }
  NewBB,                             { New ButtonBar rect in screen coords }
  R: TRect;                          { Scratch }
  S2PC, S2BB: TPoint;                { Screen to local coord conversion offsets }
  XOfs: Integer;

  procedure SetWidth(Item: PTool); far;
  begin
    Item^.CalculateWidth(PC.Right - PC.Left, XOfs);
  end;

begin
  PC := PRect(Msg.LParam)^;
  R := PC;
  ClientToScreen(Parent^.HWindow, PPoint(@PC)^);
  ClientToScreen(Parent^.HWindow, PPoint(@PC.Right)^);
  S2PC.X := PC.Left - R.Left;
  S2PC.Y := PC.Top - R.Top;

  GetWindowRect(HWindow, BB);
  S2BB.X := BB.Left;
  S2BB.Y := BB.Top;

  if Bool(Msg.WParam) then           { We have permission to repaint & reposition }
  begin
    if BB.Right <> PC.Right then     { Parent client relative coords }
      SetWindowPos(HWindow, 0, -1, 0, PC.Right - S2BB.X, BB.Bottom - S2BB.Y, swp_NoZOrder);
    if BB.Right < PC.Right then      { Width increases, paint new area }
    begin
      SetRect(R, BB.Right - S2BB.X - 2, BB.Top - S2BB.Y - 1, PC.Right - S2BB.X + 1, BB.Bottom - S2BB.Y + 1);
      InvalidateRect(HWindow, @R, True);
    end;
    if PC.Top < BB.Bottom then
      PC.Top := BB.Bottom;
  end;

  { Map the screen coordinates PC record back into parent relative coords }
  SetRect(PRect(Msg.LParam)^, PC.Left - S2PC.X, PC.Top - S2PC.Y, PC.Right - S2PC.X, PC.Bottom - S2PC.Y);

  XOfs := 0;
  Buttons.ForEach(@SetWidth);
end;

procedure TButtonBar.ToolSizingEnd(var Msg: TMessage);
var
  R, PR, SR: TRect;
  P: TPoint;
  PBtn, SBtn: PTool;
  Index: Integer;
  TWidth, PWidth, SWidth, BWidth: Integer;
  PPart, SPart, Total: Real;
  XOfs: Integer;

  function FoundIt(Item: PTool): Boolean; far;
  begin
    FoundIt := Item^.HasCommand(Msg.wParam);
  end;

  procedure AllPart(Item: PTool); far;
  begin
    Total := Total + Item^.GetPart;
  end;

  procedure SetWidth(Item: PTool); far;
  begin
    Item^.CalculateWidth(BWidth, XOfs);
  end;

begin
  GetClientRect(HWindow, R);
  BWidth := R.Right - R.Left + 1;
  P := TPoint(Msg.LParam);
  PBtn := Buttons.FirstThat(@FoundIt);
  Index := Buttons.Indexof(PBtn);
  SBtn := Buttons.At(Succ(Index));
  PBtn^.GetRect(PR);
  if (P.X - PR.Left) < 10 then
    P.X := PR.Left + 10;
  SBtn^.GetRect(SR);
  if (SR.Right - P.X) < 10 then
    P.X := SR.Right - 10;
  TWidth := PBtn^.GetWidth + SBtn^.GetWidth;
  PWidth := P.X - PR.Left;
  if (PWidth <> 0) then
    PPart := PWidth / BWidth
  else
    PPart := 0;
  SWidth := TWidth - PWidth;
  if (SWidth <> 0) then
    SPart := SWidth / BWidth
  else
    SPart := 0;
  PBtn^.Resize(PPart);
  SBtn^.Resize(SPart);
  Total := 0;
  Buttons.ForEach(@AllPart);
  SPart := SPart - (Total - 1);
  SBtn^.Resize(SPart);
  XOfs := 0;
  Buttons.ForEach(@SetWidth);
  SendMessage(Parent^.HWindow, tm_SizingEnd, 0, 0);
end;

procedure TButtonBar.CheckTool(Command: Word);
var
  P: PTool;

  function FoundIt(P: PTool): Boolean; far;
  begin
    FoundIt := P^.HasCommand(Command);
  end;

  procedure UnCheck(Item: PTool); far;
  begin
    Item^.Check(False);
  end;

begin
  P := nil;
  P := Buttons.FirstThat(@FoundIt);
  if P <> nil then
  begin
    Buttons.ForEach(@UnCheck);
    P^.Check(True);
  end;
end;

{ ********** Mouse operation processes ********** }

procedure TButtonBar.WMLButtonDown(var Msg: TMessage);
var
  NCapture, SCapture: PTool;

  function IsHit(Item: PTool): Boolean; far;
  begin
    IsHit := Item^.HitTest(TPoint(Msg.LParam));
  end;

  function IsSizeHit(Item: PTool): Boolean; far;
  begin
    IsSizeHit := Item^.HitSize(TPoint(Msg.LParam));
  end;

begin
  NCapture := Buttons.FirstThat(@IsHit);
  SCapture := Buttons.FirstThat(@IsSizeHit);
  if (SCapture <> nil) and (Buttons.IndexOf(SCapture) <> Pred(ButtonsCount)) then
  begin
    Sizing := True;
    Capture := SCapture;
    Capture^.BeginSCapture(TPoint(Msg.LParam));
  end
  else
    if NCapture <> nil then
    begin
      Capture := NCapture;
      Capture^.BeginNCapture(TPoint(Msg.LParam));
    end;
end;

procedure TButtonBar.WMMouseMove(var Msg: TMessage);
var
  SB: PTool;

  function IsSizeHit(Item: PTool): Boolean; far;
  begin
    IsSizeHit := Item^.HitSize(TPoint(Msg.LParam));
  end;

begin
  if (Capture <> nil) then
    if Sizing then
      Capture^.ContinueSCapture(TPoint(Msg.LParam))
    else
      Capture^.ContinueNCapture(TPoint(Msg.LParam))
  else
  begin
    SB := Buttons.FirstThat(@IsSizeHit);
    if SB <> nil then
      begin
        if Buttons.IndexOf(SB) <> Pred(ButtonsCount) then
          SetCursor(SizCursor);
      end
    else
      SetCursor(ArrowCursor);
  end;
end;

procedure TButtonBar.WMLButtonUp(var Msg: TMessage);

  procedure UnCheck(Item: PTool); far;
  begin
    Item^.Check(False);
  end;

begin
  if (Capture <> nil) then
  begin
    if Sizing then
    begin
       if Capture^.EndSCapture(HWindow, TPoint(Msg.LParam)) then
       begin
         Sizing := False;
         Capture := nil;
       end;
    end
  else
    begin
       if Capture^.EndNCapture(Parent^.HWindow, TPoint(Msg.LParam)) then
       begin
         if Capture^.HitTest(TPoint(Msg.LParam)) then
         begin
           Buttons.ForEach(@UnCheck);
           Capture^.Check(True);
         end;
         Capture := nil;
       end;
    end;
  end;
end;

{ ********** TBarButton ********** }

constructor TBarButton.Init(AParent: PWindowsObject; ACommand: Word;
                            AName: PChar; APart: Real; AnAlign: Word; AColor: TColorRef);
begin
  inherited Init(AParent);
  CapDC := 0;
  BarColor := AColor;
  MemDC := 0;
  IsPressed := False;
  NCapturing := False;
  SCapturing := False;
  IsEnabled := True;
  IsChecked := False;
  Command := ACommand;
  Align := AnAlign;
  GetMem(Caption, StrLen(AName) + 1);
  StrCopy(Caption, AName);
  Part := APart;
  GlyphSize.Y := 19;
end;

destructor TBarButton.Done;
begin
  if NCapturing then
  begin
    DeleteDC(MemDC);
    ReleaseDC(Parent^.HWindow, CapDC);
    ReleaseCapture;
  end;
  if SCapturing then
  begin
    ReleaseCapture;
  end;
  FreeMem(Caption, StrLen(Caption) + 1);
  inherited Done;
end;

function TBarButton.HasCommand(ACommand: Word): Boolean;
begin
  HasCommand := (Command = ACommand);
end;

procedure TBarButton.Enable(State: Boolean);
begin
  if (IsEnabled <> State) and (Parent^.HWindow <> 0) then
    InvalidateRect(Parent^.HWindow, @R, False);
  IsEnabled := State;
end;

procedure TBarButton.Check(State: Boolean);
begin
  if (not State) and IsPressed then Exit;
  if (IsChecked <> State) and (Parent^.Hwindow <> 0) then
    InvalidateRect(Parent^.HWindow, @R, False);
  IsChecked := State;
  IsPressed := False;
end;

function TBarButton.GetWidth: Integer;
begin
  GetWidth := R.Right - R.Left;
end;

function TBarButton.GetHeight: Integer;
begin
  GetHeight := R.Bottom - R.Top;
end;

procedure TBarButton.GetRect(var AR: TRect);
begin
  Move(R, AR, SizeOf(TRect));
end;

function TBarButton.GetPart: Real;
begin
  GetPart := Part;
end;

procedure TBarButton.Resize(APart: Real);
begin
  Part := APart;
end;

procedure TBarButton.SetOrigin(X, Y: Integer);
begin
  SetRect(R, X, Y, X + GlyphSize.X, Y + GlyphSize.Y);
end;

function TBarButton.HitTest(P: TPoint): Boolean;
begin
  HitTest := Boolean(PtInRect(R, P));
end;

function TBarButton.HitSize(P: TPoint): Boolean;
var
  InActive: TRect;
begin
  Move(R, InActive, SizeOf(TRect));
  InflateRect(InActive, -2, 0);
  OffsetRect(InActive, -4, 0);
  HitSize := not Boolean(PtInRect(InActive, P)) and Boolean(PtInRect(R, P));
end;

procedure TBarButton.Paint(DC, AMemDC: HDC; var PS: TPaintStruct);
begin
  PaintState(DC, AMemDC);
end;

procedure TBarButton.PaintState(DC, AMemDC: HDC);
const
  RectDelta = 3;
var
  OldBrush: HBrush;
  OldPen: HPen;
  OldFont: HFont;
  Offset, OffsetX: Integer;
  TextR: TRect;
begin
  OldPen := SelectObject(DC, BlackPen);
  OldBrush := SelectObject(DC, GrayBrush);
  OldFont := SelectObject(DC, ButtonFont);
  With R do
  begin
    FillRect(DC, R, GrayBrush);
    Rectangle(DC, Left, Top - 1, Right + 1, Bottom + 1);
    if (not IsPressed) and (not IsChecked) then
    begin
      Offset := BorderWidth;
      SelectObject(DC, WhitePen);
      MoveTo(DC, Left + 1, Bottom - 1);
      LineTo(DC, Left + 1, Top);
      LineTo(DC, Right - 2, Top);
      SelectObject(DC, DarkGrayPen);
      MoveTo(DC, Right - 1, Top);
      LineTo(DC, Right - 1, Bottom - 2);
      LineTo(DC, Left + 1, Bottom - 2);
    end
    else
    begin
      Offset := BorderWidth + 1;
      SelectObject(DC, DarkGrayPen);
      MoveTo(DC, Left + 1, Bottom - 1);
      LineTo(DC, Left + 1, Top);
      LineTo(DC, Right, Top);
    end;
  end;

  SetBkMode(DC, Transparent);
  if IsEnabled then
    SetTextColor(DC, BarColor)
  else
    SetTextColor(DC, coDarkGray);

  Move(R, TextR, SizeOf(TRect));
  Inc(TextR.Left, RectDelta);
  Dec(TextR.Right, RectDelta);
  Inc(TextR.Top, Offset);
  case Align of
    DT_LEFT, DT_CENTER: Inc(TextR.Left, Offset + 2);
    DT_RIGHT: Dec(TextR.Right, (Offset*-1) + 3);
  end;
  DrawText(DC, Caption, StrLen(Caption), TextR, Align or DT_TOP);
  SelectObject(DC, OldBrush);
  SelectObject(DC, OldPen);
  SelectObject(DC, OldFont);
end;

procedure TBarButton.PressIn;
begin
  if (not IsPressed) and IsEnabled and (not IsChecked) then
  begin
    IsPressed := True;
    PaintState(CapDC, MemDC);
  end;
end;

procedure TBarButton.PressOut;
begin
  if IsPressed and (not IsChecked) then
  begin
    IsPressed := False;
    PaintState(CapDC, MemDC);
  end;
end;

procedure TBarButton.BeginNCapture(P: TPoint);
begin
  CapDC := GetDC(Parent^.HWindow);
  MemDC := CreateCompatibleDC(CapDC);
  IsPressed := False;
  NCapturing := True;
  SetCapture(Parent^.HWindow);
  if HitTest(P) then
    PressIn;
end;

procedure TBarButton.BeginSCapture(P: TPoint);
begin
  IsPressed := False;
  SCapturing := True;
  SendMessage(Parent^.Parent^.HWindow, tm_FirstColSize, 0, Longint(P));
  SetCapture(Parent^.HWindow);
end;

procedure TBarButton.ContinueNCapture(P: TPoint);
begin
  if HitTest(P) then
    PressIn
  else
    PressOut;
end;

procedure TBarButton.ContinueSCapture(P: TPoint);
begin
  { Draw Dotted line in CapDC }
  SendMessage(Parent^.Parent^.HWindow, tm_NewColSize, 0, Longint(P));
end;

{ The boolean function result of EndCapture indicates whether the tool button
  has released the mouse capture or not.  The Toolbar should not clear its
  capture field/state until the toolbutton says to.

  The SendTo parameter is the HWindow to notify that the tool button was clicked
  upon, if such is the case.  This code emulates a menu command message, but
  any message type could be used. }

function TBarButton.EndNCapture(SendTo: HWnd; P: TPoint): Boolean;
begin
  if HitTest(P) then
    if not IsChecked then PostMessage(SendTo, wm_Command, Command, 0);
  EndNCapture := True;
  ReleaseCapture;
  NCapturing := False;
  DeleteDC(MemDC);
  ReleaseDC(Parent^.HWindow, CapDC);
  MemDC := 0;
  CapDC := 0;
end;

function TBarButton.EndSCapture(SendTo: HWnd; P: TPoint): Boolean;
begin
  PostMessage(SendTo, tm_SizingEnd, Command, LongInt(P));
  EndSCapture := True;
  ReleaseCapture;
  NCapturing := False;
  MemDC := 0;
  CapDC := 0;
end;

procedure TBarButton.CalculateWidth(BarWidth: Word; var XOfs: Integer);
begin
  GlyphSize.X := Round((BarWidth*Part) + 1);
  if (BarWidth - (XOfs + GlyphSize.X)) < 0 then
    GlyphSize.X := BarWidth - XOfs - 1;
  SetRect(R, XOfs, 0, XOfs + GlyphSize.X, GlyphSize.Y);
  Inc(XOfs, GlyphSize.X);
end;

function TBarButton.IsToolChecked: Boolean;
begin
  IsToolChecked := IsChecked;
end;

function TBarButton.GetCommand: Word;
begin
  GetCommand := Command;
end;

{ Allocate unit wide resources }

procedure AllocateResources;
var
  LBrush: TLogBrush;
  lButtonFont: TLogFont;

begin
  { Allocate graying brush (used to disable buttons) }
  LBrush.lbStyle := bs_Pattern;
  Word(LBrush.lbHatch) := LoadBitMap(HInstance, 'GrayingBitmap');
  GrayingBrush := CreateBrushIndirect(LBrush);
  DeleteObject(Word(LBrush.lbHatch));

  { Allocate font for buttons captions }
  with lButtonFont do
  begin
    lfHeight        := 10;
    lfWidth         := 0;
    lfEscapement    := 0;
    lfOrientation   := 0;
    lfWeight        := fw_Regular;
    lfItalic        := 0;
    lfUnderline     := 0;
    lfStrikeOut     := 0;
    lfCharSet       := Default_CharSet;
    lfOutPrecision  := Out_Default_Precis;
    lfClipPrecision := Clip_Default_Precis;
    lfQuality       := Proof_Quality;
    lfPitchAndFamily:= Variable_Pitch or FF_Swiss;
    StrCopy(lfFaceName, 'MS Sans Serif');
  end;
  ButtonFont := CreateFontIndirect(lButtonFont);

  { Allocate drawing pens and brushes }
  GrayBrush := GetStockObject(LtGray_Brush);
  WhitePen := GetStockObject(White_Pen);
  BlackPen := GetStockObject(Black_Pen);
  DarkGrayPen := CreatePen(ps_Solid, 1, coDarkGray);
  DotPen := CreatePen(ps_Dot, 1, 0);

  { Allocate column size cursor }
  SizCursor := LoadCursor(HInstance, 'COLSIZE');
  ArrowCursor := LoadCursor(0, IDC_ARROW);
end;

{ Free allocated resources }

procedure DeallocateResources;
begin
  DeleteObject(GrayingBrush);
  DeleteObject(ButtonFont);
  DeleteObject(DarkGrayPen);
  DeleteObject(DotPen);
  DestroyCursor(SizCursor);
end;

var
  SaveExit: Pointer;

procedure ExitBtnBar; far;
begin
  DeallocateResources;
  ExitProc := SaveExit;
end;

begin
  SaveExit := ExitProc;
  ExitProc := @ExitBtnBar;
  AllocateResources;
end.
