{ == GR_COL2F.PAS =============================================================

  Test writeln in color within graphics. Shows that you CAN write color text
  in graphics, with the correct background!

  The "background" cannot be set via TextBackground, but must be drawn with
  graphics, such as Bar

  This program demonstrates all 16 VGA text colors against the 16 VGA colors
  as graphics backgrounds.  It shows the background RGB "combined value" (see
  below), the foreground RGB "combined value", and the resulting XOR combined
  value.

  The XorColConv procedure calculates the correct VGA color number ( Black:0
  to White:15) which results when a background and foreground color are XOR'd.
  Diagnostic prints within the procedure show the values being manipulated.

  If the background and foreground colors COR to a "standard" 16-color VGA
  combined value, TRUE is shown, while FALSE is shown if the "exception"
  look-up table has been used to find the output VGA color.

  Take a look at the procedure code and comments to tell you what's happening.

  Written by Jerry Rivers. Last modified: 04/16/93

  ============================================================================= }

PROGRAM Gr_Color;

  uses Graph, Crt, DosUtil;
  const
    ColNam : array[0..15] of string[12] = (
              'Black',    'Blue',         'Green',      'Cyan',
              'Red',      'Magenta',      'Brown',      'LtGray',
              'DarkGray', 'LtBlue',       'LtGreen',    'LtCyan',
              'LtRed',    'LtMagenta',    'Yellow',     'White' );

  var
    Ch              : char;
    ColorBack,
    ColorIn,
    ColorOut        : integer;                   { color parms for XorColConv }
    Colr            : text;
    Driver          : integer;
    GRreturn        : integer;
    I               : byte;
    Mode            : integer;
    Txt             : text;

{ == XorColConv ===============================================================

(*// Procedure: XorColConv

Purpose:  Calculates color resulting from background XOR foreground in graphics

Syntax:   PROCEDURE
          XorColConv( ColBkGnd, ColSet : integer; var ColResult : integer );

Parameters:

  ColBkGnd     Input background color: VGA 0 (Black) to 15 (White)
  ColSet       Input foreground color: VGA 0-15
  ColResult    Output resulting VGA color of ColSet XOR on ColBkGnd

Remarks:  RGB colors in decimal for the 16 "standard" VGA colors

          These numbers are calculated by this formula:

             ColorValue := ( Red * 65536 ) + ( Green * 256 ) + Blue

          example, "standard" LightBlue is Red = $15, Green = $15, Blue = $3F
          This "combines" to give $15153F = 1381695 decimal

          The default palette is:

           0; Red = $00; Green = $00; Blue = $00)  Black        =       0
           1; Red = $00; Green = $00; Blue = $2A)  Blue         =      42
           2; Red = $00; Green = $2A; Blue = $00)  Green        =   10752
           3; Red = $00; Green = $2A; Blue = $2A)  Cyan         =   10794
           4; Red = $2A; Green = $00; Blue = $00)  Red          = 2752512
           5; Red = $2A; Green = $00; Blue = $2A)  Magenta      = 2752554
           6; Red = $2A; Green = $15; Blue = $00)  Brown        = 2757888
           7; Red = $2A; Green = $2A; Blue = $2A)  LightGray    = 2763306
           8; Red = $15; Green = $15; Blue = $15)  DarkGray     = 1381653
           9; Red = $15; Green = $15; Blue = $3F)  LightBlue    = 1381695
          10; Red = $15; Green = $3F; Blue = $15)  LightGreen   = 1392405
          11; Red = $15; Green = $3F; Blue = $3F)  LightCyan    = 1392447
          12; Red = $3F; Green = $15; Blue = $15)  LightRed     = 4134165
          13; Red = $3F; Green = $15; Blue = $3F)  LightMagenta = 4134207
          14; Red = $3F; Green = $3F; Blue = $15)  Yellow       = 4144917
          15; Red = $3F; Green = $3F; Blue = $3F)  White        = 4144959

          This version of XorColConv also writes two text files to the current
          directory: 'coltxt.txt', which is the same as the diagnostic prints
          to the screen and 'col.txt' which is just the raw "exception" values
          before the duplicates are removed.

          Duplicates are removed and the exception tables are built by another
          utility program: RECOL3.PAS

Refer
Globals:  ColNam[]
          Txt
          Colr

Modify
Globals:  None

See also:

Example:

    (*
       Set the input parameters
    *)
    ColorBack := White;
    ColorIn   := LightRed;
    (*
       Create a color graphics "background"
    *)
    SetFillStyle( SolidFill, ColorBack );
    Bar( 0, 0, 639, 479 );
    (*
       Convert White to its resulting XOR output
       In this case, White XOR LightRed = Cyan

       Cyan is the inverse of LightRed so it creates true LightRed against
       a white background when used by TextColor.  the "+ 128" forces bit 7
       of TextAttr "on" which causes BIOS writes in graphics to be XOR'd
       unto the screen. Voila! You get LightRed text on White!
    *)
    XorColConv( ColorBack, ColorIn, ColorOut );
    TextColor( ColorOut + 128 );
    writeln( ColNam[ColorIn]:12, ColNam[ColorOut]:12 );
    (*
       Will print: LtRed  Cyan
    *)
    writeln(' This should be ', ColNam[ColorIn] );
    (*
       Will print: This should be LtRed
    *)

\\*)

  Written by Jerry Rivers. Last modified: 04/16/93

  ============================================================================= }

PROCEDURE XorColConv( ColBkGnd, ColSet : integer; var ColResult : integer );
  const
    ColVal : array[0..15] of longint = (
                    0,         42,          10752,        10794,
              2752512,    2752554,        2757888,      2763306,
              1381653,    1381695,        1392405,      1392447,
              4134168,    4134207,        4144917,      4144959 );
  var
    Color1, Color2, Color3 : longint;
    I                      : byte;
    OK                     : boolean;
    RGB                    : array[1..30] of longint;        { RGB exceptions }
    Col                    : array[1..30] of integer;  { corresponding colors }
  begin
    {
      Initialize the Color exception tables - developed visually

      My tests show there are 16 "exact" color matches (to standard 16-color
      VGA values) plus 29 "exceptions". These exceptions result in nearly but
      not quite exact visual representations of the standard colors. Exception
      values can be mapped to a standard color which looks OK to the eye.
    }

    RGB[ 1] := 2757930;  Col[ 1] :=  7;
    RGB[ 2] := 2763264;  Col[ 2] :=  6;
    RGB[ 3] := 4134194;  Col[ 3] := 13;
    RGB[ 4] := 4134165;  Col[ 4] := 12;
    RGB[ 5] := 2768640;  Col[ 5] :=  4;
    RGB[ 6] := 4144920;  Col[ 6] := 14;
    RGB[ 7] := 2768682;  Col[ 7] :=  5;
    RGB[ 8] := 4144946;  Col[ 8] := 15;
    RGB[ 9] :=    5376;  Col[ 9] :=  2;
    RGB[10] := 1381656;  Col[10] :=  8;
    RGB[11] :=    5418;  Col[11] :=  3;
    RGB[12] := 1381682;  Col[12] :=  9;
    RGB[13] :=   16170;  Col[13] :=  1;
    RGB[14] := 4128789;  Col[14] := 14;
    RGB[15] := 4128831;  Col[15] := 15;
    RGB[16] := 4139541;  Col[16] := 12;
    RGB[17] := 4139583;  Col[17] := 13;
    RGB[18] := 1376280;  Col[18] := 10;
    RGB[19] := 1376319;  Col[19] := 11;
    RGB[20] := 1387029;  Col[20] :=  8;
    RGB[21] := 1387071;  Col[21] :=  9;
    RGB[22] := 1392434;  Col[22] := 11;
    RGB[23] := 2752525;  Col[23] :=  4;
    RGB[24] := 2752551;  Col[24] :=  5;
    RGB[25] := 2763277;  Col[25] :=  6;
    RGB[26] := 2763303;  Col[26] :=  7;
    RGB[27] :=      39;  Col[27] :=  1;
    RGB[28] :=   10765;  Col[28] :=  2;
    RGB[29] :=   10791;  Col[29] :=  3;
    {
      Get long integer RGB color numbers for background and "desired" color
    }
    Color1 := ColVal[ ColBkGnd ];
    Color2 := ColVal[ ColSet ];
    Color3 := Color1 XOR Color2;
    {
      If the XOR color matches a "standard" VGA color, you're done!
    }
    I := 0; OK := false;
    repeat
      if Color3 = ColVal[ I ] then
        begin
          ColResult := I;                 { return corresponding color number }
          OK := true;                                     { signal OK to quit }
          write( Txt, OK:5);
          write( Txt, '**' );
          write( OK:5);
          write( '**' );
        end;
        inc( I );
    until ( I > White );
    {
      If the XOR color isn't "standard", look at the "exception" list

      This list was built by checking all the Color3's that aren't
      "standard" and visually identifying the matching color

      A 29-long look-up table may not be the most efficient, but it works!
    }
    if not OK then begin
        I := 1;
        repeat
          if Color3 = RGB[ I ] then
            begin
              write( Txt, OK );
              write( OK );
              ColResult := Col[ I ];      { return corresponding color number }
              OK := true;                                 { signal OK to quit }
              write( Txt, I:2 );
              write( I:2 );
            end;
          inc( I ) ;
        until OK or ( I > 29 );
      end;
    {
      If the bell rings, a color slipped thru the exception table;
      This should never happen
    }
    if not OK then writeln( chr(7) );
    {
      Set the resulting color the the rest of the diagnostic prints
    }
    TextColor( ColSet + 128 );
    write( (I-1):4, Color2:8, Color1:8, Color3:8, ColResult:3 );
    writeln( ColNam[ ColorIn ]:10, ' ^ ', ColNam[ ColorBack ],
                ' = ', ColNam[ ColResult ] );

    write( Txt, (I-1):4, Color2:8, Color1:8, Color3:8, ColResult:3 );
    writeln( Txt, ColNam[ ColorIn ]:10, ' XOR ', ColNam[ ColorBack ],
                ' = ', ColNam[ ColResult ] );

    writeln( Colr, ColorBack:2, color3:8, ColResult:3 );

  end; { XorColConv }

  BEGIN  { --- Main Program --- }
    {
      This program assumes you are running standard VGA 16-color graphics
      If you're not, an error will result and the program will stop
    }
    Driver := VGA;
    Mode   := VGAhi;

    InitGraph(Driver, Mode, '');

    GRreturn := GraphResult;
    if GRreturn <> 0 then begin
        writeln( 'Couldn''t start Scene-may be graphics or video problem' );
        writeln( GraphErrorMsg( GRreturn ) );
        halt;
      end;
    {
      Turn direct video off so standard writeln works in graphics
    }
    DirectVideo := false;
    SetRGBpalette( Brown, $22, $15, $00 );  { adjust VGA Brown to look "browner"
                                              and less like VGA Red }
    {
      Take care of the output text files!
    }
    Assign( Txt, 'coltxt.txt' );
    assign( Colr, 'col.txt' );
    rewrite( Txt );
    rewrite( Colr );
    {
      Do the color matching for every combination of background and foreground
      VGA colors
    }
    for ColorBack := Black to white do
      begin

      SetFillStyle( SolidFill, ColorBack );
      Bar( 0, 0, 639, 479 );

      TextColor( Black + 128 );        gotoXY( 72, 5 );  writeln( 'Black' );
      TextColor( Blue + 128 );         gotoXY( 72, 6 );  writeln( 'Blue' );
      TextColor( Green + 128 );        gotoXY( 72, 7 );  writeln( 'Green' );
      TextColor( Cyan + 128 );         gotoXY( 72, 8 );  writeln( 'Cyan' );
      TextColor( Red + 128 );          gotoXY( 72, 9 );  writeln( 'Red' );
      TextColor( Magenta + 128 );      gotoXY( 72, 10 ); writeln( 'Magenta' );
      TextColor( Brown + 128 );        gotoXY( 72, 11 ); writeln( 'Brown' );
      TextColor( Lightgray + 128 );    gotoXY( 72, 12 ); writeln( 'LtGray' );
      TextColor( DarkGray + 128 );     gotoXY( 72, 13 ); writeln( 'DkGray' );
      TextColor( Lightblue + 128 );    gotoXY( 72, 14 ); writeln( 'Ltblue' );
      TextColor( Lightgreen + 128 );   gotoXY( 72, 15 ); writeln( 'Ltgreen' );
      TextColor( LightCyan + 128 );    gotoXY( 72, 16 ); writeln( 'Ltcyan' );
      TextColor( Lightred + 128 );     gotoXY( 72, 17 ); writeln( 'Ltred' );
      TextColor( Lightmagenta + 128 ); gotoXY( 72, 18 ); writeln( 'Ltmagenta' );
      TextColor( Yellow + 128 );       gotoXY( 72, 19 ); writeln( 'Yellow' );
      TextColor( White + 128 );        gotoXY( 72, 20 ); writeln( 'White' );

      gotoXY( 1, 5 );
      for ColorIn := Black to White do
        XorColConv( ColorBack, ColorIn, ColorOut );

      writeln;
      write( 'Press any key to continue, ESC to quit now ' );
      Ch := ReadKey;
      if Ch = #27 then halt;

    end; { for ColorBack }

    close( Txt );
    close( Colr );

  END.
