program DoDIB;

uses
  WinTypes, WinProcs, WinDOS, Strings,
  OWindows, ODialogs, OPrinter, OStdDlgs;

{ Menu Command constants }
const
  cm_Print = 100;
  cm_Open = 101;

{ Constants used for loading bitmaps into memory }
const
  OneIO = 32768;  { Number bytes handled per huge IO operation }
  BMType = $4D42;  { = 'BM' }

{ Types used when loading the bitmap into memory }
type
  IOFunction = function(FP: integer; Buf: PChar; Size: Integer): Word;

  PtrRec = record
    Lo, Hi: Word
  end;

{ Application's MainWindow Object }
type
  PMainWin = ^TMainWin;
  TMainWin = object(TWindow)
    Bitmap: HBitmap;
    Bits: Pointer;
    BitsInfo: PBitmapInfo;
    ScanLines: Integer;
    HMem: THandle;
    hPal: HPalette;
    Printer: PPrinter;
    Filename: array[0..fsPathName] of Char;
    constructor Init(AParent: PWindowsObject; ATitle: PChar);
    destructor Done; virtual;
    procedure OpenBMP; virtual cm_First + cm_Open;
    procedure Paint(PaintDC: HDC;
      var PaintStruct: TPaintStruct); virtual;
    procedure QueryNewPalette(var Msg: TMessage);
      virtual wm_First + wm_QueryNewPalette;
    procedure PaletteChanged(var Msg: TMessage);
      virtual wm_First + wm_PaletteChanged;
    procedure Print(var Msg: TMessage);
      virtual cm_First + cm_Print;
  end;

{ Application Object }
  PMainApp = ^TMainApp;
  TMainApp = object(TApplication)
    procedure InitMainWindow; virtual;
  end;

{******* Input Functions for the Bitmap ***********************}

{ Selector Increment Factor }
procedure AHIncr; far; external 'KERNEL' index 114;

function HugeIO(IOFunc: IOFunction; F: Integer; P: Pointer;
  Size: Longint): Word;
var
  L, N: Longint;	       	 { L maintains total bytes }
begin   			 { N maintains bytes for
                                   current pass }
  HugeIO := 1;
  L := 0;
  while L < Size do
  begin
    N := Size - L;
    if N > OneIO then N := OneIO;
    if IOFunc(F,
    { Compute the segment and offset reached.
      The Hi word of P contains the initial segment.
      Think of the following as performing arithmetic
        modulo segment-size, since the size of a segment
        fills one word:
      The Hi word of L contains the number of segments crossed
        (the size of one segment fills the Lo word, so Hi word
        will roll over as segments are filled).
        Multiply by Ofs(AHIncr) to obtain the number used to
        indicate this number of segments.
      The Lo word of L contains the number of bytes already
        passed in the present segment.
     }
	       Ptr(PtrRec(P).Hi + PtrRec(L).Hi * Ofs(AHIncr),
               PtrRec(L).Lo),
               Integer(N))     { Guaranteed to be in Integer
                                 range }
       <> N then
    begin
      HugeIO := 0;
      Exit; { abnormal termination }
    end;
    Inc(L, N);
  end;
end;

function _LFileSize(F : integer) : longint;
{- an equivalent to TP's FileSize() function }
var
  CurPos : longint;
begin
  CurPos := _llseek(F,0,1);
  _LFileSize := _llseek(F,0,2);
  _llseek(F,CurPos,0);
end;

function LoadBitmapFile(FileName: PChar; var HPal: HPalette): HBitmap;
{ Loads a bitmap and its palette. }
var
  F: Integer;			{ File Handle for Windows file functions }
  H: THandle;			{ Handle to memory for bitmap }
  DC: HDC;			{ Drawing context for application }
  Size, N: Longint;		{ Size of bitmap, Size of color spec }
  P: PBitmapInfo;		{ Windows bitmap format info header }
  Header: TBitmapFileHeader;    { Bitmap file header }

  NumColors: Word;              { NumColors involved with the palette }
  I: Word;                      { Loop counter }
  Pal: PLogPalette;             { Logical Palette pointer }
  OldPal: HPalette;             { Place holder palette handle }
begin
  LoadBitmapFile := 0;
  F := _LOpen(FileName, of_Read);
  if F = -1 then Exit;

  { read in the Bitmap file header }
  if (_LRead(F, @Header, SizeOf(Header)) <> SizeOf(Header)) or
    (Header.bfType <> BMType) then
  begin
    _LClose(F);
    Exit;
  end;

  { read the rest of the file }
  Size := _LFileSize(F) - SizeOf(TBitmapFileHeader);
  H := GlobalAlloc(gmem_Moveable, Size);	{ Allocate the memory }
  if H = 0 then
  begin
    _LClose(F);
    Exit;
  end;

  P := GlobalLock(H);				{ Lock it down }

  Pal := nil;
  if (HugeIO(_LRead, F, P, Size) <> 0) and
    (P^.bmiHeader.biSize = SizeOf(TBitmapInfoHeader)) then
  begin
    { Compute the offset from the beginning of P^ }
    { where the actual image begins }
    N := Header.bfOffBits - SizeOf(TBitmapFileHeader);


    {!! Create the Palette !!}
    NumColors := P^.bmiHeader.biClrUsed;
    if NumColors <> 0 then
      begin
        GetMem(Pal, SizeOf(TLogPalette) + (NumColors - 1) * SizeOf(TPaletteEntry));
        Pal^.palVersion := $300;
        Pal^.palNumEntries := NumColors;
        for I := 0 to NumColors-1 do
          with P^.bmiColors[I], Pal^.palPalEntry[I] do
            begin
              peRed := rgbRed;
              peGreen := rgbGreen;
              peBlue := rgbBlue;
              peFlags := 0;
            end;
      end;
    if NumColors <> 0 then
      begin
        HPal := CreatePalette(Pal^);
        FreeMem(Pal, SizeOf(TLogPalette) + (NumColors - 1) * SizeOf(TPaletteEntry));
      end
    else
      HPal := GetStockObject(DEFAULT_PALETTE);

    DC := GetDC(0);

    {!! Select and Realize the Palette from the Bitmap !!}
    OldPal := SelectPalette(DC, HPal, false);
    RealizePalette(DC);

    { actually create the Bitmap }
    LoadBitmapFile := CreateDIBitmap(DC, P^.bmiHeader,
      cbm_Init, Ptr(PtrRec(P).Hi,N),P^, dib_RGB_Colors);

    {!! Restore the original bitmap !!}
    SelectPalette(DC, OldPal, False);

    { clean up }
    ReleaseDC(0, DC);
  end;

  GlobalUnlock(H);
  GlobalFree(H);
  _LClose(F);
end;

{*************** MainWindow Routines **************************}

constructor TMainWin.Init(AParent: PWindowsObject; ATitle: PChar);
{ Create the menu, initialize the printer interface and clear
  the Filename of the bitmap }
begin
  inherited Init(AParent, ATitle);
  Attr.Menu := CreateMenu;
  AppendMenu(Attr.Menu, mf_String, cm_Open, '&Open');
  AppendMenu(Attr.Menu, mf_String, cm_Print, '&Print');
  Printer := New(PPrinter, Init);
  Filename[0] := #0;
end;

destructor TMainWin.Done;
{ Shut down the Window }
begin
  inherited Done;
  GlobalFree(HMem);
  DeleteObject(HPal);
  Dispose(Printer, Done);
end;

procedure TMainWin.OpenBMP;
  { Takes the size in bits and returns the (aligned) size in bytes.
    Bitmap data format requires word alignment.
  }
  function bmAlignDouble(Size: Longint): Longint;
  begin
    bmAlignDouble := (Size + 31) div 32 * 4;
  end;
var
  DC: HDC;
  BM: TBitmap;
  BitCount: Word;
  ColorSize, DataSize: Longint;
var
  LogPal : PLogPalette;
  i : integer;
  PalSize : integer;
  sz : Longint;
  OldPal: THandle;
  Dlg: PDialog;
begin
  inherited SetUpWindow;

  { Get the filename from the user }
  StrCopy(Filename, '*.bmp');
  Dlg := New(PFileDialog, Init(@Self, PChar(sd_FileOpen), FileName));
  if Application^.ExecDialog(Dlg) <> ID_CANCEL then
    begin

      { Load bitmap }
      Bitmap := LoadBitMapFile(Filename, HPal);

      { Get the bitmap information }
      GetObject(Bitmap, SizeOf(TBitmap), @BM);

      DC := GetDC(HWindow);

      { Allocate memory for BitsInfo }
      sz := SizeOf(TBitMapInfoHeader) + 256 * SizeOf(TRGBQuad);
      GetMem(BitsInfo, sz);
      FillChar(BitsInfo^, sz, 0);

      with BitsInfo^.bmiHeader do
        begin
          biSize := SizeOf(BitsInfo^.bmiHeader);
          biWidth := BM.bmWidth;
          biHeight := BM.bmHeight;
          biPlanes := 1;
          biBitCount := 8;
          biCompression := BI_RGB;
        end;

      { Use the palette coming with the bitmap }
      OldPal := SelectPalette(DC, HPal, False);
      RealizePalette(HPal);

      { Allocate the memory for the bitmap's bits }
      ScanLines := GetDIBits(DC, Bitmap, 0, $FFFF, nil, BitsInfo^, DIB_RGB_COLORS);
      HMem := GlobalAlloc(gmem_Fixed, BitsInfo^.bmiHeader.biSizeImage);
      Bits := GlobalLock(HMem);

      { Obtain the bits for the bitmap's bits }
      ScanLines := GetDIBits(DC, Bitmap, 0, ScanLines, Bits, BitsInfo^, DIB_RGB_COLORS);

      { Restore the palette }
      SelectPalette(DC, OldPal, False);
      InvalidateRect(HWindow, nil, True);
    end
  else
    Filename[0] := #0;
end;

procedure TMainWin.Paint(PaintDC: HDC; var PaintStruct: TPaintStruct);
var
  OldPal: HPalette;
  I : word;
  Factor: Real;
  Width, Height: Word;
  Stuff: TRect;
begin
  { If a file has been selected show it's bitmap }
  if Filename[0] <> #0 then
    begin

      { Realize its palette }
      OldPal := SelectPalette(PaintDC, hPal, False);
      I := RealizePalette(PaintDC);

      { Get the clipping rectangle for the output device }
      GetClipBox(PaintDC, Stuff);

      { Display the bitmap scaled for the device }
      StretchDIBits(PaintDC,
        Stuff.Left, Stuff.Top,
        Stuff.Right - Stuff.Left, Stuff.Bottom - Stuff.Top,
        0, 0, BitsInfo^.bmiHeader.biWidth, BitsInfo^.bmiHeader.biHeight,
        Bits, BitsInfo^, DIB_RGB_COLORS, SRCCOPY);

      { Reset the palette }
      SelectPalette(PaintDC, OldPal, False);
    end;
end;

procedure TMainWin.QueryNewPalette(var Msg: TMessage);
{ Selects the bitmap's palette when called for }
var
  DC: HDC;
  I: Word;
  OldPal: THandle;
begin
  DC := GetDC(HWindow);
  OldPal := SelectPalette(DC, HPal, False);
  I := RealizePalette(DC);
  SelectPalette(DC, OldPal, False);
  ReleaseDC(HWindow, DC);

  if I <> 0 then
    begin
      InvalidateRect(HWindow, nil, True);
      Msg.Result := 1;
    end
  else
    Msg.Result := 0;
end;

procedure TMainWin.PaletteChanged(var Msg: TMessage);
{ Realizes the bitmap's palette when other app's change
  the palette }
var
  OldPal: THandle;
  I: Word;
  DC: HDC;
begin
  if Msg.WParam <> HWindow then
    begin
      DC := GetDC(HWindow);
      OldPal := SelectPalette(DC, HPal, False);
      I := RealizePalette(DC);
      if I <> 0 then
        UpdateColors(DC)
      else
        InvalidateRect(HWindow, nil, True);
      SelectPalette(DC, OldPal, False);
      ReleaseDC(HWindow, DC);
    end;
end;

procedure TMainWin.Print(var Msg: TMessage);
{ Uses the MainWindow's Paint method for printing }
var P: PPrintOut;
begin
  P := New(PWindowPrintOut, Init('DoDIB', @Self));
  Printer^.Print(@Self, P);
  Dispose(P, Done);
end;

procedure TMainApp.InitMainWindow;
begin
  MainWindow := New(PMainWin, Init(nil, 'DoDIB'));
end;

var
  MainApp: TMainApp;

begin
  MainApp.Init('DoDIB');
  MainApp.Run;
  MainApp.Done
end.
