UNIT Dates;

INTERFACE

  { These routines all assume that the year (y, y1) value is supplied in a }
  { form that includes the century (i.e., in YYYY form).  No checking is   }
  { performed to ensure that a month (m, m1) value is in the range 1..12   }
  { or that a day (d, d1) value is in the range 1..28,29,30,31.  The       }
  { FUNCTION ValidDate may be used to check for valid month and day        }
  { parameters. ValidDate uses the "Da2" routines described below.         }
  { The routines which have "Da2" as part of their name use algorithms     }
  { that do not rely on the Gregorian calendar (however implemented) and   }
  { are purported to be valid for dates BC as well as AD.  N.B. in using   }
  { these routines that there is no Year 0 : 1 BC was followed by AD 1.    }
  
  { Because the Gregorian calendar was not implemented in all countries at }
  { the same time, these routines are not guaranteed to be valid for all   }
  { dates.  The "Da2" routines implemented herein assume that the Gregorian}
  { calendar was adopted on 1582 10 15.  The real usefulness of these      }
  { routines is that they will not fail on December 31, 1999 - as will many}
  { algorithms used in MIS programs implemented on mainframes.             }

  { The routines are NOT highly optimized - I have tried to maintain the   }
  { style of the algorithms presented in the sources I indicate. Any       }
  { suggestions for algorithmic or code improvements will be gratefully    }
  { accepted.  This implementation is in the public domain - no copyright  }
  { is claimed.  No warranty either express or implied is given as to the  }
  { correctness of the algorithms or their implementation.                 }

  { Author: Charles B. Chapman, London, Ontario, Canada [74370,516]        }

  FUNCTION IsLeap (y : WORD) : BOOLEAN;
  FUNCTION ValidDate (y : INTEGER; m, d : WORD) : BOOLEAN;
  FUNCTION DayOfYear (y, m, d : WORD) : WORD;
  FUNCTION JulianDay (y, m, d : WORD) : LONGINT;
  FUNCTION JulianDa2 (y : INTEGER; m, d : WORD) : LONGINT;
  FUNCTION DayOfWeek (y, m, d : WORD) : WORD;
  FUNCTION Da2OfWeek (y : INTEGER; m, d : WORD) : WORD;
  FUNCTION TimeStr (h, m, s, c : WORD) : STRING;
  FUNCTION SIDateStr (y, m, d : WORD; SLen : BYTE; FillCh : CHAR) : STRING;
  FUNCTION Secs100 (h, m, s, c : WORD) : DOUBLE;
  PROCEDURE DayOfYearToDate (nd, y : WORD; VAR m, d : WORD);
  PROCEDURE JulianDayToDate (nd : LONGINT; VAR y, m, d : WORD);
  PROCEDURE JulianDa2ToDate (nd : LONGINT; VAR y : INTEGER; VAR m, d : WORD);
  PROCEDURE DateOfEaster (y : WORD; VAR m, d : WORD);
  PROCEDURE AddDays (y, m, d : WORD; plus : LONGINT; VAR y1, m1, d1 : WORD);

IMPLEMENTATION
  USES
    Dos;
  VAR
    t1, t2, t3, t4, t5 : LONGINT;

{==========================================================================}

  FUNCTION IsLeap (y : WORD) : BOOLEAN;

  { Returns TRUE if <y> is a leap-year                                     }

  BEGIN
    IF y MOD 4 <> 0 THEN
      IsLeap := FALSE
    ELSE
      IF y MOD 100 = 0 THEN
        IF y MOD 400 = 0 THEN
          IsLeap := TRUE
        ELSE
          IsLeap := FALSE
      ELSE
        IsLeap := TRUE
  END;  { IsLeap }

{==========================================================================}

  FUNCTION DayOfYear (y, m, d : WORD) : WORD;

  { function IDAY from remark on CACM Algorithm 398                        }
  { Computes day of the year for a given calendar date                     }
  { GIVEN:   y - year                                                      }
  {          m - month                                                     }
  {          d - day                                                       }
  { RETURNS: day-of-the-year (1..366, given valid input)                   }

  BEGIN
    t2 := y;
    t3 := m;
    t4 := d;
    t1 := (t3 + 10) DIV 13;
    DayOfYear :=  3055 * (t3 + 2) DIV 100 - t1 * 2 - 91 +
                  (1 - (t2 - t2 DIV 4 * 4 + 3) DIV 4 +
                  (t2 - t2 DIV 100 * 100 + 99) DIV 100 -
                  (t2 - t2 DIV 400 * 400 + 399) DIV 400) * t1 + t4
  END;  { DayOfYear }

{==========================================================================}

  FUNCTION JulianDay (y, m, d : WORD) : LONGINT;

  { procedure JDAY from CACM Alorithm 199                                  }
  { Computes Julian day number for any Gregorian Calendar date             }
  { GIVEN:   y - year                                                      }
  {          m - month                                                     }
  {          d - day                                                       }
  { RETURNS: Julian day number (astronomically, for the day                }
  {          beginning at noon on the given date.                          }

  BEGIN
    IF m > 2 THEN
      BEGIN
        t1 := m - 3;
        t2 := y
      END
    ELSE
      BEGIN
        t1 := m + 9;
        t2 := y - 1
      END;
    t3 := t2 DIV 100;
    t4 := t2 MOD 100;
    t5 := d;
    JulianDay := (146097 * t3) DIV 4 + (1461 * t4) DIV 4 +
                 (153 * t1 + 2) DIV 5 + t5 + 1721119
  END;  { JulianDay }

{==========================================================================}

  FUNCTION JulianDa2 (y : INTEGER; m, d : WORD) : LONGINT;

  { function JULDAY from Numerical Recipes (ISBN 0521308119)               }
  { Computes Julian day number for any date                                }
  { GIVEN:   y - year (-ve = B.C., +ve = A.D.)                             }
  {          m - month                                                     }
  {          d - day                                                       }
  { RETURNS: Julian day number (astronomically, for the day                }
  {          beginning at noon on the given date.                          }

  CONST
    GReg : LONGINT = 15 + 31 * (10 + 12 * 1582);
  BEGIN
    IF y = 0 THEN        {  NO such year !!!  }
      JulianDa2 := -1
    ELSE
      BEGIN
        IF y < 0 THEN
          Inc (y);
        IF m > 2 THEN
          BEGIN
            t1 := y;
            t2 := m + 1
          END
        ELSE
          BEGIN
            t1 := y - 1;
            t2 := m + 13
          END;
        t3 := Trunc (365.25 * t1) + Trunc (30.6001 * t2) + d + 1720995;
        IF d + 31 * (m + 12 * y) >= GReg THEN
          BEGIN
            t4 := Trunc (0.01 * y);
            t3 := t3 + 2 - t4 + Trunc (0.25 * t4)
          END;
        JulianDa2 := t3
      END
  END;  { JulianDa2 }

{==========================================================================}
  
  PROCEDURE DayOfYearToDate (nd, y : WORD; VAR m, d : WORD);
                                                         
  { procedure CALENDAR from CACM Algorithm 398                             }
  { Computes month and day from given year and day of the year             }
  { GIVEN:   nd - day-of-the-year (1..366)                                 }
  {          y - year                                                      }
  { RETURNS: m - month                                                     }
  {          d - day                                                       }

  BEGIN
    IF nd < 366 THEN
      BEGIN
        t5 := nd;
        IF y MOD 4 = 0 THEN
          t1 := 1
        ELSE
          t1 := 0;
        IF (y MOD 400 = 0) OR (y MOD 100 <> 0) THEN
          t2 := t1
        ELSE
          t2 := 0;
        t1 := 0;
        IF t5 > t2 + 59 THEN
          t1 := 2 - t2;
        t3 := t5 + t1;
        t4 := ((t3 + 91) * 100) DIV 3055;
        d := (t3 + 91) - (t4 * 3055) DIV 100;
        m := t4 - 2
      END
    ELSE
      BEGIN
        d := 0;
        m := 0
      END
  END;  { DayOfYearToDate }

{==========================================================================}

  PROCEDURE JulianDayToDate (nd : LONGINT; VAR y, m, d : WORD);

  { procedure JDATE from CACM Algorithm 199                                }
  { Computes calendar date from a given Julian day number for any          }
  { valid Gregorian calendar date                                          }
  { GIVEN:   nd - Julian day number (2440000 --> 1968 5 23)                }
  { RETURNS: y - year                                                      }
  {          m - month                                                     }
  {          d - day                                                       }

  BEGIN
    t1 := nd - 1721119;
    t5 := (4 * t1 - 1) DIV 146097;
    t1 := (4 * t1 - 1) MOD 146097;
    t2 := t1 DIV 4;
    t1 := (4 * t2 + 3) DIV 1461;
    t2 := (4 * t2 + 3) MOD 1461;
    t2 := (t2 + 4) DIV 4;
    m := (5 * t2 - 3) DIV 153;
    t2 := (5 * t2 - 3) MOD 153;
    d := (t2 + 5) DIV 5;
    y := 100 * t5 + t1;
    IF m < 10 THEN
      m := m + 3
    ELSE
      BEGIN
        m := m - 9;
        y := y + 1
      END
  END;  { JulianDayToDate }

{==========================================================================}

  PROCEDURE JulianDa2ToDate (nd : LONGINT; VAR y : INTEGER; VAR m, d : WORD);

  { procedure CALDAT from Numerical Recipes (ISBN 0521308119)              }
  { Computes calendar date from a given Julian day number for any          }
  { calendar date                                                          }
  { GIVEN:   nd - Julian day number (2440000 --> 1968 5 23)                }
  { RETURNS: y - year (-ve = B.C.)                                         }
  {          m - month                                                     }
  {          d - day                                                       }


  CONST
    GReg : LONGINT = 2299161;
  BEGIN
    IF nd >= GReg THEN
      BEGIN
        t3 := Trunc (((nd - 1867216) - 0.25) / 36524.25);
        t1 := nd + 1 + t3 - Trunc (0.25 * t3)
      END
    ELSE
      t1 := nd;
    t2 := t1 + 1524;
    t3 := Trunc (6680.0 + ((t2 - 2439870) - 122.1) / 365.25);
    t4 := 365 * t3 + Trunc (0.25 * t3);
    t5 := Trunc ((t2 - t4) / 30.6001);
    d := t2 - t4 - Trunc (30.6001 * t5);
    m := t5 - 1;
    IF m > 12 THEN
      m := m - 12;
    y := t3 - 4715;
    IF m > 2 THEN
      Dec (y);
    IF y <= 0 THEN
      Dec (y)
  END;  { JulianDa2ToDate }

{==========================================================================}

  PROCEDURE DateOfEaster (y : WORD; VAR m, d : WORD);

  { Algorithm "E" from Knuth's "Art of Computer Programming", vol. 1       }
  { Computes date of Easter for any year in the Gregorian calendar         }
  { The variables in double angle brackets are the variable names          }
  { used by Knuth corresponding to the temporaries I have used             }
  { GIVEN:   y - year                                                      }
  { RETURNS: m - month of Easter (3 or 4)                                  }
  {          d - day of Easter                                             }

  VAR
    E, N : LONGINT;
  BEGIN
  { Golden number of the year in Metonic cycle  <<G>> }
    t1 := y MOD 19 + 1;
  { Century <<C>> }
    t2 := y DIV 100 + 1;
  { Corrections: }
  { <t3> is the no. of years in which leap-year was dropped in }
  { order to keep step with the sun  <<X>> }
  { <t4> is a special correction to synchronize Easter with the }
  { moon's orbit  <<Z>>. }
    t3 := (3 * t2) DIV 4 - 12;
    t4 := (8 * t2 + 5) DIV 25 - 5;
  { <t5> Find Sunday  <<D>> }
    t5 := (5 * y) DIV 4 - t3 - 10;
  { Set Epact <<E>> }
    E := (11 * t1 + 20 + t4 - t3) MOD 30;
    IF E < 0 THEN
      E := E + 30;
    IF ((E = 25) AND (t1 > 11)) OR (E = 24) THEN
      E := E + 1;
  { Find full moon - the <<N>>th of MARCH is a "calendar" full moon }
    N := 44 - E;
    IF N < 21 THEN
      N := N + 30;
  { Advance to Sunday }
    N := N + 7 - ((t5 + N) MOD 7);
  { Get Month and Day }
    IF N > 31 THEN
      BEGIN
        m := 4;
        d := N - 31
      END
    ELSE
      BEGIN
        m := 3;
        d := N
      END
  END; { DateOfEaster }

{==========================================================================}

  FUNCTION SIDateStr (y, m, d : WORD; SLen : BYTE; FillCh : CHAR) : STRING;

  { Returns date <y>, <m>, <d> converted to a string in SI format.  If     }
  { <Slen> = 10, the string is in form YYYY_MM_DD; If <SLen> = 8, in form  }
  { YY_MM_DD; otherwise a NULL string is returned.  The character between  }
  { values is <FillCh>.                                                    }
  { For correct Systeme-Internationale date format, the call should be:    }
  {   SIDateStr (Year, Month, Day, 10, ' ');                               }
  { IF <y>, <m> & <d> are all = 0, Runtime library PROCEDURE GetDate is    }
  { called to obtain the current date.                                     }

  VAR
    s2 : STRING[2];
    s4 : STRING[4];
    DStr : STRING[10];
    Index : BYTE;
    dw : WORD;
  BEGIN
    IF (SLen <> 10) AND (SLen <> 8) THEN
      DStr[0] := Chr (0)
    ELSE
      BEGIN
        IF (y = 0) AND (m = 0) AND (d = 0) THEN
          GetDate (y, m, d, dw);
        IF SLen = 10 THEN
          BEGIN
            Str (y:4, s4);
            DStr[1] := s4[1];
            DStr[2] := s4[2];
            DStr[3] := s4[3];
            DStr[4] := s4[4];
            Index := 5
          END
        ELSE
          IF SLen = 8 THEN
            BEGIN
              Str (y MOD 100:2, s2);
              DStr[1] := s2[1];
              DStr[2] := s2[2];
              Index := 3
            END;
        DStr[Index] := FillCh;
        Inc (Index);
        Str (m:2, s2);
        IF s2[1] = ' ' THEN
          DStr[Index] := '0'
        ELSE
          DStr[Index] := s2[1];
        DStr[Index+1] := s2[2];
        Index := Index + 2;
        DStr[Index] := FillCh;
        Inc (Index);
        Str (d:2, s2);
        IF s2[1] = ' ' THEN
          DStr[Index] := '0'
        ELSE
          DStr[Index] := s2[1];
        DStr[Index+1] := s2[2];
        DStr[0] := Chr (SLen)
      END;
    SIDateStr := DStr
  END;  { SIDateStr }
 
{==========================================================================}

  FUNCTION TimeStr (h, m, s, c : WORD) : STRING;

  { Returns the time <h>, <m>, <s> and <c> formatted in a string:          }
  { "HH:MM:SS.CC"                                                          }
  { This function does NOT check for valid string length.                  }
  {                                                                        }
  { IF <h>, <m>, <s> & <c> all = 0, the Runtime PROCEDURE GetTime is       }
  { called to get the current time.                                        }

  VAR
    sh, sm, ss, sc : STRING[2];
  BEGIN
    IF h + m + s + c = 0 THEN
      GetTime (h, m, s, c);
    Str (h:2, sh);
    IF sh[1] = ' ' THEN
      sh[1] := '0';
    Str (m:2, sm);
    IF sm[1] = ' ' THEN
      sm[1] := '0';
    Str (s:2, ss);
    IF ss[1] = ' ' THEN
      ss[1] := '0';
    Str (c:2, sc);
    IF sc[1] = ' ' THEN
      sc[1] := '0';
    TimeStr := Concat (sh, ':', sm, ':', ss, '.', sc)
  END;  { TimeStr }

{==========================================================================}

  FUNCTION Secs100 (h, m, s, c : WORD) : DOUBLE;

  { Returns the given time <h>, <m>, <s> and <c> as a floating-point       }
  { value in seconds (presumably valid to .01 of a second).                }
  {                                                                        }
  { IF <h>, <m>, <s> & <c> all = 0, the Runtime PROCEDURE GetTime is       }
  { called to get the current time.                                        }

  BEGIN
    IF h + m + s + c = 0 THEN
      GetTime (h, m, s, c);
    Secs100 :=  (h * 60.0 + m) * 60.0 + s + (c * 0.01)
  END;  { Secs100 }

{==========================================================================}

  PROCEDURE AddDays (y, m, d : WORD; plus : LONGINT; VAR y1, m1, d1 : WORD);

  { Computes the date <y1>, <m1>, <d1> resulting from the addition of      }
  { <plus> days to the calendar date <y>, <m>, <d>.                        }

  BEGIN
    t1 := JulianDay (y, m, d) + plus;
    JulianDayToDate (t1, y1, m1, d1)
  END;  { AddDays }

{==========================================================================}

  FUNCTION ValidDate (y : INTEGER; m, d : WORD) : BOOLEAN;

  { Returns TRUE if the date <y> <m> <d> is valid.                         }

  VAR
    ycal : INTEGER;
    mcal, dcal : WORD;
  BEGIN
    t1 := JulianDa2 (y, m, d);
    JulianDa2ToDate (t1, ycal, mcal, dcal);
    ValidDate := (y = ycal) AND (m = mcal) AND (d = dcal)
  END;  { ValidDate }

{==========================================================================}

  FUNCTION Da2OfWeek (y : INTEGER; m, d : WORD) : WORD;

  { Returns the Day-of-the-week (0 = Sunday) corresponding to the calendar }
  { date <y>, <m>, <d>. The algorithm is taken from Numerical Recipes.     }

  BEGIN
    Da2OfWeek := (JulianDa2 (y, m, d) + 1) MOD 7
  END;  { Da2OfWeek }

{==========================================================================}

  FUNCTION DayOfWeek (y, m, d : WORD) : WORD;

  { Returns the Day-of-the-week (0 = Sunday) (Zeller's congruence) from an }
  { algorithm IZLR given in a remark on CACM Algorithm 398.                }

  BEGIN
    t1 := m + 10;
    t2 := y;
    t3 := m - 14;
    t2 := t2 + t3 DIV 12;
    t3 := d;
    DayOfWeek :=  ((13 *  (t1 - t1 DIV 13 * 12) - 1) DIV 5 +
                  t3 + 77 + 5 * (t2 - t2 DIV 100 * 100) DIV 4 +
                  t2 DIV 400 - t2 DIV 100 * 2) MOD 7 
  END;  { DayOfWeek }

{==========================================================================}
END.
