
UNIT ColorPrm;

{ ****************************************************************************
  *
  *  Unit Name       :    Color Prompt Window
  *  Version         :    2.3
  *  Author          :    David C. Doty / FM Software, Inc.
  *  Date            :    30 January 1990
  *  Description     :    This file contains unit that allows the user of a
  *                       program to specify a color for an item by selecting
  *                       the color from a window of alternatives, rather than
  *                       by entering a number or choosing from a menu.
  *                       Functions are also provided to translate a color
  *                       byte into an English description.
  *
  *                       Function hooks:
  *
  *                       o  The variable ColorKeyPtr may be used to provide
  *                          "background" processing similar to that provided
  *                          by the TPpick, TPedit, TPmemo, TPhelp, etc units.
  *                          Set the variable to the address of a function that
  *                          takes no arguments and returns a WORD containing
  *                          the keystoke.  Note that this may point to a
  *                          function that checks both KeyPressed and
  *                          MouseKeyPressed to get mouse button presses.
  *
  *                       o  The variable ColorUserPtr may be used to provide
  *                          special processing each time the color cursor moves
  *                          onto a color selection.  Set the variable to the
  *                          address of a procedure that takes one argument (the
  *                          color byte for the selected spot).
  *
  *                       o  The variables ColorHelpPtr and ColorHelpXtra may
  *                          be used to provide context-sensitive help for the
  *                          item under the selection cursor.  Set the variable
  *                          ColorHelpPtr to the address of a procedure that
  *                          takes a BYTE argument holding the unit identifier
  *                          for which help is requested (which will be set to
  *                          the constant HelpForColor), a POINTER argument
  *                          (which will be set to ColorHelpXtra), and a WORD
  *                          argument (which will be set to the color under the
  *                          color selection cursor).  You must, of course,
  *                          determine the help index to use in the call to
  *                          ShowHelp within the routine at ColorHelpPtr.
  *
  *                       Other hooks : you can change the character(s) used
  *                       to represent the unselected, selected, and default
  *                       colors by changing the values of ColorSpots or
  *                       BigColorSpots, which are interfaced for that reason.
  *                       Similarly, you can change the language of the color
  *                       to English conversions by changing ColorStringArray
  *                       and BlinkingString.
  *
  *  Revision history:    version 1.0 : color pick window 13 June 1988
  *
  *                       version 2.0 : added color string functions 20 July 88
  *
  *                       version 2.1 : changed cell size to 2x2 12 August 88
  *
  *                       version 2.2 : added hooks for background processing,
  *                                     mouse, "user function", and context-
  *                                     sensitive help.  Included functions for
  *                                     both 1x1 and 2x2 cells.
  *
  *                       version 2.3 : included support for monochrome screens.
  *
  *  Copyright (C) FM Software, Inc. 1990.  All rights reserved.
  *
  *  This unit may be distributed freely to users of Turbo Professional from
  *  TurboPower Software provided the copyright notice and this distribution
  *  notice remain in the file.
  *
  *************************************************************************** }


{================================} INTERFACE {===============================}


{$I TPDEFINE.INC - read a number of conditional compile directives, including
                   UseMouse }

USES TPcrt, TPwindow{$IFDEF UseMouse}, TPMouse{$ENDIF};


TYPE
   ColorCharSet = SET OF CHAR;
   ColorString  = STRING[ 15 ];


CONST
   HelpForColor = 255;

   ColorStringArray : ARRAY [ 0 .. 15 ] OF ColorString = (
      'black', 'blue', 'green', 'cyan', 'red', 'magenta', 'brown',
      'light grey', 'dark grey', 'light blue', 'light green', 'light cyan',
      'light red', 'light magenta', 'yellow', 'white' );

   BlinkingString : ColorString = 'blinking';


FUNCTION PromptForColor( { input } Default,
                         { input } Xwindow,
                         { input } Ywindow : BYTE;
                         { input } FrameAttr,
                         { input } HeaderAttr : BYTE;
                         { input } Header : STRING;
                         { input } AcceptSet : ColorCharSet;
                         { output } VAR Color : BYTE ) : BOOLEAN;
{  This function displays a window of allowed colors from which the user can
   select one.  If the user presses the Escape key, the function returns FALSE
   to indicate "no selection"; otherwise, the color selected by the user (i.e.,
   the color that is shown with a prompt character when one of the characters
   in AcceptSet is pressed) is returns in Color and the function returns TRUE.
   The prompt window is labelled with the string passed in Header, to tell the
   user what color is being requested.  Xwindow and Ywindow form the upper-
   left corner of the window to hold the colors.  FrameAttr and HeaderAttr are
   the colors of the frame around the window and the header string,
   respectively.  The body of the window is covered by the color choices. }


FUNCTION BigPromptForColor( { input } Default,
                            { input } Xwindow,
                            { input } Ywindow : BYTE;
                            { input } FrameAttr,
                            { input } HeaderAttr : BYTE;
                            { input } Header : STRING;
                            { input } AcceptSet : ColorCharSet;
                            { output } VAR Color : BYTE ) : BOOLEAN;
{  This function behaves identically with PromptForColor except that the cells
   in the color pick window are 2x2 characters instead of 1x1.  The window is,
   naturally, somewhat larger for this case. }


FUNCTION ForegroundColorString( { input } Color : BYTE ) : STRING;
{  This function returns an English description of the foreground color,
   including whether the text is flashing or not and whether the color is
   brightened or not. }


FUNCTION BackgroundColorString( { input } Color : BYTE ) : STRING;
{  This function returns an English description of the background color. }


FUNCTION FullColorString( { input } Color : BYTE ) : STRING;
{  This function returns an English description of both the fore- and back-
   ground colors, in the form "fore on back". }


{$IFDEF UseMouse}
PROCEDURE EnableColorMouse;
{  This procedure must be called before the mouse may be used within the
   PromptForColor or BigPromptForColor functions. }


PROCEDURE DisableColorMouse;
{  This procedure must be called to prevent use of the mouse within the
   PromptForColor or BigPromptForColor functions. }
{$ENDIF}


CONST
   ColorKeyPtr   : POINTER = NIL;   { pointer to keyboard input function }
   ColorUserPtr  : POINTER = NIL;   { pointer to user action procedure }
   ColorHelpPtr  : POINTER = NIL;   { pointer to context-sensitive help proc }
   ColorHelpXtra : POINTER = NIL;   { this variable may be set to any value
                                      by users of this unit.  It is passed
                                      to the context-sensitive help routine. }

TYPE
   ColorSpotType = ( SelectedColorSpot, UnselectedColorSpot, DefaultColorSpot );

   ColorCharArray = ARRAY [ ColorSpotType ] OF CHAR;

   BigColorCorners = ( ColorUpperLeft, ColorUpperRight, ColorLowerLeft,
                       ColorLowerRight );

   BigColorCharArray = ARRAY [ ColorUpperLeft .. ColorLowerRight ] OF CHAR;


CONST
   ColorSpots : ColorCharArray = ( #8 , #4 , #254 );

   BigColorSpots : ARRAY[ ColorSpotType ] OF BigColorCharArray = (
        ( #220, #220, #223, #223 ),
        ( #218, #191, #192, #217 ),
        ( #201, #187, #200, #188 ) );


{==============================} IMPLEMENTATION {============================}


TYPE
   ForeArray = ARRAY [ 0 .. 15 ] OF BYTE;

   BackArray = ARRAY [ 0 ..  7 ] OF BYTE;

   ForeArrayPtr = ^ForeArray;

   BackArrayPtr = ^BackArray;


CONST
   {  Foreground mono: 000 = no disp
                       001 = underline
                       111 = normal

      Background mono: 000 = normal
                       111 = reverse }

   ColorForeMax = 15;
   ColorBackMax =  7;

   MonoForeMax  =  5;
   MonoBackMax  =  1;

   ColorForeTbl : ForeArray = ( 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13,
                               14, 15 );
   ColorBackTbl : BackArray = ( 0, 1, 2, 3, 4, 5, 6, 7 );

   MonoForeTbl  : ForeArray = ( 0, 1, 7, 8, 9, 15, 0, 0, 0, 0, 0, 0, 0, 0,
                                0, 0 );
   MonoBackTbl  : BackArray = ( 0, 7, 0, 0, 0, 0, 0, 0 );

   {$IFDEF UseMouse}
   UseMouse     : BOOLEAN = FALSE;
   {$ENDIF}

   LeftArrow  = $4B00;
   RiteArrow  = $4D00;
   UpArrow    = $4800;
   DownArrow  = $5000;
   Escape     = $011B;
   F1         = $3B00;
   Enter      = $1C0D;


VAR
   ForePtr : ForeArrayPtr;

   BackPtr : BackArrayPtr;

   XwindowMax,
   YwindowMax,
   {$IFDEF UseMouse}
   MouseSaveXLo,
   MouseSaveYLo,
   MouseSaveXHi,
   MouseSaveYHi,
   MouseSaveX,
   MouseSaveY,
   {$ENDIF}
   SpotSize,
   ForeMax,
   BackMax : BYTE;

   DrawColorSpotProc : POINTER; {  For purely TP 5.X, would be
                       PROCEDURE( Fore, Back : BYTE; Spot : ColorSpotType ); }


PROCEDURE CallUserProc( Item : WORD );
{  This inline macro calls the user procedure pointed to by ColorUserPtr. }
   INLINE( $FF/$1E/>ColorUserPtr );  { CALL DWORD PTR [>ColorUserPtr] }


PROCEDURE CallHelpProc( HelpFor : BYTE; IDPtr : POINTER; HelpIndex : WORD );
{  This inline macro calls the help procedure pointed to by ColorHelpPtr. }
   INLINE( $FF/$1E/>ColorHelpPtr );  { CALL DWORD PTR [>ColorHelpPtr] }


PROCEDURE CallDrawColorSpotProc( Fore, Back : BYTE; Spot : ColorSpotType );
{  This inline macro calls the DrawColorSpot routine selected by the caller.}
   INLINE( $FF/$1E/>DrawColorSpotProc ); { CALL DWORD PTR [>Draw..Proc] }


FUNCTION CallKeyFunc : WORD;
{  This inline macro calls the keyboard function pointed to by ColorKeyPtr. }
   INLINE( $FF/$1E/>ColorKeyPtr );  { CALL DWORD PTR [>ColorKeyPtr] }


{$F+ - these two procedures must be FAR since they will be assigned to a
       procedure variable }
PROCEDURE DrawColorSpot( Fore, Back : BYTE; Spot : ColorSpotType );
{  Draw one entry of the color map.}
BEGIN
   IF Spot <> UnselectedColorSpot
   THEN FastWriteWindow( ColorSpots[ Spot ], Succ( Back ), Succ( Fore ),
                      Blink OR ( BackPtr^[ Back ] SHL 4 ) OR ForePtr^[ Fore ] )
   ELSE FastWriteWindow( ColorSpots[ Spot ], Succ( Back ), Succ( Fore ),
                               ( BackPtr^[ Back ] SHL 4 ) OR ForePtr^[ Fore ] )
END { PROCEDURE DrawColorSpot };


PROCEDURE DrawBigColorSpot( Fore, Back : BYTE; Spot : ColorSpotType );
{  Draw one entry of the color map using a 2x2 spot size. }
VAR
   Color : BYTE;
BEGIN { PROCEDURE DrawBigColorSpot }
   Color := ( BackPtr^[ Back ] SHL 4 ) OR ForePtr^[ Fore ];
   IF Spot <> UnselectedColorSpot
   THEN Color := Color OR Blink;
   Back := Succ( Back SHL 1 );
   Fore := Succ( Fore SHL 1 );
   FastWriteWindow( BigColorSpots[ Spot, ColorUpperLeft ], Back, Fore, Color );
   FastWriteWindow( BigColorSpots[ Spot, ColorUpperRight ], Back, Succ( Fore ),
                    Color );
   FastWriteWindow( BigColorSpots[ Spot, ColorLowerLeft ], Succ( Back ), Fore,
                    Color );
   FastWriteWindow( BigColorSpots[ Spot, ColorLowerRight ], Succ( Back ),
                    Succ( Fore ), Color )
END { PROCEDURE DrawBigColorSpot };
{$F- - turn off FAR call forcing }


PROCEDURE DrawColorMap;
{  Draw the complete color map.}
VAR
   Fore,
   Back : BYTE;
BEGIN { PROCEDURE DrawColorMap }
   FOR Fore := 0 to ForeMax DO
      FOR Back := 0 to BackMax DO
         CallDrawColorSpotProc( Fore, Back, UnselectedColorSpot )
END { PROCEDURE DrawColorMap };


FUNCTION GenericPromptForColor( { input } Default,
                                { input } Xwindow,
                                { input } Ywindow,
                                { input } FrameAttr,
                                { input } HeaderAttr : BYTE;
                                { input } Header : STRING;
                                { input } AcceptSet : ColorCharSet;
                                { output } VAR Color : BYTE ) : BOOLEAN;
{  This function implements both color prompt functions, allowing the
   programmer to choose the size of the window elements without duplicating
   too much code. }
VAR
   CursorX,
   CursorY,
   Fore,
   Back,
   ForeDef,
   BackDef : BYTE;
   ColorWindow : WindowPtr;
   Key : WORD;
   {$IFDEF UseMouse}
   MouseOn: BOOLEAN;
   {$ENDIF}
BEGIN { FUNCTION GenericPromptForColor }
   GenericPromptForColor := FALSE; { assume failure }

   CursorX := WhereXabs;
   CursorY := WhereYabs;

   IF Xwindow > XwindowMax
   THEN Xwindow := XwindowMax
   ELSE IF Xwindow = 0
   THEN Xwindow := 1;

   IF Ywindow > YwindowMax
   THEN Ywindow := YwindowMax
   ELSE IF Ywindow = 0
   THEN Ywindow := 1;

   IF ( CurrentMode = Mono ) OR
      ( CurrentMode = BW40 ) OR
      ( CurrentMode = BW80 )
   THEN BEGIN { using monochrome display }
      ForePtr := @MonoForeTbl;
      BackPtr := @MonoBackTbl;
      ForeMax := MonoForeMax;
      BackMax := MonoBackMax;
   END { mono }
   ELSE BEGIN { using color display }
      ForePtr := @ColorForeTbl;
      BackPtr := @ColorBackTbl;
      ForeMax := ColorForeMax;
      BackMax := ColorBackMax;
   END { color };

   { force the default color to match the limits of the color/mono display }

   ForeDef := Default AND $F;
   BackDef := Default SHR 4;

   IF BackDef > 7
   THEN Dec( BackDef, 8 );

   Fore := 0;
   Back := 0;

   WHILE ( Fore < ForeMax ) AND
         ( ForePtr^[ Fore ] <> ForeDef )
   DO Inc( Fore );

   WHILE ( Back < BackMax ) AND
         ( BackPtr^[ Back ] <> BackDef )
   DO Inc( Back );

   ForeDef := Fore;
   BackDef := Back;

   {$IFDEF UseMouse}
   IF UseMouse
   THEN BEGIN
      MouseOn := MouseCursorOn;

      MouseSaveXLo := MouseXLo;
      MouseSaveYLo := MouseYLo;
      MouseSaveXHi := MouseXHi;
      MouseSaveYHi := MouseYLo;

      MouseSaveX   := MouseWhereX;
      MouseSaveY   := MouseWhereY;

      MouseWindow( Succ( Xwindow ), Succ( Ywindow ),
                   Xwindow + SpotSize * Succ( ForeMax ),
                   Ywindow + SpotSize * Succ( BackMax ) );

      MouseGoToXY( Succ( SpotSize * ForeDef ), Succ( SpotSize * BackDef ) );

      HideMouse
   END
   ELSE MouseOn := FALSE;
   {$ENDIF}

   IF NOT MakeWindow( ColorWindow, Xwindow, Ywindow,
                      Succ( Xwindow + SpotSize * Succ( ForeMax ) ),
                      Succ( Ywindow + SpotSize * Succ( BackMax ) ),
                      TRUE, TRUE, FALSE, $07, FrameAttr,
                      HeaderAttr, Header )
   THEN Exit { could not create window }
   ELSE IF NOT DisplayWindow( ColorWindow )
      THEN BEGIN { could create but not show window }
         DisposeWindow( ColorWindow );
         Exit
      END { could create but not show window };

   {ELSE BEGIN - everything else runs with a window displayed }

   DrawColorMap;

   REPEAT
      Default := ( BackPtr^[ Back ] SHL 4 ) OR ForePtr^[ Fore ];

      CallDrawColorSpotProc( Fore, Back, SelectedColorSpot );

      IF ColorUserPtr <> NIL
      THEN CallUserProc( Default );

      {$IFDEF UseMouse}
      IF UseMouse
      THEN ShowMouse;
      {$ENDIF}

      Key := CallKeyFunc;

      {$IFDEF UseMouse}
      IF UseMouse
      THEN HideMouse;
      {$ENDIF}

      {Clear the previous selection}

      IF ( Fore = ForeDef ) AND
         ( Back = BackDef )
      THEN CallDrawColorSpotProc( Fore, Back, DefaultColorSpot )
      ELSE CallDrawColorSpotProc( Fore, Back, UnselectedColorSpot );

      CASE Key OF { move color cursor depending on user key press }
         LeftArrow : IF Fore > 0
                     THEN Dec( Fore )
                     ELSE Fore := ForeMax;

         RiteArrow : IF Fore < ForeMax
                     THEN Inc( Fore )
                     ELSE Fore := 0;

         UpArrow   : IF Back > 0
                     THEN Dec( Back )
                     ELSE Back := BackMax;

         DownArrow : IF Back < BackMax
                     THEN Inc( Back )
                     ELSE Back := 0;

         F1        : IF ColorHelpPtr <> NIL
                     THEN CallHelpProc( HelpForColor, ColorHelpXtra, Default );

         Escape    : { user aborts select - do nothing };

         {$IFDEF UseMouse}
         MouseRt   : IF UseMouse
                     THEN Key := Escape;

         MouseBoth : IF ( ColorHelpPtr <> NIL ) AND UseMouse
                     THEN CallHelpProc( HelpForColor, ColorHelpXtra, Default );

         MouseLft  : IF UseMouse
                     THEN BEGIN
                        Key := Enter;
                        Default :=
                           ForePtr^[ Pred( MouseWhereX ) DIV SpotSize ] OR
                         ( BackPtr^[ Pred( MouseWhereY ) DIV SpotSize ] SHL 4 )
                     END;
         {$ENDIF}

         ELSE        { nothing };

      END { CASE Key };

   UNTIL Char( Lo( Key ) ) IN AcceptSet;

   IF Key <> Escape { i.e., user didn't abort select }
   THEN BEGIN
      GenericPromptForColor := TRUE;
      Color := Default
   END;

   {$IFDEF UseMouse}
   IF UseMouse
   THEN BEGIN
      HideMouse;

      MouseWindow( Succ( MouseSaveXLo ), Succ( MouseSaveYLo ), MouseSaveXHi,
                   MouseSaveYHi );

      MouseGoToXY( MouseSaveX, MouseSaveY )
   END;
   {$ENDIF}

   DisposeWindow( EraseTopWindow );
   GoToXYAbs( CursorX, CursorY );

   {$IFDEF UseMouse}
   IF MouseOn
   THEN ShowMouse
   {$ENDIF}

END { FUNCTION GenericPromptForColor };


FUNCTION PromptForColor( { input } Default,
                         { input } Xwindow,
                         { input } Ywindow,
                         { input } FrameAttr,
                         { input } HeaderAttr : BYTE;
                         { input } Header : STRING;
                         { input } AcceptSet : ColorCharSet;
                         { output } VAR Color : BYTE ) : BOOLEAN;
{  This function displays a window of allowed colors from which the user can
   select one.  If the user presses the Escape key, the function returns FALSE
   to indicate "no selection"; otherwise, the color selected by the user (i.e.,
   the color that is shown with a prompt character when one of the characters
   in AcceptSet is pressed) is returns in Color and the function returns TRUE.
   The prompt window is labelled with the string passed in Header, to tell the
   user what color is being requested.  Xwindow and Y window form the upper-
   left corner of the window to hold the colors.  FrameAttr and HeaderAttr are
   the colors of the frame around the window and the header string,
   respectively.  The body of the window is covered by the color choices. }
BEGIN
   SpotSize   :=  1;
   XwindowMax := 60;
   YwindowMax := 15;
   DrawColorSpotProc := @DrawColorSpot;
   PromptForColor := GenericPromptForColor( Default, Xwindow, Ywindow,
                                            FrameAttr, HeaderAttr, Header,
                                            AcceptSet, Color )
END { FUNCTION PromptForColor };


FUNCTION BigPromptForColor( { input } Default,
                            { input } Xwindow,
                            { input } Ywindow,
                            { input } FrameAttr,
                            { input } HeaderAttr : BYTE;
                            { input } Header : STRING;
                            { input } AcceptSet : ColorCharSet;
                            { output } VAR Color : BYTE ) : BOOLEAN;
{  This function behaves exactly as PromptForColor except that the color spots
   are 2x2 characters instead of the 1x1 spots use in the smaller window. }
BEGIN
   SpotSize   :=  2;
   XwindowMax := 44;
   YwindowMax :=  7;
   DrawColorSpotProc := @DrawBigColorSpot;
   BigPromptForColor := GenericPromptForColor( Default, Xwindow, Ywindow,
                                               FrameAttr, HeaderAttr, Header,
                                               AcceptSet, Color )
END { FUNCTION BigPromptForColor };


FUNCTION ForegroundColorString( { input } Color : BYTE ) : STRING;
{  This function returns an English description of the foreground color,
   including whether the text is flashing or not and whether the color is
   brightened or not. }
BEGIN
   IF Color >= Blink
   THEN ForegroundColorString := BlinkingString + ' ' + ColorStringArray[ Color AND $0F ]
   ELSE ForegroundColorString := ColorStringArray[ Color AND $0F ]
END { FUNCTION ForegroundColorString };


FUNCTION BackgroundColorString( { input } Color : BYTE ) : STRING;
{  This function returns an English description of the background color. }
BEGIN
   BackgroundColorString := ColorStringArray[ ( Color AND $70 ) SHR 4 ]
END { BackgroundColorString };


FUNCTION FullColorString( { input } Color : BYTE ) : STRING;
{  This function returns an English description of both the fore- and back-
   ground colors, in the form "fore on back". }
BEGIN
   FullColorString := ForegroundColorString( Color ) + ' on ' +
                      BackgroundColorString( Color )
END { FUNCTION FullColorString };


{$IFDEF UseMouse}
PROCEDURE EnableColorMouse;
BEGIN
   UseMouse := TRUE
END { PROCEDURE EnableColorMouse };


PROCEDURE DisableColorMouse;
BEGIN
   UseMouse := FALSE
END { PROCEDURE DisableColorMouse };
{$ENDIF}


{============================== INITIALIZATION ==============================}


BEGIN { UNIT ColorPrm initialization }

   {$IFDEF UseMouse}
   ColorKeyPtr := @ReadKeyOrButton
   {$ELSE}
   ColorKeyPtr := @ReadKeyWord
   {$ENDIF}

END { UNIT ColorPrm }.
