Unit Buttons;
{***************************************************************************}
{*  Buttons.pas  by Daniel Thomas (CIS: 72301,2164)                        *}
{*                                                                         *}
{*    This code is hereby donated to tbe Public Domain.  Have fun!         *}
{*                                                                         *}
{*                                                                         *}
{* There are 2 kinds of Button objects in this library.  Both of them      *}
{* are for creating buttons made up of bitmaps.  The first object uses     *}
{* only one bitmap, and draws any needed "effects".  The second object     *}
{* uses three bitmaps, one for each state (up, down, and disabled).        *}
{*                                                                         *}
{* Neither of these objects supports drawing on Dialogs, but it's a        *}
{* start!  Also, neither of these buttons looks any different if it's      *}
{* the default button.  Again, you're on your own!                         *}
{*                                                                         *}
{* For either type of button, just Init the object in the parent's Init    *}
{* method, and you've got a button!  If you want to create it in some      *}
{* other method than the parent's Init method, use Application^.MakeWindow *}
{* to make it visible.                                                     *}
{*                                                                         *}
{* Add a method to the parent window, like this:                           *}
{*                                                                         *}
{*    Procedure wmDrawItem(var Msg:tMessage);virtual wm_First+wm_DrawItem; *}
{*                                                                         *}
{* It should look like this:                                               *}
{*                                                                         *}
{*    Procedure tMainWindow.wmDrawItem(var Msg:tMessage);                  *}
{*    begin                                                                *}
{*      with pDrawItemStruct(Msg.lParam)^ do                               *}
{*        case CtlType of                                                  *}
{*            odt_Button:                                                  *}
{*              case CtlID of                                              *}
{*                  id_Button1 : Button1^.DrawItem(Msg);                   *}
{*                  id_Button2 : Button2^.DrawItem(Msg);                   *}
{*              end;                                                       *}
{*        end;                                                             *}
{*    end;                                                                 *}
{*                                                                         *}
{*                                                                         *}
{* tSingleBitmapButton                                                     *}
{*                                                                         *}
{*   Two Init methods.  The second one allows you to specify the color     *}
{*   of a surrounding box.                                                 *}
{*                                                                         *}
{*   Specify the parent object (@self), an ID number, the location of      *}
{*   the button (x & y), whether it's the default, and the name of the     *}
{*   bitmap.  (The second Init method also let's you specify an RGB        *}
{*   color for a border).                                                  *}
{*                                                                         *}
{*   Create the bitmap as large as the button needs to be, minus the       *}
{*   surrounding black box.  It is assumed that the button's "background"  *}
{*   color is light gray.  The "shading" effects will be drawn for you.    *}
{*                                                                         *}
{* tMultiBitmapButton                                                      *}
{*                                                                         *}
{*   Specify the parent object (@self), an ID number, the location of      *}
{*   the button (x & y), whether it's the default, and the name of the     *}
{*   three bitmaps (one for a non-pressed button, one for a pressed        *}
{*   button, and one for a disabled button).  ALL THREE MUST BE THE SAME   *}
{*   SIZE.                                                                 *}
{*                                                                         *}
{***************************************************************************}

interface

uses WinTypes,WinProcs,WObjects;

type
  pSingleBitmapButton=^tSingleBitmapButton;
  tSingleBitmapButton=object(tButton)
      Bitmap             : hBitmap;
      UseSpecialBorder   : boolean;
      SpecialBorderColor : longint;

      constructor Init(aParent: pWindowsObject; aID: Integer;
                       X,Y: Integer; IsDefault: Boolean;
                       aBitmap: pChar);
      constructor InitWithSpecialBorder(
                       aParent: pWindowsObject; aID: Integer;
                       X,Y: Integer; IsDefault: Boolean;
                       aBitmap: pChar;
                       aSpecialBorderColor: longint);
      destructor Done; virtual;
      procedure DrawItem(var Msg:tMessage); virtual;
    end;

  pMultiBitmapButton=^tMultiBitmapButton;
  tMultiBitmapButton=object(tButton)
      NormalBitmap,
      DownBitmap,
      DisabledBitmap : hBitmap;

      constructor Init(aParent: pWindowsObject; aID: Integer;
                       X,Y: Integer; IsDefault: Boolean;
                       aNormalBitmap,aDownBitmap,aDisabledBitmap: pChar);
      destructor Done; virtual;
      procedure DrawItem(var Msg:tMessage); virtual;
    end;

implementation

const
  cBlackColor     = $00000000;
  cWhiteColor     = $00FFFFFF;
  cDarkGrayColor  = $00808080;
  cLightGrayColor = $00C0C0C0;

type
  pDrawItemStruct = ^tDrawItemStruct;

var
  bmp : hBitmap;

{**************************************************************************}
{*  tSingleBitmapButton                                                   *}
{**************************************************************************}

constructor tSingleBitmapButton.Init(aParent: pWindowsObject; aID: Integer;
                                     X,Y: Integer; IsDefault: Boolean;
                                     aBitmap: pChar);

var
  bm  : tBitMap;
  w,h : integer;

begin
  bmp := LoadBitmap(hInstance,aBitmap);
  GetObject(bmp,sizeof(bm),@bm);
  tButton.Init(aParent,aID,'Dummy',x,y,bm.bmWidth+2,bm.bmHeight+2,IsDefault);
  Attr.Style := Attr.Style or bs_OwnerDraw;
  Bitmap := bmp;
  UseSpecialBorder := false;
end; {tSingleBitmapButton.Init}

constructor tSingleBitmapButton.InitWithSpecialBorder(
                                     aParent: pWindowsObject; aID: Integer;
                                     X,Y: Integer; IsDefault: Boolean;
                                     aBitmap: pChar;
                                     aSpecialBorderColor: longint);

var
  bm  : tBitMap;
  w,h : integer;

begin
  bmp := LoadBitmap(hInstance,aBitmap);
  GetObject(bmp,sizeof(bm),@bm);
  tButton.Init(aParent,aID,'Dummy',x,y,bm.bmWidth+4,bm.bmHeight+4,IsDefault);
  Attr.Style := Attr.Style or bs_OwnerDraw;
  Bitmap := bmp;
  UseSpecialBorder := true;
  SpecialBorderColor := aSpecialBorderColor;
end; {tSingleBitmapButton.Init}

destructor tSingleBitmapButton.Done;

begin
  tButton.Done;
  DeleteObject(Bitmap);
end; {tSingleBitmapButton.Done}

procedure tSingleBitmapButton.DrawItem(var Msg: tMessage);

var
  Down            : boolean;
  OldMode,
  w,h,x1,y1       : integer;
  MemDC           : hDC;
  OldBmp          : hBitmap;
  Pts             : array[0..4] of tPoint;
  Pen,
  OldPen          : hPen;
  Brush,
  OldBrush        : hBrush;
  TopLeftLine1,
  TopLeftLine2,
  BottomRightLine1,
  BottomRightLine2: array[0..2] of TPoint;

begin
  with pDrawItemStruct(Msg.lParam)^, rcItem do begin
    if ItemAction = oda_Focus then
      exit;

    Down := ((ItemAction and oda_Select) > 0)
            and ((ItemState and ods_Selected) > 0);
    if UseSpecialBorder then
      begin
        x1 := 1;
        y1 := 1;
      end
    else
      begin
        x1 := 0;
        y1 := 0;
      end;
    w := right - left - (2*x1);
    h := bottom - top - (2*y1);


{Draw the bitmap - offset to the left and down if the button is down}
    MemDC := CreateCompatibleDC(hDC);
    SelectObject(MemDC,Bitmap);
    if Down then
      BitBlt(hDC,left+3+x1,top+3+y1,w-4,h-4,MemDC,0,0,SrcCopy)
    else
      BitBlt(hDC,left+1+x1,top+1+y1,w-2,h-2,MemDC,0,0,SrcCopy);
    DeleteDC(MemDC);

{"Gray" the button, if it is disabled}
    if (itemState and ods_Disabled <> 0)
        or (itemState and ods_Grayed <> 0) then
      begin
        Pen := CreatePen(ps_Solid,1,cBlackColor);
        OldPen := SelectObject(hDC,Pen);
        Brush := CreateHatchBrush(hs_bDiagonal,cBlackColor);
        OldBrush := SelectObject(hDC,Brush);
        OldMode := SetBkMode(hDC,Transparent);
        Rectangle(Hdc,left+x1,top+y1,right-x1,bottom-y1);
        SelectObject(hDC,OldPen);
        DeleteObject(Pen);
        SelectObject(hDC,OldBrush);
        DeleteObject(Brush);
        SetBkMode(hDC,OldMode);
      end;

{Draw the surrounding rectangle}
    Pen := CreatePen(ps_Solid,1,cBlackColor);
    OldPen := SelectObject(hDC,Pen);
    Brush := GetStockObject(Null_Brush);
    OldBrush := SelectObject(hDC,Brush);
    Rectangle(Hdc,left+x1,top+y1,right-x1,bottom-y1);
    SelectObject(hDC,OldPen);
    DeleteObject(Pen);
    if UseSpecialBorder then
      begin
        Pen := CreatePen(ps_Solid,1,SpecialBorderColor);
        OldPen := SelectObject(hDC,Pen);
        Rectangle(Hdc,left,top,right,bottom);
        SelectObject(hDC,OldPen);
        DeleteObject(Pen);
      end;
    SelectObject(hDC,OldBrush);

{Draw the "shading"}
    TopLeftLine1[0].x := right-(2+x1);
    TopLeftLine1[0].y := top+1+y1;
    TopLeftLine1[1].x := left+1+x1;
    TopLeftLine1[1].y := top+1+y1;
    TopLeftLine1[2].x := left+1+x1;
    TopLeftLine1[2].y := bottom-(1+y1);
    TopLeftLine2[0].x := right-(3+x1);
    TopLeftLine2[0].y := top+2+y1;
    TopLeftLine2[1].x := left+2+x1;
    TopLeftLine2[1].y := top+2+y1;
    TopLeftLine2[2].x := left+2+x1;
    TopLeftLine2[2].y := bottom-(2+y1);
    if not down then
      begin
        BottomRightLine1[0].x := right-(2+x1);
        BottomRightLine1[0].y := top+2+y1;
        BottomRightLine1[1].x := right-(2+x1);
        BottomRightLine1[1].y := bottom-(2+y1);
        BottomRightLine1[2].x := left+1+x1;
        BottomRightLine1[2].y := bottom-(2+y1);
        BottomRightLine2[0].x := right-(3+x1);
        BottomRightLine2[0].y := top+3+y1;
        BottomRightLine2[1].x := right-(3+x1);
        BottomRightLine2[1].y := bottom-(3+y1);
        BottomRightLine2[2].x := left+2+x1;
        BottomRightLine2[2].y := bottom-(3+y1);
      end;

    if Down then
      begin
        Pen := CreatePen(ps_Solid,1,cDarkGrayColor);
        OldPen := SelectObject(hDC,Pen);
        PolyLine(hDC,TopLeftLine1,3);
        PolyLine(hDC,TopLeftLine2,3);
        SetPixel(hDC,right-(2+x1),top+2+y1,cLightGrayColor);
        SetPixel(hDC,left+2+x1,bottom-(2+y1),cLightGrayColor);
        SelectObject(hDC,OldPen);
        DeleteObject(Pen);
      end
    else
      begin
        Pen := CreatePen(ps_Solid,1,cWhiteColor);
        OldPen := SelectObject(hDC,Pen);
        PolyLine(hDC,TopLeftLine1,3);
        if w > 24 then
          PolyLine(hDC,TopLeftLine2,3);
        SelectObject(hDC,OldPen);
        DeleteObject(Pen);
        Pen := CreatePen(ps_Solid,1,cDarkGrayColor);
        OldPen := SelectObject(hDC,Pen);
        PolyLine(hDC,BottomRightLine1,3);
        PolyLine(hDC,BottomRightLine2,3);
        SelectObject(hDC,OldPen);
        DeleteObject(Pen);
      end;
  end; {of with}
end; {tSingleBitmapButton.DrawItem}

{**************************************************************************}
{*  tMultiBitmapButton                                                    *}
{**************************************************************************}

constructor tMultiBitmapButton.Init(aParent: pWindowsObject; aID: Integer;
                                    X,Y: Integer; IsDefault: Boolean;
                                    aNormalBitmap,aDownBitmap,aDisabledBitmap: pChar);

var
  bm  : tBitMap;
  w,h : integer;

begin
  bmp := LoadBitmap(hInstance,aNormalBitmap);
  GetObject(bmp,sizeof(bm),@bm);
  tButton.Init(aParent,aID,'Dummy',x,y,bm.bmWidth,bm.bmHeight,IsDefault);
  Attr.Style := Attr.Style or bs_OwnerDraw;
  NormalBitmap := bmp;
  DownBitmap := LoadBitmap(hInstance,aDownBitmap);
  DisabledBitmap := LoadBitmap(hInstance,aDisabledBitmap);
end; {tMultiBitmapButton.Init}

destructor tMultiBitmapButton.Done;

begin
  tButton.Done;
  DeleteObject(NormalBitmap);
  DeleteObject(DownBitmap);
  DeleteObject(DisabledBitmap);
end; {tMultiBitmapButton.Done}

procedure tMultiBitmapButton.DrawItem(var Msg: tMessage);

var
  Down,
  Disabled : boolean;
  MemDC    : hDC;

begin
  with pDrawItemStruct(Msg.lParam)^, rcItem do begin
    if ItemAction = oda_Focus then
      exit;

    Down := ((ItemAction and oda_Select) > 0)
            and ((ItemState and ods_Selected) > 0);
    Disabled := (itemState and ods_Disabled <> 0)
                or (itemState and ods_Grayed <> 0);

    MemDC := CreateCompatibleDC(hDC);
    if Down then
      SelectObject(MemDC,DownBitmap)
    else
    if Disabled then
      SelectObject(MemDC,DisabledBitmap)
    else
      SelectObject(MemDC,NormalBitmap);
    BitBlt(hDC,left,top,right-left,bottom-top,MemDC,0,0,SrcCopy);
    DeleteDC(MemDC);
  end; {of with}
end; {tMultiBitmapButton.DrawItem}


end.
