Unit Ktbpos;
{$Define Test}
{	Remove the preceding line to activate the production compiler directives. }

{	Version 1.0.1.P.
	Created Sunday, January 1, 1995.

	Requires Borland Turbo Pascal version 6.0 or later to compile.

	Author:  Bruce J. Lackore of Kestrel Technologies.  
	Copyright (c) 1995 Bruce J. Lackore.  ALL RIGHTS RESERVED.
}

{$IFDEF Test}
	{$A+,B-,D+,F+,G-,I+,L+,O+,R+,S+,V-,X+}
{$ELSE}
	{$A+,B-,D-,F+,G-,I-,L-,O+,R-,S-,V-,X+}
{$ENDIF}

{	This unit contains an extended search routine developed as a result of an
	article read in PC Techniques.  In the original article, a set of routines
	was presented that allowed multiple string searches in a single source
	string and a boolean result generated from the tests.  This set of routines
	allows such searching with a single call AND allows for not only AND and OR
	processing, but NOT processing as well.  The programmer may change the two
	characters used as parentheses and the codes used as Boolean operators by
	using the Use_for_ops function.
}

Interface

Type
	String5										= String[5];

Procedure Use_for_ops(NLf_pn, NRt_pn, NAnd_op, NOr_op, NNot_op:  Char);

{	This procedure takes 5 characters and will replace the default operator
	characters as desired.  If NO change is desired for a particluar operator,
	simply use a space in place of the character or use the old character.
}

Function Fixup_srch_str(Srch_str:  String):  String;

{	This function prepares the Srch_str for use in the search engine of the
	BPos gadget.  It verifes that the correct number of paren pairs are found,
	that the entire Srch_str is enclosed in at least one pair of parens
	(necessary for the evaluation engine) and that no illegal operator
	combinations are in the Srch_str.  If successful, the function returns the
	"fixed up" Srch_str, otherwise it returns a null string, signifying failure.

	NOTE:	This function is called by EFixup_srch_str since the items performed
				here are necessary to both forms of BPos.
}

Function Evaluate_BPos_token_str(Srch_str:  String):  Boolean;

{	This function is simply an external call to the BPos Boolean evaluation
	engine.  If the programmer supplies a string in the correct form, this
	function will evaluate it and return the resultant Boolean value from the
	calculation.  Valid characters are:  T, F, &, |, (, ) where T and F are
	used in place of the strings to search for.  Naturally, if the programmer
	has changed the Trigger_chars, the new characters will be used in place of
	the defaults.

	NOTE:	This function AUTOMATICALLY makes a call to Fixup_srch_str.
}

Function BPos(Srch_str, Src_str:  String;
							Ignore_case, Call_fixup:  Boolean):  Boolean;

{	This function accepts a Srch_str to use as a blueprint to guide the search
	engine and a Src_str to look in.  If Ignore_case is set to True, both the
	Srch_str and the Src_str will be upper cased before processing begins.  If
	Call_fixup is set to True, a call to Fixup_srch_str will be made before
	processing the Srch_str.  If set to False, the function ASSUMES that the
	Srch_str is in correct form.  THERE IS NO ERROR CHECKING UNDER THESE
	CIRCUMSTANCES!  This option is useful, however, in that it will save a great
	deal of time when using the same Srch_str against many Src_str strings by
	not rechecking the Srch_str each time a search is made.  Valid Boolean
	operators are:

		& - And
		| - Or

	Parentheses are supported.  Evaluation of Boolean operators within parens is
	from Left to Right.  HOWEVER, processing of paren levels themselves is from
	Right to Left.  In addition, the And and Or operators are treated as if they
	have the SAME precedence!  This could cause problems if the programmer is
	thinking in terms of standard Boolean evaluation.  I.e. suppose a search
	string of (word1|word2&word3).  Normally, in Boolean algebra, word2&word3
	would be processed and the result ORed with word1.  NOT SO IN THIS FUNCTION!
	Since the search engine is looking only for characters and scans from Left
	to Right, the ACTUAL evaluation is word1|word2, the result ANDed with word3.
}

Function EFixup_srch_str(Srch_str:  String):  String;

{	This function allows the user to prepare for multiple calls to EBPos with
	the same Srch_str by fixing up the Srch_str one time, then allowing that
	string to be used by EBPos for all calls by setting Call_fixup to False.
	It performs the same function as Fixup_srch_str EXCEPT that there is
	additional code to process the NOT portion of the BPos functionality.
}

Function Evaluate_EBPos_token_str(Srch_str:  String):  Boolean;

{	This function simply uses the Boolean engine to process a string that is in
	the same form as that generated by EBPos when checking for strings.  Useful
	for parsing engines that evaluate flag settings (i.e. Skeleton).  Again,
	the only difference between this and the Evaluate_BPos_token_str function is
	the capability to perform NOT processing.  Naturally, if the programmer
	has changed the Trigger_chars, the new characters will be used in place of
	the defaults.

	NOTE:	A call is made to EFixup_srch_str AUTOMATICALLY.
}

Function EBPos(Srch_str, Src_str:  String;
								Ignore_case, Call_fixup:  Boolean):  Boolean;

{	This funtion accepts a Srch_str (used as the blueprint for searching) and a
	Src_string to look in.  Ignore_case will cause both the Srch_str and the
	Src_str to be up cased if set to True.  Call_fixup will cause the function
	to call the EFixup_srch_str function before processing the Srch_str to ensure
	that the Srch_str is in the proper form.  This function performs the same
	function as that of the BPos function except that it is capable of doing
	NOT processing as well.  The default Boolean operators are:

		& - And
		| - Or
		~ - Not

	Parentheses are supported.  Evaluation of Boolean operators within parens is
	from Left to Right.  HOWEVER, processing of paren levels themselves is from
	Right to Left.  In addition, the And and Or operators are treated as if they
	have the SAME precedence!  This could cause problems if the programmer is
	thinking in terms of standard Boolean evaluation.  I.e. suppose a search
	string of (word1|word2&word3).  Normally, in Boolean algebra, word2&word3
	would be processed and the result ORed with word1.  NOT SO IN THIS FUNCTION!
	Since the search engine is looking only for characters and scans from Left
	to Right, the ACTUAL evaluation is word1|word2, the result ANDed with word3.

	The NOT operator acts a little differently as well:

	The function first searches for all occurences of each phrase in the
	Srch_str and replaces them with a T or F depending on whether or not they
	were found in the Src_str.  NO NOT processing takes place at this time!
	Once all phrases are evaluated, the now-tokenized Srch_str is passed to the
	search engine (Process_EBPos_token_str).  At that point, all ~T and ~F
	tokens are converted to their final form.  Then a level of parens is
	extracted, stripped of parens and processed.  As this can create another
	~F or ~T, those are processed again.  This process repeats until all
	tokenized phrases are reduced to a single T or F, at which point the
	evaluation is made and the function returns the appropriate result.  The
	end result is that the NOT operator has precedence over AND and OR as in
	Boolean operations BUT the evaluation process is a little twisted, first
	evaluating at the individual search phrase level, THEN evaluating at the
	parenthese level.

	Please note:	The NOT operator must FOLLOW a Boolean operator (the
								Fixup_srch_str function ENSURES this), precede a left paren
								or, if only a single phrase is being searched for, it must be
								the FIRST character following the left paren (or the first
								character period, as the Fixup_srch_str function adds a set
								of parens if the user does not).
}

{ ************************************************************************** }

Implementation

Uses BPstr;

Const
	Lf_pn:										Char = '(';
	Rt_pn:										Char = ')';
	And_op:										Char = '&';
	Or_op:										Char = '|';
	Not_op:										Char = '~';
	Trigger_chars:						String5 = '()&|~';

Function Tokenize_srch_str(Srch_str, Src_str:  String):  String;

{	This function accepts the Srch_str in raw form and the Src_str.  Each
	phrase of the Srch_str (those portions that are not part of the Trigger_char
	set) is evaluated against the Src_str to test its existance.  The results of
	each test are placed in the Srch_str in place of the phrase that was
	searched for (i.e. Srch_str = TEST, Src_str = TEST OF PROGRAM, the resultant
	tokenizing replaces TEST with T as TEST was found in TEST OF PROGRAM).

	NOTE:	This function is used by BOTH Process_EBPos_token_str and
				Process_BPos_token_str.
}

	Var
		Token_str,
		Srch_phrase:						String;
		End_phrase:							Boolean;

	Begin  { Function Tokenize_srch_str }
		Token_str		:= '';
		Srch_phrase	:= '';
		End_phrase	:= False;
		While Srch_str <> '' Do
			Begin
				If PosC(Srch_str[1], Trigger_chars) Then
					Begin
						End_phrase := Srch_phrase <> '';
						If End_phrase Then
							Begin
								If Pos(Srch_phrase, Src_str) <> 0 Then
									Token_str := Token_str + 'T'
								Else
									Token_str := Token_str + 'F';
								Srch_phrase := '';
								End_phrase	:= False
							End;  { If End_phrase }
						Token_str := Token_str + Srch_str[1]
					End
				Else
					Srch_phrase := Srch_phrase + Srch_str[1];
				Delete(Srch_str, 1, 1)
			End;  { While Srch_str <> '' }
		Tokenize_srch_str := Token_str
	End;  { Function Tokenize_srch_str }

Function Process_tokenized_phrase(Srch_str:  String):  String;

{	This function accepts a tokenized phrase (a level of parens) and returns a
	single char (a T or a F) denoting the resultant Boolean processing of the
	tokenized phrase.

	NOTE:	This function is used by both Process_EBPos_token_str and
				Process_BPos_token_str.
}

	Var
		Lcv,
		Lf_para,
		Phrase_len:							Byte;
		Phrase:									String;
		Curr_answer,
		Do_and:									Boolean;

	Begin  { Function Process_tokenized_phrase }

		{	Find the leftmost left paren }

		Lf_para := Last_cpos(Lf_pn, Srch_str);

		{	Calculate the phrase length by finding the first right paren FOLLOWING
			the LAST left paren.
		}

		Phrase_len := Succ(Next_CPos(Rt_pn, Srch_str, Lf_para) - Lf_para);

		{	Get the resultant phrase INCLUDING the parens. }

		Phrase := Copy(Srch_str, Lf_para, Phrase_len);

		{	Now, dump the parens. }

		Dec(Phrase[0]);
		Delete(Phrase, 1, 1);

		{	Establish the initial condition of the resultant "answer". }

		Curr_answer := Phrase[1] = 'T';

		{	Since only 4 characters are actually processed by this portion of the
			routine, it is safe to make the processing of the else clause of the
			Case statement as brief as it is.  The 4 characters are T, F, the
			And_op and the Or_op.
		}

		For Lcv := 2 to Length(Phrase) Do
			Case Phrase[Lcv] of
				'T':	If Do_and Then
								Curr_answer := Curr_answer And True  { If And }
							Else
								Curr_answer := True;          { If Or }
				'F':	If Do_and Then                  { If And (Or stays T) }
								Curr_answer := False;
				Else
					Do_and := (Phrase[Lcv] = And_op)
			End;  { Case }

		{	Delete the old phrase }

		Delete(Srch_str, Lf_para, Phrase_len);

		{	Replace it with the evaluated result }

		If Curr_answer Then
			Insert('T', Srch_str, Lf_para)
		Else
			Insert('F', Srch_str, Lf_para);
		Process_tokenized_phrase := Srch_str
	End;  { Function Process_tokenized_phrase }

Function Process_EBPos_token_str(Srch_str:  String):  Boolean;

{	This function is actually the heart of the EBPos function.  It accepts the
	tokenized Srch_str as input and processes it using T for True, F for False
	and ~ for NOT and returns a Boolean value based on the evaluation of the
	input Srch_str.
}

	Procedure Process_nots;

	{	This subprocedure is used to search for specific instances of either the
		~T or ~F character sets.  It would be shorter to just look for the ~
		operator, but that might destroy ~( character sets - not good.
	}

		Var
			Where_at:								Byte;

		Begin  { Procedure Process_nots }
			Where_at := Pos(Not_op + 'F', Srch_str);
			While Where_at <> 0 Do
				Begin
					Delete(Srch_str, Where_at, 2);
					Insert('T', Srch_str, Where_at);
					Where_at := Pos(Not_op + 'F', Srch_str)
				End;  { While Where_at <> 0 }
			Where_at := Pos(Not_op + 'T', Srch_str);
			While Where_at <> 0 Do
				Begin
					Delete(Srch_str, Where_at, 2);
					Insert('F', Srch_str, Where_at);
					Where_at := Pos(Not_op + 'T', Srch_str)
				End  { While Where_at <> 0 }
		End;  { Procedure Process_nots }

	Begin  { Function Process_EBPos_token_str }
		Process_nots;
		Repeat

			{	Make a call to locate and process a tokenized phrase (a level of
				parens) and return the modified Srch_str.
			}

			Srch_str := Process_tokenized_phrase(Srch_str);

			{	Since another level of parens has been replaced with a single answer,
				this might generate a condition of ~T or ~F (by removing the previous
				~(B#B#B...#B) level), go ahead and process_nots again.
			}

			Process_nots
		Until Length(Srch_str) = 1;
		Process_EBPos_token_str := (Srch_str = 'T')
	End;  { Function Process_EBPos_token_str }

Function Process_BPos_token_str(Srch_str:  String):  Boolean;

{	This function is actually the heart of the BPos function.  It accepts the
	tokenized Srch_str as input and processes it using T for True, F for False
	and returns a Boolean value based on the evaluation of the input Srch_str.
}

	Var
		Lcv,
		Lf_para,
		Phrase_len:							Byte;
		Phrase:									String;
		Curr_answer,
		Do_and:									Boolean;

	Begin  { Function Process_BPos_token_str }
		Repeat

			{	Make a call to locate and process a tokenized phrase (a level of
				parens) and return the modified Srch_str.
			}

			Srch_str := Process_tokenized_phrase(Srch_str)

		Until Length(Srch_str) = 1;
		Process_BPos_token_str := (Srch_str = 'T')
	End;  { Function Process_BPos_token_str }

Procedure Use_for_ops;

	Begin  { Procedure Use_for_ops }
		If NLf_pn <> ' ' Then
			Lf_pn := NLf_pn;
		If NRt_pn <> ' ' THen
			Rt_pn := NRt_pn;
		If NAnd_op <> ' ' Then
			And_op := NAnd_op;
		If NOr_op <> ' ' Then
			Or_op := NOr_op;
		If NNot_op <> ' ' Then
			Not_op := NNot_op
	End;  { Procedure Use_for_ops }

Function Fixup_srch_str;

	Var
		Lf_para,
		Rt_para,
		How_many:								Integer;
		Where_at:								Byte;
		Srch_not_str:						String;

	Begin  {  Function Fixup_srch_str }
		Fixup_srch_str := '';

		{	Start the parentheses matching process.  Start by checking for No
			Parens.
		}

		If Not(PosC(Lf_pn, Srch_str)) And (Not(PosC(Rt_pn, Srch_str))) Then
			Srch_str := Lf_pn + Srch_str + Rt_pn;

		{	Now make sure that the entire Srch_str is enclosed in parens }

		If CPos(Lf_pn, Srch_str) <> 1 Then
			Srch_str := Lf_pn + Srch_str;
		If CPos(Rt_pn, Srch_str) <> Length(Srch_str) Then
			If Length(Srch_str) < 255 Then
				Srch_str := Srch_str + Rt_pn
			Else
				Exit;

		{	Now make sure that the correct number of each kind is available to make
			pairs of parens.

			THIS WILL NOT ENSURE THAT THE USER GETS WHAT THEY INTENDED!  ONLY THAT
			THE CORRECT NUMBER OF PARENS IS PRESENT, NOT PLACED, IS GUARANTEED!

		}

		Lf_para 	:= Cnt_ch(Lf_pn, Srch_str);					{	Count the parens }
		Rt_para 	:= Cnt_ch(Rt_pn, Srch_str);
		How_many 	:= Abs(Lf_para - Rt_para);     			{ Get the difference }
		If How_many > 0 Then
			If Rt_para < Lf_para Then
				Srch_str := Srch_str + Fill_str(Rt_pn, How_many)
			Else
				Srch_str := Fill_str(Lf_pn, How_many) + Srch_str;

		{	Check for the illegal values of (& &) (| |)  }

		If (Pos(Lf_pn + And_op, Srch_str) <> 0) Or
			(Pos(And_op + Rt_pn, Srch_str) <> 0) Or
			(Pos(Lf_pn + Or_op, Srch_str) <> 0) Or
			(Pos(Or_op + Rt_pn, Srch_str) <> 0) Then
				Exit;

		Fixup_srch_str := Srch_str
	End;  { Function Fixup_srch_str }

Function Evaluate_BPos_token_str;

	Begin  { Function Evaluate_BPos_token_str }
		Srch_str := EFixup_srch_str(Srch_str);
		If Srch_str <> '' Then
			Evaluate_BPos_token_str := Process_BPos_token_str(Srch_str)
		Else
			Evaluate_BPos_token_str := False
	End;  { Function Evaluate_BPos_token_str }

Function BPos;

	Begin  { Function BPos }
		Trigger_chars := Lf_pn + Rt_pn + And_op + Or_op;
		If Ignore_case Then
			Begin
				Srch_str	:= Up_cs(Srch_str);
				Src_str		:= Up_cs(Src_str)
			End;  { If Ignore_case }
		If Not(PosC(And_op, Srch_str) Or (PosC(Or_op, Srch_str))) Then
			BPos := Pos(Srch_str, Src_str) <> 0
		Else
			If Call_fixup Then
				Begin
					Srch_str := Fixup_srch_str(Srch_str);
					If Srch_str <> '' Then
						BPos :=
							Process_BPos_token_str(Tokenize_srch_str(Srch_str, Src_str))
					Else
						BPos := False
				End
			Else
				BPos := Process_BPos_token_str(Tokenize_srch_str(Srch_str, Src_str))
	End;  { Function BPos }

Function EFixup_srch_str;

	Var
		Lf_para,
		Rt_para,
		How_many:								Integer;
		Where_at:								Byte;
		Srch_not_str:						String;

	Begin  {  Function EFixup_srch_str }
		EFixup_srch_str := '';

		{	Make a call to perform the initial evaluation of the Srch_str and exit
			if an error was found.
		}

		Srch_str := Fixup_srch_str(Srch_str);
		If Srch_str = '' Then Exit;

		{	Make sure that no calls to ~& and ~| are made - correct them rather
			than exit with an error.
		}

		Where_at := Pos(Not_op + And_op, Srch_str);
		While Where_at <> 0 Do
			Begin
				Delete(Srch_str, Where_at, 2);
				Insert(And_op + Not_op, Srch_str, Where_at);
				Where_at := Pos(Not_op + And_op, Srch_str)
			End;  { While Where_at <> 0 }
		Where_at := Pos(Not_op + Or_op, Srch_str);
		While Where_at <> 0 Do
			Begin
				Delete(Srch_str, Where_at, 2);
				Insert(Or_op + Not_op, Srch_str, Where_at);
				Where_at := Pos(Not_op + Or_op, Srch_str)
			End;  { While Where_at <> 0 }

		{	Ensure next to last character is other than NOT operator, the last being
			the locallly or externally placed right parentheses.
		}

		If Srch_str[Pred(Length(Srch_str))] = Not_op Then Exit;

		{	Make sure that no NOT operator (~) is placed by itself in the middle
			of a search string.  Such a circumstance is ILLEGAL EXCEPT when located
			in the SECOND postion of the Srch_str, the first being taken up by the
			locally or externally placed left parentheses.  Legal character
			combinations are:

				&~
				|~
				~(
				(~

			Any other instance of the ~ operator is ILLEGAL and triggers an exit.
		}

		Srch_not_str	:= Srch_str;
		Where_at			:= CPos(Not_op, Srch_not_str);
		While Where_at <> 0 Do
			If (Srch_not_str[Pred(Where_at)] In [Lf_pn, And_op, Or_op]) Or
				(Srch_not_str[Succ(Where_at)] = Lf_pn) Then
					Begin
						Delete(Srch_not_str, Where_at, 1);
						Where_at := CPos(Not_op, Srch_not_str)
					End
			Else
				Exit;
		EFixup_srch_str := Srch_str
	End;  { Function EFixup_srch_str }

Function Evaluate_EBPos_token_str;

	Begin  { Function Evaluate_EBPos_token_str }
		Srch_str := EFixup_srch_str(Srch_str);
		If Srch_str <> '' Then
			Evaluate_EBPos_token_str := Process_EBPos_token_str(Srch_str)
		Else
			Evaluate_EBPos_token_str := False
	End;  { Function Evaluate_EBPos_token_str }

Function EBPos;

	Begin  { Function EBPos }
		Trigger_chars := Lf_pn + Rt_pn + And_op + Or_op + Not_op;
		If Ignore_case Then
			Begin
				Srch_str	:= Up_cs(Srch_str);
				Src_str		:= Up_cs(Src_str)
			End;  { If Ignore_case }
		If Not(PosC(And_op, Srch_str) Or (PosC(Or_op, Srch_str))) Then
			If PosC(Not_op, Srch_str) Then
				Begin
					Delete(Srch_str, CPos(Not_op, Srch_str), 1);
					EBPos := Pos(Srch_str, Src_str) = 0
				End
			Else
				EBPos := Pos(Srch_str, Src_str) <> 0
		Else
			If Call_fixup Then
				Begin
					Srch_str := EFixup_srch_str(Srch_str);
					If Srch_str <> '' Then
						EBPos :=
							Process_EBPos_token_str(Tokenize_srch_str(Srch_str, Src_str))
					Else
						EBPos := False
				End
			Else
				EBPos := Process_EBPos_token_str(Tokenize_srch_str(Srch_str, Src_str))
	End;  { Function EBPos }

End.  { Unit Ktbpos }