//----------------------------------------------------------------
// Copyright  Antony Corbett 1995. All Rights Reserved
// Author:	Antony Corbett
//				Compuserve	100277,330
//				INTERNET		antony@corbetteng.co.uk
//				Tel +44 (1926) 856131,  Fax +44 (1926) 856727
//
// Description:
//		TXListBox implementation.
//
//		TListBox derivatives that provide disabling of selected
//		list items, text colouring of items, horizontal scrolling
//		and dynamic switching between single & multiple selection.
//
//		History:
//			15 Dec 1995			Initial release
//
//			21 Dec 1995	 		Added support for Tab Stops
//									Added MakeSingleSel()
//									Removed oldCaret_
//									Added colored items support
//									Altered 'look' of single-sel TCheckedListBox
//
//       22 Dec 1995			Modified SetSel()
//									Added SetSelIndexes()
//									Added SetSelIndex()
//								 	Added SetSelItemRange()
//									Added SetSelString()
//								 	Added SetSelStrings()
//
//			29 Dec 1995			Fixed bug in TXListBox::SetSel()
//										(Thanks to Greg Bullock)
//									Changed TItemData from struct to class
//									Added enum TXListBoxMode
//									Added TXListBox::GetListBoxMode()
//									Added a TXListBox c'tor for a window
//									Modified Enable() to handle single-sel
//										lists when we disable the current
//										selection.
//									Modified Get/Set Selection members
//										to handle pseudo selection lists
//										better.
//
//			30 Dec 1995			Added GetNumTabs() member
//									Added GetTabs() member
//									Added TXButtonListBox class
//									Fixed bug in EvMeasureItem()
//										(cannot use HWindow of ctrl
//										or parent when handling this msg.
//										This only surfaced in Win32).
//
//----------------------------------------------------------------
#include <owl\owlpch.h>
#pragma hdrstop

#include "xlb.h"


//***************************************
//		TXListBox
//***************************************

DEFINE_RESPONSE_TABLE1(TXListBox, TListBox)

	// owl routes this message from parent back to control...
	EV_WM_MEASUREITEM,

	// The LBN_SELCHANGE notification is sent for a
	//	multiple-selection list box whenever the user
	// presses an arrow key, even if the selection
	// does not change....
	EV_NOTIFY_AT_CHILD(LBN_SELCHANGE, CmSelChange),
	EV_NOTIFY_AT_CHILD(LBN_KILLFOCUS, CmKillFocus),

	EV_WM_LBUTTONDOWN,
//   EV_WM_MOUSEMOVE,

END_RESPONSE_TABLE;



//----------------------------------------------------------------
// Description:
//		c'tor. Constructs a TXListBox control to be associated
//		with a TDialog.
//
// Arguments:
//		useHScroll = flag indicating if we should use a
//		horizontal scroll bar.
//
//----------------------------------------------------------------
TXListBox::TXListBox(TWindow* parent, int resourceId,
										BOOL useHScroll, // = FALSE
										TModule* module	// = 0
										)
: TListBox(parent, resourceId, module)
, textExtents_(*new SortedIntegers(20, 0, 20))	// store extents of strings
, hasHScroll_(useHScroll)
, tabs_(NULL)
, numTabs_(0)
, pseudoSingleSel_(FALSE)
{

}


//----------------------------------------------------------------
// Description:
//		c'tor. Constructs a TXListBox control to be associated
//		with a window
//
// Arguments:
//		useHScroll = flag indicating if we should use a
//		horizontal scroll bar.
//
//----------------------------------------------------------------
TXListBox::TXListBox(TWindow* Parent, int Id, int x, int y,
								int w, int h, BOOL useHScroll, // = FALSE
								TModule* module // = 0
								)
: TListBox(Parent, Id, x, y, w, h, module)
, textExtents_(*new SortedIntegers(20, 0, 20))	// store extents of strings
, hasHScroll_(useHScroll)
, tabs_(NULL)
, numTabs_(0)
, pseudoSingleSel_(FALSE)
{
Attr.Style |= LBS_OWNERDRAWFIXED;
Attr.Style |= LBS_HASSTRINGS;
}



//----------------------------------------------------------------
// Description:
//		d'tor. Deletes the integer array used to hold text extents.
//
//----------------------------------------------------------------
TXListBox::~TXListBox()
{
Destroy();
delete &textExtents_;
delete[] tabs_;
}


//****************************************************
//
// operations on individual list box items:
//
//		int AddString(const char *str);
//		int AddDisabledString(const char *str);
//		int InsertString(const char *str, int index);
//		int DeleteString(int index);
//		void Enable(int index, BOOL enable = TRUE);
//		int SetItemData(int index, uint32 data);
//		int SetSel(int index, BOOL select);
//		int SetSelIndexes(int* indexes, int numSelections, BOOL shouldSet);
//		int SetSelIndex(int index);
//		int SetSelItemRange(bool select, int first, int last);
//		int SetSelString(const char far* str, int searchIndex);
//		int SetSelStrings(const char far** prefixes, int numSelections, bool shouldSet);
//		void SetTextColor(int index, TColor color);
//		void ReSetTextColor(int index);
//
//****************************************************


//----------------------------------------------------------------
// Description:
//		Adds a string to the list.
//
// Arguments:
//		str = string to add
//
// Returns:
//		zero-based index of added string.
//----------------------------------------------------------------
int TXListBox::AddString(const char far *str)
{
// add the string...
int index = TListBox::AddString(str);

if(index >= 0)
	{
   if(hasHScroll_)
		{
		// store the extent of string in
		// sorted container...
		StoreStrExtent(str);
		}
	// now the item data...
	TListBox::SetItemData(index, (uint32)(new TItemData()));
	}
return index;
}


//----------------------------------------------------------------
// Description:
//		Adds a string to the list and disables it.
//
// Arguments:
//		str = string to add
//
// Returns:
//		zero-based index of added string.
//----------------------------------------------------------------
int TXListBox::AddDisabledString(const char far *str)
{
int index = AddString(str);
if(index >= 0) Enable(index, FALSE);
return index;
}


//----------------------------------------------------------------
// Description:
//		Inserts a string at the given position in the list.
//
// Arguments:
//		str = string to insert
//		index = zero-based index giving insertion position.
//		If index == -1 the string is appended to end of list.
//
// Returns:
//		zero-based index of inserted string.
//----------------------------------------------------------------
int TXListBox::InsertString(const char far *str, int index)
{
int actualIndex = TListBox::InsertString(str, index);

if(actualIndex >= 0)
	{
	if(hasHScroll_)
		{
		// store the extent of string
		// in sorted container...
		StoreStrExtent(str);
		}
	// now the item data...
	TListBox::SetItemData(actualIndex, (uint32)(new TItemData()));
	}
return actualIndex;
}


//----------------------------------------------------------------
// Description:
//		Deletes a string in the list.
//
// Arguments:
//		index = zero-based index of string to remove.
//
// Returns:
//		Number of remaining items or -ve if an error.
//----------------------------------------------------------------
int TXListBox::DeleteString(int index)
{
// first remove any item data...
TItemData *id = GetItemDataPtr(index);
CHECK(id);
delete id;

if(hasHScroll_)
	{
	// find extent of string to be deleted...
	char *tempStr = new char[256];
	GetString(tempStr, index);

	int *length = new int;
	*length = FindTextExtent(tempStr);
	delete[] tempStr;

	textExtents_.Detach(length, TShouldDelete::Delete);
	delete length;

	UpdateHorizontalExtent();
	}

return (TListBox::DeleteString(index));
}


//----------------------------------------------------------------
// Description:
//		Changes the enable state of an item in the list.
//
// Arguments:
//		index = zero-based index of item to enable/disable.
//			If index == -1 the all items are enabled/disabled.
//
//		enable = TRUE/FALSE flag indicating if item
//		should be enabled or disabled. Defaults to TRUE
//
//----------------------------------------------------------------
void TXListBox::Enable(int index, BOOL enable/* = TRUE*/)
{
TXListBoxMode mode = GetListBoxMode();

if(index == -1)
	{
	for(int n=0; n<GetCount(); n++)
		{
		if(GetEnabled(n) != enable)
			{
			// change in status required...
			GetItemDataPtr(n)->SetEnabled(enable);
			if(mode == SINGLE)
				{
				// native single-sel listbox...
				if(GetSelIndex() == n)
					TListBox::SetSelIndex(-1);	// clear selection.
				}
			else TListBox::SetSel(n, FALSE);
			}
		}
	Invalidate();
	}
else
	{
	if(GetEnabled(index) != enable)
		{
		// change in status required...
		GetItemDataPtr(index)->SetEnabled(enable);

		// if the enable status of an item changes,
		// it should never result in selection, so
		// make sure we're not selected...

		if(mode == SINGLE)
			{
			// native single-sel listbox...
			if(GetSelIndex() == index)
				TListBox::SetSelIndex(-1);	// clear selection.
			}
		else TListBox::SetSel(index, FALSE);

		// now update the display...
		TRect rc;
		int rv = GetItemRect(index, rc);
		if(rv != LB_ERR) InvalidateRect(rc);
		}
	}
}


//----------------------------------------------------------------
// Description:
//		Sets list item data.
//		NB - we've replaced the standard behaviour to allow us
//		to store the item's enable status, color etc
//
// Arguments:
//		index = zero-based index of item.
//		data = 32-bit data to set.
//
// Returns:
//		LB_ERR if an error.
//----------------------------------------------------------------
int TXListBox::SetItemData(int index, uint32 data)
{
// the data is actually a ptr to a TItemData struct.
GetItemDataPtr(index)->SetData(data);
return !LB_ERR;
}


//----------------------------------------------------------------
// Description:
//		Selects an item at the position specified in index.
//
//		NB - Only for multiple-selection list boxes and
//		pseudo single selection listboxes.
//
//		Enforces single selection when used with a pseudo single
//		selection listbox, i.e. if you use SetSel to select an
//		item in a pseudo single selection listbox, any existing
//		selection is removed.
//
// Arguments:
//		index = zero-based index of item to select. If -1, then the
//			select status is added to all strings.
//		select = TRUE/FALSE
//
// Returns:
//		LB_ERR if an error.
//
//		It is an error to select all items in a pseudo single
//		selection listbox.
//
//		If the item is disabled, we do not select and no error
//		is returned.
//----------------------------------------------------------------
int TXListBox::SetSel(int index, BOOL select)
{
TXListBoxMode mode = GetListBoxMode();

if(mode == SINGLE)	// real single-sel listbox...
	return (TListBox::SetSel(index, select));

if(index == -1)
	{
	// want to select/deselect all items...

	if(mode == PSEUDO_SINGLE && select)
		return LB_ERR;	// can't select all items in pseudo-single mode.

	for(int n=0; n<GetCount(); n++)
		{
		// only change selection status if item is enabled...
		if(GetEnabled(n))
			TListBox::SetSel(n, select);
		}
	}
else
	{
	if(!(select && !GetEnabled(index)))
		return (TListBox::SetSel(index, select));
	}
return !LB_ERR;
}


//----------------------------------------------------------------
// Description:
//		For multiple-selection list boxes. Selects or deselects
//		the strings in the associated list box at the indexes
//		specified in the Indexes array.
//
//		NB - the only valid operation on pseudo selection
//		listboxes is to deselect all items (numSelections < 0).
//
// Arguments:
//		indexes = array of items indexes to select/deselect.
//		numSelections = number of items in indexes array.
//				If numSelections is less than 0, all strings
//				are selected or deselected.
//		shouldSet = flag. If true, the indexed strings are
//				selected and highlighted; if false the highlight
//				is removed and they are no longer selected.
//
// Returns:
//		Returns the number of strings successfully selected
//		or deselected (LB_ERR for single-selection list boxes).
//
//		A negative value is returned on failure.
//
//		It is an error to try to select all items
//		in a pseudo single selection listbox.
//
//----------------------------------------------------------------
int TXListBox::SetSelIndexes(int* indexes, int numSelections, BOOL shouldSet)
{
switch(GetListBoxMode())
	{
	case SINGLE:
		return LB_ERR;

	case PSEUDO_SINGLE:
		if(numSelections < 0)
			{
			if(shouldSet) return LB_ERR;	// can't select all items.
			else
				{
				int rv = (GetSelIndex() >= 0);
				SetSel(-1, FALSE); // we _can_ deselect all items...
				return rv;;
				}
			}
		return LB_ERR;

	case MULTIPLE:
		return (TListBox::SetSelIndexes(indexes, numSelections, shouldSet));
	};

return LB_ERR;
}


//----------------------------------------------------------------
// Description:
//		For single-selection list boxes. Forces the selection
//		of the item at the zero-based index.
//
//		NB - may also be used with pseudo single selection
//		listboxes.
//
// Arguments:
//		index = zero-based index of item to select.
//			If index is -1, the list box is cleared
//			of any selection.
//
// Returns:
//		A negative number if an error occurs.
//----------------------------------------------------------------
int TXListBox::SetSelIndex(int index)
{
switch(GetListBoxMode())
	{
	case MULTIPLE:
		return LB_ERR;

	case PSEUDO_SINGLE:
		// clear all selections...
		TListBox::SetSelIndexes(NULL, -1, FALSE);
		return (SetSel(index, TRUE));

	case SINGLE:
		return (TListBox::SetSelIndex(index));
	};

return LB_ERR;
}



//----------------------------------------------------------------
// Description:
//		Selects the range of items specified from first to last.
//
// Arguments:
//		select = flag indicating if specified items should be
//			selected or deselected.
//		first = zero-based index of first item to select/deselect.
//		last = zero-based index of last item to select/deselect.
//
// Returns:
//		LB_ERR if error.
//----------------------------------------------------------------
int TXListBox::SetSelItemRange(BOOL select, int first, int last)
{
if(GetListBoxMode() != MULTIPLE) return LB_ERR;
return (TListBox::SetSelItemRange(select, first, last));
}


//----------------------------------------------------------------
// Description:
//		For single-selection list boxes. Forces the selection
//		of the first item beginning with the text supplied in
//		str that appears beyond the position (starting at 0)
//		supplied in searchIndex.
//
//		NB - can also be used with Pseudo single selection
//		listboxes.
//
// Arguments:
//		str = test to search for.
//		searchIndex = zero-based index of item in list
//				to start the search at. If searchIndex is -1,
//				the entire list is searched, beginning with the
//				first item.
//
// Returns:
//		The position of the newly selected item, or LB_ERR
//		in the case of an error.
//----------------------------------------------------------------
int TXListBox::SetSelString(const char far* str, int searchIndex)
{
switch(GetListBoxMode())
	{
	case SINGLE:
		return (TListBox::SetSelString(str, searchIndex));

	case MULTIPLE:
		return LB_ERR;

	case PSEUDO_SINGLE:
		int index = FindString(str, searchIndex);
		if(index != LB_ERR)
			{
			if(GetEnabled(index))
				TListBox::SetSel(index, TRUE);
			}
		return index;
	};

return LB_ERR;
}


//----------------------------------------------------------------
// Description:
//		For multiple-selection list boxes, selects the strings
//		in the associated list box that begin with the prefixes
//		specified in the prefixes array. For each string the
//		search begins at the beginning of the list and continues
//		until a match is found or until the list has been completely
//		traversed.
//
// Arguments:
//		If shouldSet is true, the matched strings are selected
//		and highlighted; if false, the highlight is removed
//		from the matched strings and they are no longer selected.
//
//		If numSelections is less than 0, all strings are selected
//		or deselected.
//
// Returns:
//		The number of strings successfully selected or deselected
//		(LB_ERR for single-selection list boxes or failure)
//----------------------------------------------------------------
int TXListBox::SetSelStrings(const char far** prefixes, int numSelections, bool shouldSet)
{
if(GetListBoxMode() != MULTIPLE) return LB_ERR;
return (TListBox::SetSelStrings(prefixes, numSelections, shouldSet));
}




//----------------------------------------------------------------
// Description:
//		Sets the color of an item or all items in the list
//
// Arguments:
//		index = zero-based index of item to color.
//			If index == -1 then all items are set to the
//			specified colour
//		color = color of item text
//
//----------------------------------------------------------------
void TXListBox::SetTextColor(int index, TColor color)
{
if(index == -1)
	{
	for(int n=0; n<GetCount(); n++)
		{
		TItemData *id = GetItemDataPtr(n);
		id->SetTextColor(color);
		}
	Invalidate();
	}
else
	{
	TItemData *id = GetItemDataPtr(index);
	id->SetTextColor(color);

	TRect rc;
	if(GetItemRect(index, rc) != LB_ERR) InvalidateRect(rc);
	}
}


//----------------------------------------------------------------
// Description:
//		Sets the color of an item or all items to the
//		default system color.
//
// Arguments:
//		index = zero-based index of item to reset color for.
//			If index == -1 then all items are reset.
//
//----------------------------------------------------------------
void TXListBox::ResetTextColor(int index)
{
if(index == -1)
	{
	for(int n=0; n<GetCount(); n++)
		{
		TItemData *id = GetItemDataPtr(n);
		id->SetUseSysColor(TRUE);
		}
	Invalidate();
	}
else
	{
	TItemData *id = GetItemDataPtr(index);
	id->SetUseSysColor(TRUE);

	TRect rc;
	if(GetItemRect(index, rc) != LB_ERR) InvalidateRect(rc);
	}
}


//****************************************************
//
// operations on listbox:
//
//			void ClearList();
//			BOOL SetTabStops(int numTabs, int far* tabs);
//			BOOL MakeSingleSel(BOOL single);
//
//****************************************************

//----------------------------------------------------------------
// Description:
//		Clears the list of all items. We override to make sure
//		all the TItemData objects are deleted and the hscrolling
//    cleaned up.
//
//----------------------------------------------------------------
void TXListBox::ClearList()
{
for(int n=0; n<GetCount(); n++)
	delete GetItemDataPtr(n);

if(hasHScroll_)
	{
	textExtents_.Flush();
	UpdateHorizontalExtent();
	}

TListBox::ClearList();
}


//----------------------------------------------------------------
// Description:
//		Sets tab stops.
//
// Arguments:
// 	numTabs = number of tab stops
//		tabs = array of integers representing the tab positions
//
// Returns:
//		non-zero if all tab stops could be set
//----------------------------------------------------------------
BOOL TXListBox::SetTabStops(int numTabs, int far* tabs)
{
BOOL rv = !0;

numTabs_ = numTabs;
delete[] tabs_;
tabs_ = new int[numTabs];

// check they're in size order...
for(int n=0, lastTab = -1; n<numTabs; n++)
	{
	if(tabs[n] <= lastTab)
		{
		// out of sequence!
		rv = 0;
		numTabs_ = n;	// modify num tabs
		break;
		}
	lastTab = tabs[n];
	tabs_[n] = tabs[n];
	}

return rv;
}


//----------------------------------------------------------------
// Description:
//		Forces a multi-selection list box into pseudo single
//		selection mode.
//
//	Returns:
//    non-zero if change could be made. NOTE - cannot change
//		a native single-sel list into a multi-sel.
//----------------------------------------------------------------
BOOL TXListBox::MakeSingleSel(BOOL single /* = TRUE */)
{
BOOL rv = !0;

if(single)
	{
	if(IsMultipleSel())
		{
		pseudoSingleSel_ = TRUE;
		CmSelChange();
		}
	}
else
	{
	// want to convert to multi-sel...
	if(IsMultipleSel())
		pseudoSingleSel_ = FALSE;
	else rv = 0;
	}

Invalidate();
return rv;
}



//****************************************************
//
// query individual items:
//
//		uint32 GetItemData(int index) const;
//		BOOL GetSel(int index) const;
//		BOOL GetEnabled(int index) const;
//  	TItemData *GetItemDataPtr(int index) const;
//
//****************************************************


//----------------------------------------------------------------
// Description:
//		Gets the 32-bit data stored for the specified item
//
// Arguments:
//		index = zero-based index of item to get data for
//
// Returns:
//		32-bit data value
//----------------------------------------------------------------
uint32 TXListBox::GetItemData(int index) const
{
// the actual data is a ptr to a TItemData struct
// which contains a 32-bit member 'data'
return (GetItemDataPtr(index)->GetData());
}


//----------------------------------------------------------------
// Description:
//		Returns the selection state of the listbox item at
//		given index.
//
// Arguments:
//		index = zero-based index of item to test.
//
// Returns:
//		TRUE/FALSE
//----------------------------------------------------------------
BOOL TXListBox::GetSel(int index) const
{
// if the item _is_ selected but is disabled,
// we say it isn't selected.
return (TListBox::GetSel(index)?
				GetEnabled(index) : FALSE);
}



//----------------------------------------------------------------
// Description:
//		Gets the enable status of an item.
//
// Arguments:
//		index = zero-based index of item to get status for.
//
// Returns:
//		TRUE/FALSE
//----------------------------------------------------------------
BOOL TXListBox::GetEnabled(int index) const
{
return (GetItemDataPtr(index)->GetEnabled());
}



//----------------------------------------------------------------
// Description:
//		Gets a ptr to the TItemData object associated with
//		the given item. We retain a ptr to the TItemData object
//		in the 32-bit item data.  
//
// Arguments:
//		zero-based index of item to get TItemData object for.
//
// Returns:
//		TItemData ptr
//----------------------------------------------------------------
TItemData *TXListBox::GetItemDataPtr(int index) const
{
return ((TItemData *)(TListBox::GetItemData(index)));
}



//****************************************************
//
// query:
//
//		int GetSelCount() const;
//		int GetSelIndex() const;
//		int GetSelIndexes(int* indexes, int maxCount) const;
//		BOOL IsMultipleSel() const; (protected)
//		TXListBoxMode GetListBoxMode() const;
//
//****************************************************


//----------------------------------------------------------------
// Description:
//		Returns the number of selected items in the list.
//
//----------------------------------------------------------------
int TXListBox::GetSelCount() const
{
if(GetListBoxMode() == SINGLE)
	return (GetSelIndex() < 0 ? 0 : 1);	// We've overridden GetSelIndex

// multi-sel listboxes...
int tot = 0;
for(int n=0; n<GetCount(); n++)
	{
	// GetSel is overridden so that
	// we only count those items that
	// are selected _and_ enabled
	if(GetSel(n)) tot++;
	}
return tot;
}

//----------------------------------------------------------------
// Description:
//		For single-sel listboxes. Returns the index of the
//		current selection.
//
//		Can also be used for pseudo single selection listboxes.
//
// Returns:
//		zero-based index of current selection or -ve number if
//		no selection.
//----------------------------------------------------------------
int TXListBox::GetSelIndex() const
{
int index = -1;

if(GetListBoxMode() == PSEUDO_SINGLE)
	{
	int numSels = GetSelIndexes(&index, 1);
   if(numSels == 0) index = -1; 
	}
else
	{
	index = TListBox::GetSelIndex();
	}

if(index != -1)
	return (GetEnabled(index)? index : -1);
else return -1;
}



//----------------------------------------------------------------
// Description:
//		For multiple-selection list boxes. Fills the indexes
//		array with the indexes of up to maxCount selected
//		strings.
//
//		Can also be used for pseudo single selection listboxes.
//
// Arguments:
//		indexes = array of indexes to be retrieved (retval).
//		maxCount = max number of indexes to retrieve.
//
// Returns:
//		Number of items put in indexes or
//		LB_ERR for single-sel listbox.
//----------------------------------------------------------------
int TXListBox::GetSelIndexes(int far *indexes, int maxCount) const
{
if(GetListBoxMode() == SINGLE)
	return LB_ERR;

int tot = 0;
for(int n=0; n<GetCount() && tot < maxCount; n++)
	{
	// GetSel is overridden so that
	// we only count those items that
	// are selected _and_ enabled
	if(GetSel(n)) indexes[tot++] = n;
	}
return tot;
}



//----------------------------------------------------------------
// Description:
//		Determines if the list box has the multiple select style.
//		i.e. is a native multiple selection list.
//
// Returns:
//		TRUE/FALSE
//----------------------------------------------------------------
BOOL TXListBox::IsMultipleSel() const
{
return ((Attr.Style & (LBS_MULTIPLESEL | LBS_EXTENDEDSEL)) != 0);
}



//----------------------------------------------------------------
// Description:
//		Determines the listbox mode; single-selection,
//		multiple-selection or pseudo-single selection.
//
// Returns:
//		SINGLE; MULTIPLE or PSEUDO-SINGLE
//----------------------------------------------------------------
TXListBox::TXListBoxMode TXListBox::GetListBoxMode() const
{
if(IsMultipleSel())	// native multiple selection list
	return (pseudoSingleSel_? PSEUDO_SINGLE : MULTIPLE);

return SINGLE;
}



//****************************************************
// protected methods...
//****************************************************

//----------------------------------------------------------------
// Description:
//		Cleans up interface elements
//
//----------------------------------------------------------------
void TXListBox::CleanupWindow()
{
ClearList();
TListBox::CleanupWindow();
}


//----------------------------------------------------------------
// Description:
//		Finds the extent of a given listbox string. Used in
//		calculating horizontal scrolling.
//
// Arguments:
//		str = string to find extent of
//
// Returns:
//		extent of string in logical units
//----------------------------------------------------------------
int TXListBox::FindTextExtent(const char far *str)
{
TSize extent;

TClientDC dc(HWindow);
HFONT hfont = GetWindowFont();

if(hfont)	// non system font...
	dc.SelectObject(hfont);

extent = dc.GetTabbedTextExtent(str, lstrlen(str), numTabs_, tabs_);

if(hfont) dc.RestoreFont();

return extent.cx;
}


//----------------------------------------------------------------
// Description:
//    Updates the extent of horizontal scrolling
//
//----------------------------------------------------------------
void TXListBox::UpdateHorizontalExtent(void)
{
int greatestExtent;

// find the extent of the largest string and
// set the horizontal extent accordingly...
int lastElement = textExtents_.GetItemsInContainer() - 1;
if(lastElement < 0)
	{
	// no elements in listbox.
	greatestExtent = 0;
	}
else
	greatestExtent = *(textExtents_[lastElement]);

// add a bit of space so that last char is visible
// when we scroll to right...
TClientDC dc(HWindow);
greatestExtent += (dc.GetTextExtent("X", 1)).cx;

// if longest str fits completely then scroll
// listbox to the left, so Windows will hide
// the scrollbar...
TRect rect;
GetClientRect(rect);

int listWidth = rect.right - rect.left;
if(listWidth >= greatestExtent)
	HandleMessage(WM_HSCROLL, SB_TOP, 0);

SetHorizontalExtent(greatestExtent);
}






//----------------------------------------------------------------
// Description:
//		Stores the extent of the given string in our integer array.
//		Used by horizontal scrolling mechanism.
//
// Arguments:
//		str = string to store extent of.
//
//----------------------------------------------------------------
void TXListBox::StoreStrExtent(const char far *str)
{
int *length = new int;
*length = FindTextExtent(str);
textExtents_.Add(length);
UpdateHorizontalExtent();
}



//----------------------------------------------------------------
// Description:
//		Responds to notification to draw entire control (actually
//		just a single item in the list)
//
//----------------------------------------------------------------
void TXListBox::ODADrawEntire(DRAWITEMSTRUCT &drawInfo)
{
// get text to display...
int len = TListBox::GetStringLen(drawInfo.itemID);
char *buffer = new char[len+1];

TListBox::GetString(buffer, drawInfo.itemID);
DrawListItem(drawInfo, buffer);

delete[] buffer;
}


//----------------------------------------------------------------
// Description:
//		Responds to notification that focus has shifted to or from
//		an item.
//
//----------------------------------------------------------------
void TXListBox::ODAFocus(DRAWITEMSTRUCT &drawInfo)
{
TDC DC(drawInfo.hDC);
TRect rc(drawInfo.rcItem);
DC.DrawFocusRect(rc);
}


//----------------------------------------------------------------
// Description:
//		Responds to notification that selection state of item
//		has changed
//
//----------------------------------------------------------------
void TXListBox::ODASelect(DRAWITEMSTRUCT &drawInfo)
{
ODADrawEntire(drawInfo);
}


//----------------------------------------------------------------
// Description:
//		Handles WM_MEASUREITEM msg. owl sends this message from
//		parent back to the control.
//
//----------------------------------------------------------------
void TXListBox::EvMeasureItem(UINT /*ctrlId*/, MEASUREITEMSTRUCT &measure)
{
// NB - can't used HWindow since HWindow will be 0
// when first called...
TScreenDC dc;

// only good if using system font...
dc.SelectStockObject(SYSTEM_FONT);

TEXTMETRIC tm;
dc.GetTextMetrics(tm);

// use tmHeight to give a larger spacing,
// but use tmAscent to duplicate Windows behaviour...
measure.itemHeight = tm.tmAscent;

dc.RestoreFont();
}


//----------------------------------------------------------------
// Description:
//		Query the background color to use for given item.
//
// Returns:
//		Color to use.
//----------------------------------------------------------------
TColor TXListBox::QueryBkColor(DRAWITEMSTRUCT &drawInfo) const
{
if((drawInfo.itemState & ODS_SELECTED) &&
				GetEnabled(drawInfo.itemID))
	{
	// item is selected and
	// not disabled and
	// it's not a pseduo single-sel list with
	// this item unfocused...

	return TColor(::GetSysColor(COLOR_HIGHLIGHT));
	}

else
	{
	// light gray background if bwcc...
	return (GetApplication()->Ctl3dEnabled() ||
					GetApplication()->BWCCEnabled()?
						TColor(192,192,192) :
							TColor(::GetSysColor(COLOR_WINDOW)));
	}
}




//----------------------------------------------------------------
// Description:
//		Query the text color to use for given item.
//
// Returns:
//		Text color.
//----------------------------------------------------------------
TColor TXListBox::QueryTextColor(DRAWITEMSTRUCT &drawInfo) const
{
TColor color;

if(!GetEnabled(drawInfo.itemID))	// item disabled
	{
	// note we ignore the case where the device doesn't
	// support solid gray...
	COLORREF gray = ::GetSysColor(COLOR_GRAYTEXT);
	TDC DC(drawInfo.hDC);
	if(DC.GetBkColor() == TColor(gray))
		{
		// listbox background is same color as graytext
		// so choose the other gray...
		gray = (DC.GetBkColor() == TColor::LtGray?
						TColor::Gray : TColor::LtGray);
		}
	color = gray;
	}
else
	{
	TItemData *id = GetItemDataPtr(drawInfo.itemID);

	color = (drawInfo.itemState & ODS_SELECTED?
			TColor(::GetSysColor(COLOR_HIGHLIGHTTEXT)) :
				(id->GetUseSysColor()?
					TColor(::GetSysColor(COLOR_WINDOWTEXT)) :
						id->GetTextColor()));
	}
return color;
}



//----------------------------------------------------------------
// Description:
//		Draws an item in the list.
//
// Arguments:
//		str = text to draw.
//
//----------------------------------------------------------------
void TXListBox::DrawListItem(DRAWITEMSTRUCT &drawInfo,
														const char far *str)
{
TDC DC(drawInfo.hDC);
TRect rc(drawInfo.rcItem);

// first erase the item...
TBrush brush(QueryBkColor(drawInfo));
DC.FillRect(rc, brush);

// make some space between left border of listbox and
// first character of item. (This seems to match the space
// allowed for in a standard listbox).
rc.left += (2 * ::GetSystemMetrics(SM_CXBORDER));
DrawText(drawInfo, rc, str);
}


//----------------------------------------------------------------
// Description:
//		Draws an item's text as part of the DrawListItem operation.
//
// Arguments:
//		rc = bounding rect
//		str = string to draw
//
//----------------------------------------------------------------
void TXListBox::DrawText(DRAWITEMSTRUCT &drawInfo,
									const TRect& rc,
									const char far *str)
{
TDC DC(drawInfo.hDC);
DC.SetBkMode(TRANSPARENT);

DC.SetTextColor(QueryTextColor(drawInfo));

// need to take care of tabs...
DC.TabbedTextOut(rc.TopLeft(), str, -1,
							GetNumTabs(), GetTabs(), 0);
}



void TXListBox::CmKillFocus()
{
static BOOL forParent = FALSE; // to prevent infinite loop

if(forParent) DefaultProcessing();
else
	{
   // clean up the focus rect...
	TRect rc;
	if(GetItemRect(GetCaretIndex(), rc) != LB_ERR)
		InvalidateRect(rc);

	forParent = TRUE;

	// notify parent...
	Parent->SendMessage(WM_COMMAND, Attr.Id,
							MAKELPARAM(HWindow, LBN_KILLFOCUS));
	forParent = FALSE;
	}
}


//----------------------------------------------------------------
// Description:
//		Responds to LBN_SELCHANGE
//
//----------------------------------------------------------------
void TXListBox::CmSelChange()
{
static BOOL forParent = FALSE; // to prevent infinite loop

if(forParent) DefaultProcessing();
else
	{
	if(GetListBoxMode() == PSEUDO_SINGLE)
		{
		// native multiple selection list but
		// in pseudo single sel mode...

		// deselect all...
		TListBox::SetSelIndexes(NULL, -1, FALSE);

		// and just select the item with focus...
		TListBox::SetSel(GetCaretIndex(), TRUE);
		}

	forParent = TRUE;
	// notify parent...
	if(Attr.Style & LBS_NOTIFY)
			Parent->SendMessage(WM_COMMAND, Attr.Id,
							MAKELPARAM(HWindow, LBN_SELCHANGE));
	forParent = FALSE;
	}
}



//----------------------------------------------------------------
// Description:
//		Hook here to remove unpleasant drawing
//
//----------------------------------------------------------------
void TXListBox::EvLButtonDown(uint modkeys, TPoint& point)
{
if(GetListBoxMode() == PSEUDO_SINGLE)
	{
	// native multiple selection list but
	// in pseudo single sel mode...

	// deselect all...
	TListBox::SetSelIndexes(NULL, -1, FALSE);
	}
TListBox::EvLButtonDown(modkeys, point);
}



//void TXListBox::EvMouseMove(UINT modKeys, TPoint& point)
//{

//TListBox::EvMouseMove(modKeys, point);
//}



