//----------------------------------------------------------------
// Copyright  Antony Corbett 1996. All Rights Reserved
// Author:	Antony Corbett
//				Compuserve	100277,330
//				INTERNET		antony@corbetteng.co.uk
//				Tel +44 (1926) 856131,  Fax +44 (1926) 856727
//
// Description:
//		TXButtonListBox implementation.
//
//		TXButtonListBox provides 'button-style' selection.
//
//----------------------------------------------------------------

#include <owl\owlpch.h>
#pragma hdrstop

#include "xblb.h"



//***************************************
//		TXButtonListBox
//***************************************

DEFINE_RESPONSE_TABLE1(TXButtonListBox, TXListBox)

	EV_WM_MEASUREITEM,

END_RESPONSE_TABLE;


//----------------------------------------------------------------
// Description:
//		c'tor. Constructs a TXButtonListBox control to be associated
//		with a TDialog.
//
// Arguments:
//		useHScroll = flag indicating if we should use a
//		horizontal scroll bar.
//
//----------------------------------------------------------------
TXButtonListBox::TXButtonListBox(TWindow* parent,
												int resourceId,
												TModule* module	// = 0
												)
: TXListBox(parent, resourceId, FALSE, module)
{

}


//----------------------------------------------------------------
// Description:
//		Query the background color to use for given item.
//
// Returns:
//		Color to use.
//----------------------------------------------------------------
TColor TXButtonListBox::QueryBkColor(DRAWITEMSTRUCT &/*drawInfo*/) const
{
return (::GetSysColor(COLOR_BTNFACE)); 	// usually ltGray
}


//----------------------------------------------------------------
// Description:
//		Query the text color to use for given item.
//
// Returns:
//		Color to use.
//----------------------------------------------------------------
TColor TXButtonListBox::QueryTextColor(DRAWITEMSTRUCT &drawInfo) const
{
TColor color;

if(!GetEnabled(drawInfo.itemID))	// item disabled.
	color = TXListBox::QueryTextColor(drawInfo);
else
	{
	TItemData *id = GetItemDataPtr(drawInfo.itemID);

	color = (id->GetUseSysColor()?
						TColor(::GetSysColor(COLOR_BTNTEXT)) :	// usually black
							id->GetTextColor());
	}

return color;
}


//----------------------------------------------------------------
// Description:
//		Handles WM_MEASUREITEM msg. owl sends this message from
//		parent back to the control.
//
//		Make the height a bit larger than normal to allow for
//		button drawing.
//
//----------------------------------------------------------------
void TXButtonListBox::EvMeasureItem(UINT /*ctrlId*/, MEASUREITEMSTRUCT &measure)
{
// NB - can't used HWindow since HWindow will be 0
// when first called...
TScreenDC dc;

// only good if using system font...
dc.SelectStockObject(SYSTEM_FONT);

TEXTMETRIC tm;
dc.GetTextMetrics(tm);

measure.itemHeight = tm.tmHeight + 5;

dc.RestoreFont();
}



//----------------------------------------------------------------
// Description:
//		Draws an item's text as part of the DrawListItem operation.
//
//		Override TXListBox behaviour to centre text etc.
//
// Arguments:
//		rc = bounding rect
//		str = string to draw
//
//----------------------------------------------------------------
void TXButtonListBox::DrawText(DRAWITEMSTRUCT &drawInfo,
									const TRect& rc,
									const char far *str)
{
TDC DC(drawInfo.hDC);
DC.SetBkMode(TRANSPARENT);

uint oldAlign = DC.SetTextAlign(TA_CENTER);

DC.SetTextColor(QueryTextColor(drawInfo));

// truncate if too large...
int numCharsToDraw = lstrlen(str);
while(DC.GetTabbedTextExtent(str, numCharsToDraw,
				GetNumTabs(), GetTabs()).cx > (rc.Width() - 6) &&
						numCharsToDraw > 0)
	{
	--numCharsToDraw;
	}

TPoint centre(rc.Width() / 2, rc.top + 3);

if((drawInfo.itemState & ODS_SELECTED) &&
							GetEnabled(drawInfo.itemID))
	{
	DC.TabbedTextOut(centre.OffsetBy(0, 1), str, numCharsToDraw,
									GetNumTabs(), GetTabs(), 0);
	}
else
	{
	DC.TabbedTextOut(centre, str, numCharsToDraw,
									GetNumTabs(), GetTabs(), 0);
	}

DC.SetTextAlign(oldAlign);
}




//----------------------------------------------------------------
// Description:
//		Draws an item in the list.
//
//		Override TXListBox behaviour so that we can draw the button.
//
// Arguments:
//		str = text to draw.
//
//----------------------------------------------------------------
void TXButtonListBox::DrawListItem(DRAWITEMSTRUCT &drawInfo,
														const char far *str)
{
TXListBox::DrawListItem(drawInfo, str);

TDC DC(drawInfo.hDC);
TRect rc(drawInfo.rcItem);

TPen blackPen(::GetSysColor(COLOR_WINDOWFRAME));
TPen whitePen(::GetSysColor(COLOR_BTNHIGHLIGHT));
TPen ltGrayPen(::GetSysColor(COLOR_BTNFACE));
TPen dkGrayPen(::GetSysColor(COLOR_BTNSHADOW));

DC.SelectStockObject(HOLLOW_BRUSH);
DC.SelectObject(blackPen);

// now draw button border...
DC.Rectangle(rc.TopLeft().OffsetBy(1,1), rc.BottomRight().OffsetBy(-1,-1));

BOOL selected = ((drawInfo.itemState & ODS_SELECTED) &&
									GetEnabled(drawInfo.itemID));

// left border...
rc.Inflate(-2, -2);
if(selected) DC.SelectObject(dkGrayPen);
else DC.SelectObject(whitePen);

DC.MoveTo(rc.TopLeft());
if(selected)
	DC.LineTo(rc.BottomLeft());
else
	{
	DC.LineTo(rc.BottomLeft().OffsetBy(0, -1));
	DC.MoveTo(rc.TopLeft().OffsetBy(1, 0));
	DC.LineTo(rc.BottomLeft().OffsetBy(1, -2));
	}

// top border...
DC.MoveTo(rc.TopLeft());
if(selected)
	DC.LineTo(rc.TopRight());
else
	{
	DC.LineTo(rc.TopRight().OffsetBy(-1, 0));
	DC.MoveTo(rc.TopLeft().OffsetBy(0, 1));
  	DC.LineTo(rc.TopRight().OffsetBy(-2, 1));
	}

if(!selected)
	{
	// right border (only draw if not selected)...
	DC.SelectObject(dkGrayPen);

	DC.MoveTo(rc.TopRight().OffsetBy(-1, 1));
	DC.LineTo(rc.BottomRight().OffsetBy(-1, 0));
	DC.MoveTo(rc.TopRight().OffsetBy(-2, 1));
	DC.LineTo(rc.BottomRight().OffsetBy(-2, 0));

	// bottom border (only draw if not selected)...
	DC.MoveTo(rc.BottomRight().OffsetBy(-1, -1));
	DC.LineTo(rc.BottomLeft().OffsetBy(-1, -1));
	DC.MoveTo(rc.BottomRight().OffsetBy(-1, -2));
	DC.LineTo(rc.BottomLeft().OffsetBy(-2, -2));
	}

DC.RestoreObjects();
}



//----------------------------------------------------------------
// Description:
//		Responds to notification that focus has shifted to or from
//		an item.
//
//		Override TXListBox behaviour so that the focus rect is
//		drawn within border of button
//----------------------------------------------------------------
void TXButtonListBox::ODAFocus(DRAWITEMSTRUCT &drawInfo)
{
TDC DC(drawInfo.hDC);
TRect rc(drawInfo.rcItem);
DC.DrawFocusRect(rc.InflatedBy(-4, -4));
}
