//----------------------------------------------------------------
// Copyright  Antony Corbett 1995. All Rights Reserved
// Author:	Antony Corbett
//				Compuserve	100277,330
//				INTERNET		antony@corbetteng.co.uk
//				Tel +44 (1926) 856131,  Fax +44 (1926) 856727
//
// Description:
//		Header file for TXListBox class and TItemData support class.
//
//		TListBox derivatives that provide disabling of selected
//		list items, text colouring of items, horizontal scrolling
//		and dynamic switching between single & multiple selection.
//
//----------------------------------------------------------------

#ifndef __TXLB_H
#define __TXLB_H

#ifndef OWL_LISTBOX_H
	#include <owl\listbox.h>
#endif

#if defined BUILDXLBDLL
  #define XLBEXPORT	__export
#elif defined XLBDLL
  #define XLBEXPORT	__import
#else
  #define XLBEXPORT
#endif


//***************************************
//		TItemData
//
//		Used to store enable status of an
//		item in the list and text colour
//		of item. Also retains 32-bit item
//		data. A ptr to a TItemData object
//		is associated with each item added
//		to the list	using SetItemData.
//
//***************************************

class XLBEXPORT TItemData
	{
	private:
		BOOL enabled_;
		BOOL useSysColor_;		// TRUE if we are to use system colour
		uint32 data_;				// original 32-bit ItemData
		TColor textColor_;

	public:
		TItemData() :
			enabled_(TRUE),
			data_(0),
			useSysColor_(TRUE)
				{

				}

		uint32 GetData() const { return data_; }
		BOOL GetEnabled() const { return enabled_; }
		BOOL GetUseSysColor() const { return useSysColor_; }
		TColor GetTextColor() const { return textColor_; }

		void SetEnabled(BOOL enabled) { enabled_ = enabled; }
		void SetData(uint32 data) { data_ = data; }
		void SetTextColor(TColor color)
			{
			textColor_ = color;
			useSysColor_ = FALSE;
			}
		void SetUseSysColor(BOOL useSysColor)
			{
			useSysColor_ = useSysColor;
			if(useSysColor) textColor_ = 0;
			}

	};


//***************************************
//		TXListBox
//
//		Provides horizontal scrolling
//		and disabling of selected items.
//***************************************

class XLBEXPORT TXListBox : public TListBox
	{
	private:
		BOOL hasHScroll_;
		int numTabs_;				// number of tab stops
		int far *tabs_;			// tab stop values
		BOOL pseudoSingleSel_;	// forced into single-selection mode?

		typedef TISArrayAsVector<int> SortedIntegers;
		SortedIntegers& textExtents_;

	public:
		TXListBox(TWindow* parent, int resourceId,
								BOOL useHScroll = FALSE,
								TModule* module = 0);

		TXListBox(TWindow* Parent, int Id, int x, int y,
								int w, int h,
								BOOL useHScroll = FALSE,
								TModule* module = 0);

		~TXListBox();

		// to describe listbox mode...
		enum TXListBoxMode
			{
			SINGLE,				// real single selection listbox
			MULTIPLE,			// real multiple selection list				
			PSEUDO_SINGLE,		// multiple selection list forced
			};						//	into single selection mode.

		// operations on individual list box items...
		int AddString(const char far *str);
		int InsertString(const char far *str, int index);
		int DeleteString(int index);
		int AddDisabledString(const char far *str);
		void Enable(int index, BOOL enable = TRUE);
		int SetItemData(int index, uint32 data);
		int SetSel(int index, BOOL select);
		int SetSelIndexes(int* indexes, int numSelections, BOOL shouldSet);
		int SetSelIndex(int index);
		int SetSelItemRange(bool select, int first, int last);
		int SetSelString(const char far* str, int searchIndex);
		int SetSelStrings(const char far** prefixes, int numSelections, bool shouldSet);
		void SetTextColor(int index, TColor color);
		void ResetTextColor(int index);

		// operations on listbox...
		void ClearList();
		BOOL SetTabStops(int numTabs, int far* tabs);
		BOOL MakeSingleSel(BOOL single = TRUE);

		// query individual items...
		uint32 GetItemData(int index) const;
		BOOL GetSel(int index) const;
		BOOL GetEnabled(int index) const;
		TItemData *GetItemDataPtr(int index) const;

		// query...
		int GetSelCount() const;
		int GetSelIndex() const;
		int GetSelIndexes(int far *indexes, int maxCount) const;
		TXListBoxMode GetListBoxMode() const;
		int GetNumTabs() const { return numTabs_; }
		int far *GetTabs() { return tabs_; }

	protected:
		DECLARE_RESPONSE_TABLE(TXListBox);

		void CleanupWindow();

		void ODADrawEntire(DRAWITEMSTRUCT &drawInfo);
		void ODAFocus(DRAWITEMSTRUCT &drawInfo);
		void ODASelect(DRAWITEMSTRUCT &drawInfo);

      void CmKillFocus();
		void CmSelChange();
      void EvLButtonDown(uint modkeys, TPoint& point);

		virtual void DrawListItem(DRAWITEMSTRUCT &drawInfo, const char far *str);
		virtual int FindTextExtent(const char far *str);
		virtual TColor QueryTextColor(DRAWITEMSTRUCT &drawInfo) const;
		virtual TColor QueryBkColor(DRAWITEMSTRUCT &drawInfo) const;
		void StoreStrExtent(const char far *str);
		virtual void DrawText(DRAWITEMSTRUCT &drawInfo,
										const TRect& rc, const char far *str);
		void UpdateHorizontalExtent();
		BOOL IsMultipleSel() const;

		void EvMeasureItem(UINT ctrlId, MEASUREITEMSTRUCT &measure);

	};



#endif	//__TXLB_H
