/*  Project toolctrl
	DHB Software
	Copyright  1995. All Rights Reserved.

	SUBSYSTEM:    toolctrl.apx Application
	FILE:         toolbar.cpp
	AUTHOR:       David H. Borg


	OVERVIEW
	========
	Source file for implementation of ToolBar (TWindow).
*/

#include <owl\owlpch.h>
#pragma hdrstop

#include "toolbar.h"


// A command enabler for the toolbar. This makes the toolbar automatically
// enable and disable the buttons, just like the OWL TControlBar.
class ToolBarButtonEnabler : public TCommandEnabler {
	public:
		ToolBarButtonEnabler(HWND hWndReceiver, ToolBar* tb, uint id)
				: TCommandEnabler( id, hWndReceiver) {
			toolBar = tb;
		}

	// override member functions of TCommandEnabler
	void  Enable (bool enable);
	void  SetText (const char far* text);
	void  SetCheck (int state);

	protected:
		ToolBar*  toolBar;
};


void ToolBarButtonEnabler::Enable(bool enable)
{
	TCommandEnabler::Enable(enable);
	toolBar->HandleMessage( TB_ENABLEBUTTON, Id, enable);
}


void ToolBarButtonEnabler::SetText(const char far* /*text*/)
{
	// implement later, do nothing for now
}


void ToolBarButtonEnabler::SetCheck(int state)
{
	toolBar->HandleMessage( TB_CHECKBUTTON, Id, state);
}

// Initialization parameters for tbButton array. Include one line per button.
// A better way to do this would be to implement a ToolBar::InsertButton(...)
// function to insert each button in TApplication::InitMainWindow()
static TBBUTTON initTbButton[] = {
	{ STD_FILENEW, CM_FILENEW, TBSTATE_ENABLED, TBSTYLE_BUTTON, {0,0}, 0, IDS_FILENEW },
	{ STD_FILEOPEN, CM_FILEOPEN, TBSTATE_ENABLED, TBSTYLE_BUTTON, {0,0}, 0, IDS_FILEOPEN},
	{ STD_FILESAVE, CM_FILESAVE, TBSTATE_ENABLED, TBSTYLE_BUTTON, {0,0}, 0, IDS_FILESAVE},
	{ 0, 0, 0, TBSTYLE_SEP, {0,0}, 0, 0},
	{ STD_CUT, CM_EDITCUT, TBSTATE_ENABLED, TBSTYLE_BUTTON, {0,0}, 0, IDS_EDITCUT},
	{ STD_COPY, CM_EDITCOPY, TBSTATE_ENABLED, TBSTYLE_BUTTON, {0,0}, 0, IDS_EDITCOPY},
	{ STD_PASTE, CM_EDITPASTE, TBSTATE_ENABLED, TBSTYLE_BUTTON, {0,0}, 0, IDS_EDITPASTE},
	{ 0, 0, 0, TBSTYLE_SEP, {0,0}, 0, 0},
	{ STD_UNDO, CM_EDITUNDO, TBSTATE_ENABLED, TBSTYLE_BUTTON, {0,0}, 0, IDS_EDITUNDO},
	{ 0, 0, 0, TBSTYLE_SEP, {0,0}, 0, 0},
	{ CUST_BMP1, CM_CUSTOM_BITMAP1, TBSTATE_ENABLED | TBSTATE_CHECKED, TBSTYLE_CHECKGROUP, {0,0}, 0, IDS_CUST_BMP1},
	{ CUST_BMP2, CM_CUSTOM_BITMAP2, TBSTATE_ENABLED, TBSTYLE_CHECKGROUP, {0,0}, 0, IDS_CUST_BMP2},
	{ CUST_BMP3, CM_CUSTOM_BITMAP3, TBSTATE_ENABLED, TBSTYLE_CHECKGROUP, {0,0}, 0, IDS_CUST_BMP3},
	{ 0, 0, 0, TBSTYLE_SEP, {0,0}, 0, 0},
	{ CUST_BMP5, CM_CUSTOM_BITMAP4, TBSTATE_ENABLED, TBSTYLE_CHECK, {0,0}, 0, IDS_CUST_BMP4},
	{ 0, 0, 0, TBSTYLE_SEP, {0,0}, 0, 0},
	{ STD_FIND, CM_EDITFIND, TBSTATE_ENABLED, TBSTYLE_BUTTON, {0,0}, 0, IDS_EDITFIND},
	{ STD_REPLACE, CM_EDITREPLACE, TBSTATE_ENABLED, TBSTYLE_BUTTON, {0,0}, 0, IDS_EDITREPLACE}
	};  // end of inittbbuttontemplate

const int toolBarItems = sizeof( initTbButton) / sizeof( initTbButton[0]);
const int nStdItems = 15;		// there are 15 standard bitmap bitmap items
const int nCustItems = 5;		// number if bitmaps in custom bitmap set


//
// Build a response table for all messages/commands handled
// by the application.
//
DEFINE_RESPONSE_TABLE1(ToolBar, TWindow)
//{{ToolBarRSP_TBL_BEGIN}}
	EV_NOTIFY_AT_CHILD(BN_CLICKED, BnClicked),
	EV_MESSAGE( WM_NOTIFY, EvCommonControlNotify),
	EV_WM_PAINT,
//{{ToolBarRSP_TBL_END}}
END_RESPONSE_TABLE;


//{{ToolBar Implementation}}


// the tab control is derived from TWindow with specific Attributes
ToolBar::ToolBar (TWindow* parent, int id, int w, int h, bool size, bool text, TModule* module):
	TWindow(parent, 0, module)
{
	Attr.Id = id;
	Attr.X = 0;
	Attr.Y = 0;
	Attr.W = w;
	Attr.H = h;
	Attr.ExStyle = 0;
	Attr.Style = WS_CHILD | WS_VISIBLE | WS_CLIPSIBLINGS | TBSTYLE_TOOLTIPS | CCS_ADJUSTABLE;

	largeBmp = size;
	showText = text;

	tbButton = new TBBUTTON[toolBarItems];	// create structure array for this toolbar

	// copy button template array into button structure array for this toolbar
	for( int i = 0; i < toolBarItems; i++){
		tbButton[i].iBitmap = initTbButton[i].iBitmap;
		tbButton[i].idCommand = initTbButton[i].idCommand;
		tbButton[i].fsState = initTbButton[i].fsState;
		tbButton[i].fsStyle = initTbButton[i].fsStyle;
	#ifdef _WIN32
		tbButton[i].bReserved[0] = initTbButton[i].bReserved[0];
		tbButton[i].bReserved[1] = initTbButton[i].bReserved[1];
	#endif
		tbButton[i].dwData = initTbButton[i].dwData;
		tbButton[i].iString = initTbButton[i].iString;
	}
}


ToolBar::ToolBar (TWindow* parent, const char far* title, TModule* module):
	TWindow(parent, title, module)
{
	// INSERT>> Your constructor code here.

	delete[] tbButton;		// delete array created in the constructor
}


ToolBar::~ToolBar ()
{
	Destroy();

	// INSERT>> Your destructor code here.

}


void ToolBar::EvPaint ()
{
	// Let Common Control paint the window, don't call TWindow::EvPaint()
	DefaultProcessing();
}


char far* ToolBar::GetClassName ()
{
	// return the Windows 95 Toolbar common control class name
	// this is what makes the window a tool bar
	return TOOLBARCLASSNAME;
}


void ToolBar::SetupWindow ()
{
	TWindow::SetupWindow();

	// INSERT>> Your code here.

	SetupToolBar();
}


void ToolBar::SetupToolBar ()
{
	// Create the toolbar control and add the buttons, commands, and text.
	char buf[255];
	TColor colors( TColor::LtGray);
	COLORMAP mapSysColor;

	// Map custom button bitmaps background color to system button face color.
	mapSysColor.from = COLORREF( colors);
	mapSysColor.to = ::GetSysColor( COLOR_BTNFACE);
	CreateMappedBitmap( GetApplication()->GetInstance(),
						IDB_TOOLBAR_LARGE, 0, &mapSysColor, nCustItems);
	CreateMappedBitmap( GetApplication()->GetInstance(),
						IDB_TOOLBAR_SMALL, 0, &mapSysColor, nCustItems);

	// Tell the control the size of button structure.
	HandleMessage( TB_BUTTONSTRUCTSIZE, (WPARAM) sizeof(TBBUTTON), 0);

	// Initialize structures that identify the bitmaps.
	tbLargeBmp.hInst = HINST_COMMCTRL;
	tbLargeBmp.nID = IDB_STD_LARGE_COLOR;
	tbSmallBmp.hInst = HINST_COMMCTRL;
	tbSmallBmp.nID = IDB_STD_SMALL_COLOR;

	custLargeBmp.hInst = GetApplication()->GetInstance();
	custLargeBmp.nID = IDB_TOOLBAR_LARGE;
	custSmallBmp.hInst = GetApplication()->GetInstance();
	custSmallBmp.nID = IDB_TOOLBAR_SMALL;

	// initialize toolbar with large or small bitmaps
	if( largeBmp){
		HandleMessage( TB_ADDBITMAP, (WPARAM)nStdItems, (LPARAM)&tbLargeBmp);
		custBitmap = HandleMessage( TB_ADDBITMAP, (WPARAM)nCustItems, (LPARAM)&custLargeBmp);
	} else {
		HandleMessage( TB_ADDBITMAP, (WPARAM)nStdItems, (LPARAM)&tbSmallBmp);
		custBitmap = HandleMessage( TB_ADDBITMAP, (WPARAM)nCustItems, (LPARAM)&custSmallBmp);
	}

	// Execute this code for toolbar to have text on the buttons
	// Then add button string id's to buttons structure
	if( showText){
		for( int bNum, i=0; i < toolBarItems; i++){
			if( tbButton[i].iString == 0)
				continue;
			GetModule()->LoadString( tbButton[i].iString, (LPSTR)&buf, sizeof(buf) - 1);
			bNum = HandleMessage( TB_ADDSTRING, 0, (LPARAM)(LPSTR)buf);
			tbButton[i].iString   = bNum;
		}
	}

	// Add the custom bitmap base number to the offsets from the template.
	// note: this is not needed for the standard bitmaps if added first.
	for( int j=0; j < toolBarItems; j++){
		if( ((tbButton[j].idCommand >= CM_CUSTOM_BITMAP1) &&
			 (tbButton[j].idCommand <= CM_CUSTOM_BITMAP5))
				&& tbButton[j].fsStyle != TBSTYLE_SEP)
			tbButton[j].iBitmap += custBitmap;
	}

	// Send the initialized buttons structure to the common controls dll.
	HandleMessage( TB_ADDBUTTONS, (WPARAM)toolBarItems, (LPARAM)(LPTBBUTTON)tbButton);

	// Automatically size of the toolbar to the button size.
	HandleMessage( TB_AUTOSIZE, 0, 0);
}


// This function handles the messages sent by Windows 95 common controls.
LRESULT ToolBar::EvCommonControlNotify( WPARAM /*wParam*/, LPARAM lParam)
{
	LRESULT retVal = false;
	NMHDR FAR *pNmhdr = (NMHDR FAR *)lParam;
	LPTBNOTIFY lpTbNotify = (LPTBNOTIFY)lParam;

	switch( pNmhdr->code){
		case TTN_NEEDTEXT:		 // process tool tip control message
			{
				LPTOOLTIPTEXT lpText;

				lpText = (LPTOOLTIPTEXT)lParam;
				lpText->hinst = GetModule()->GetInstance();
				lpText->lpszText = MAKEINTRESOURCE(lpText->hdr.idFrom);
			}
			break;
		case TBN_GETBUTTONINFO:	// Customize Toolbar dialog nees this info
			if (lpTbNotify->iItem < toolBarItems) {
				char buf[255];
				lpTbNotify->tbButton = tbButton[lpTbNotify->iItem];
				if (lpTbNotify->pszText != 0 && showText) {
					GetModule()->LoadString( tbButton[lpTbNotify->iItem].iString,
												(LPSTR)&buf, sizeof(buf) - 1);
					strcpy(lpTbNotify->pszText, buf);
					lpTbNotify->cchText = strlen(lpTbNotify->pszText);
				}
				retVal = true;
			}
			break;
		case TBN_QUERYINSERT:	// Can Customize Dialog insert?
			retVal = true;
			break;
		case TBN_QUERYDELETE:	// Can Customize Dialog delete?
			retVal = true;
			break;
		default:
			break;
	}
	return retVal;
}


// Pressing a toolbar button causes a BN_CLICKED message to be sent.
// Re-transmit the BN_CLICKED Id as CM_... message to simulate menu selections.
void ToolBar::BnClicked()
{
	TCurrentEvent& currentEvent = GetCurrentEvent();

	// Toggle check button between sad-face and happy-face bitmaps.
	if( LOWORD(currentEvent.WParam) == CM_CUSTOM_BITMAP4){
		int state = HandleMessage( TB_GETSTATE, CM_CUSTOM_BITMAP4, 0);
		if( state & BF_CHECKED)
			HandleMessage( TB_CHANGEBITMAP, CM_CUSTOM_BITMAP4,
				MAKELPARAM( custBitmap + CUST_BMP4, custBitmap + CUST_BMP5));
		else
			HandleMessage( TB_CHANGEBITMAP, CM_CUSTOM_BITMAP4,
				MAKELPARAM( custBitmap + CUST_BMP5, custBitmap + CUST_BMP4));
	}

	// Post button Id to simulate a menu command.
	Parent->PostMessage( WM_COMMAND, LOWORD(currentEvent.WParam));
}


// IdleAction takes care of button enabling through ToolBarButtonEnabler.
bool ToolBar::IdleAction (long idleCount)
{
	if (idleCount == 0) {
		for( int i=0; i < toolBarItems; i++){
			if( tbButton[i].fsStyle != TBSTYLE_BUTTON)
				continue;
			// Must use HandleMessage here since a ptr to a temp is passed
			Parent->HandleMessage( WM_COMMAND_ENABLE, 0,
				(LPARAM)&ToolBarButtonEnabler( Parent->HWindow, this,
				tbButton[i].idCommand));
		}
	}
	return TWindow::IdleAction(idleCount);
}


// Toggles between large and small bitmaps by replacing all bitmaps.
void ToolBar::SetBitmaps (bool size)
{
	TBREPLACEBITMAP replaceStdBmp, replaceCustBmp;

	if( largeBmp == size)		// size is not changing - return
		return;

	largeBmp = size;

	if( largeBmp){
		replaceStdBmp.hInstOld = tbSmallBmp.hInst;
		replaceStdBmp.nIDOld = tbSmallBmp.nID;
		replaceStdBmp.hInstNew = tbLargeBmp.hInst;
		replaceStdBmp.nIDNew = tbLargeBmp.nID;
		replaceStdBmp.nButtons = toolBarItems;
		HandleMessage( TB_REPLACEBITMAP, (WPARAM)nStdItems, (LPARAM)&replaceStdBmp);

		replaceCustBmp.hInstOld = custSmallBmp.hInst;
		replaceCustBmp.nIDOld = custSmallBmp.nID;
		replaceCustBmp.hInstNew = custLargeBmp.hInst;
		replaceCustBmp.nIDNew = custLargeBmp.nID;
		replaceCustBmp.nButtons = toolBarItems;
		HandleMessage( TB_REPLACEBITMAP, (WPARAM)nCustItems, (LPARAM)&replaceCustBmp);

		HandleMessage( TB_SETBITMAPSIZE, 0, MAKELPARAM( 24, 24));
	} else {
		replaceStdBmp.hInstOld = tbLargeBmp.hInst;
		replaceStdBmp.nIDOld = tbLargeBmp.nID;
		replaceStdBmp.hInstNew = tbSmallBmp.hInst;
		replaceStdBmp.nIDNew = tbSmallBmp.nID;
		replaceStdBmp.nButtons = toolBarItems;
		HandleMessage( TB_REPLACEBITMAP, (WPARAM)nStdItems, (LPARAM)&replaceStdBmp);

		replaceCustBmp.hInstOld = custLargeBmp.hInst;
		replaceCustBmp.nIDOld = custLargeBmp.nID;
		replaceCustBmp.hInstNew = custSmallBmp.hInst;
		replaceCustBmp.nIDNew = custSmallBmp.nID;
		replaceCustBmp.nButtons = toolBarItems;
		HandleMessage( TB_REPLACEBITMAP, (WPARAM)nCustItems, (LPARAM)&replaceCustBmp);

		HandleMessage( TB_SETBITMAPSIZE, 0, MAKELPARAM( 16, 16));
	}

	HandleMessage( TB_AUTOSIZE, 0, 0);		// resize the toolbar
}


