///////////////////////////////////////////////////////////////////////////////
//
// Module:		ansiprsr.cpp
//
// Contents:	ANSI parser base class
//
// Product:		telnet
//
// Revisions:	13.Jul.1995	igor.milavec@uni-lj.si
//					Original code
//
///////////////////////////////////////////////////////////////////////////////

#include <ansiprsr.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>

char* TANSIParser::GetTerminalId()
{
	return "\033[?1;2c";
}

char* TANSIParser::ParseIAC(char* pszBuffer, char* pszBufferEnd)
{
	if (pszBuffer + 2 < pszBufferEnd) {
		switch (pszBuffer[1]) {
		  case 251:
		  case 252:
		  case 253:
		  case 254:
			if (pszBuffer + 3 < pszBufferEnd)
				pszBuffer += 3;
			break;

		  default:
			pszBuffer += 2;
			break;

		}
	}

	return pszBuffer;
}

#pragma argsused
char* TANSIParser::ParseEscapeANSI(char* pszBuffer, char* pszBufferEnd)
{
	int bExtMode = 0;

	const int nParam = 10;	// Maximum number of parameters
	int iParam[nParam] = {0, 0, 0, 0, 0};
	int iCurrentParam = 0;

	// Get parameters from escape sequence.
	while (!isalpha(*pszBuffer)) {

		// Check for parameter delimiter.
		if (*pszBuffer == ';') {
			pszBuffer++;
			continue;
		}
		if (*pszBuffer == '?') {
			bExtMode = 1;
			pszBuffer++;
			continue;
		}

		// pszHead should point to digit now. Otherwise we got bad escape
		// sequence, so we just get out of here!
		if (!isdigit(*pszBuffer))
			return pszBuffer;

		iParam[iCurrentParam] = strtoul(pszBuffer, &pszBuffer, 10);

		if (iCurrentParam < nParam)
			iCurrentParam++;
	}

	switch (*pszBuffer++) {

	  // Set cursor position.
	  case 'H':
	  case 'f':
		if (iCurrentParam < 2)
			iParam[1] = 1;
		if (iCurrentParam < 1)
			iParam[0] = 1;
		ConSetCursorPosition(iParam[1] - 1,	iParam[0] - 1);
		break;

	  // Move cursor up.
	  case 'A':
		if (iCurrentParam < 1)
			iParam[0] = 1;
		ConMoveCursorPosition(0, -iParam[0]);
		break;

	  // Move cursor down.
	  case 'B':
		if (iCurrentParam < 1)
			iParam[0] = 1;
		ConMoveCursorPosition(0, iParam[0]);
		break;

	  // Move cursor right.
	  case 'C':
		if (iCurrentParam < 1)
			iParam[0] = 1;
		ConMoveCursorPosition(-iParam[0], 0);
		break;

	  // Move cursor left.
	  case 'D':
		if (iCurrentParam < 1)
			iParam[0] = 1;
		ConMoveCursorPosition(iParam[0], 0);
		break;

	  case 'J': {	// Clear screen
		if (iCurrentParam < 1)
			iParam[0] = 2;
		switch (iParam[0]) {
		  case 0:
			ConClearEOScreen();
			break;
		  case 1:
			ConClearBOScreen();
			break;
		  case 2:
			ConClearScreen();
			break;
		}
		break;
	  }

	  case 'K':	{	// Clear line
		if (iCurrentParam < 1)
			iParam[0] = 2;
		switch (iParam[0]) {
		  case 0:
			ConClearEOLine();
			break;
		  case 1:
			ConClearBOLine();
			break;
		  case 2:
			ConClearLine();
			break;
		}
		break;
	  }

	  case 'c': {
		char* szTerminalId = GetTerminalId();
		NetWriteString(szTerminalId, strlen(szTerminalId));
		break;
	  }

	  case 'h': {
		if (bExtMode)
			for (int i = 0; i < iCurrentParam; i++)
				ConSetExtendedMode(iParam[i], 1);
		break;
	  }

	  case 'l': {
		if (bExtMode)
			for (int i = 0; i < iCurrentParam; i++)
				ConSetExtendedMode(iParam[i], 0);
		break;
	  }

	}

	return pszBuffer;
}

char* TANSIParser::ParseEscape(char* pszBuffer, char* pszBufferEnd)
{
/*	ConWriteString("\033", 1);
	return ++pszBuffer; //*/

	// Check if we have enough characters in buffer.
	if ((pszBufferEnd - pszBuffer) < 2)
		return pszBuffer;

	// Decode the command...
	pszBuffer++;
	switch (*pszBuffer++) {
	  case '[': {

		// Check if we have whole escape sequence in buffer.
		char* pszChar = pszBuffer + 1;
		while ((pszChar < pszBufferEnd) && (!isalpha(*pszChar)))
			pszChar++;
		if (pszChar == pszBufferEnd)
			pszBuffer -= 2;
		else
			pszBuffer = ParseEscapeANSI(pszBuffer, pszBufferEnd);
		break;
	  }

	  case ')':
		pszBuffer++;
		break;

	  case 'Z': {
		char* szTerminalId = GetTerminalId();
		NetWriteString(szTerminalId, strlen(szTerminalId));
		break;
	  }
	}

	return pszBuffer;
}

char* TANSIParser::ParseBuffer(char* pszBuffer, char* pszBufferEnd)
{
	switch (*pszBuffer) {

	  case 0:
		pszBuffer++;
		break;

	  case 12:
		pszBuffer++;
		ConClearScreen();
		ConSetCursorPosition(0, 0);
		break;

	  case 27:
		pszBuffer = ParseEscape(pszBuffer, pszBufferEnd);
		break;

	  case 255:
		pszBuffer = ParseIAC(pszBuffer, pszBufferEnd);
		break;


	  default: {
		char* pszCurrent = pszBuffer + 1;
		while ((pszCurrent < pszBufferEnd) && !iscntrl(*pszCurrent) &&
				(*pszCurrent != (char)255))
			pszCurrent++;

		pszBuffer += ConWriteString(pszBuffer, pszCurrent - pszBuffer);
		break;
	  }
	}

	return pszBuffer;
}


