/*
;;tquery_2.cpp
;;
;;  This class was designed by Chris Smith (ccas@mcs.com),
;;  with modifications from    Buc Rogers  (buc@mcs.com)
;;
;;  Design notes:
;;
;;    950503
;;      o Nice option would be to pass opt. trx type to query
*/

#pragma   hdrstop
#include  <tquery_2.hpp>

//RTL, Classlib includes

//3d party includes
#include  <stdcpp.hpp>
#include  <tdbf.hpp>

/*
;;===========================================================
;;  CLASS TQueryBase  --  Query Object Base Class
;;===========================================================
;;
;;TQueryBase::TQueryBase (TBDatabase& ptDatabaseArg )
;;
;;  Constructor for TQueryBase
;;
*/
TQueryBase::TQueryBase( TBDatabase& theDatabase )
  : engine( theDatabase.getEngine() ),
    db( &theDatabase ),
    myLanguage( qrylangSQL ),
    lastError( DBIERR_NONE )
{
}

/*
;;virtual
;;TQueryBase::~TQueryBase()
;;
;;  Destructor for TQueryBase
;;
*/
TQueryBase::~TQueryBase()
{
}

/*
;;const string&
;;TQueryBase::QueryString() const
;;
;;  Return query string streamed into Text() to this point
;;
;;  Notes:
;;    1) Streaming into Text() after this call is not supported
;;        and will have undefined results
;;    2) If Text() is a null string, "<< ends" can be called more than
;;        once if QueryString() called more than once
;;
*/
const string&
TQueryBase::QueryString() const
{
  if ( 0 == buf.length() ) //? buf not yet built
  {
    //   Terminate the iostream.
    Text() << ends;
    //
    //   Extract the text into a Buffer object (will delete text)
    char* ps = Text().str();
    const_cast<TQueryBase*>(this)->buf = ps; //ASSUMING buf deallocated by TQueryBase destructor
    delete [] ps; //Deallocate string::str() returned pointer
  }
  return ( buf );
}

/*
;;protected:
;;void
;;TQueryBase::ThrowExc( const char * fnArg,
;;                      const char * errArg = NULL,
;;                      const char * infoArg = NULL ) const
;;
*/
void
TQueryBase::ThrowExc( const char * fnArg,  const char * errArg,
                      const char * infoArg ) const
{
  TTdbfExc::ThrowExc(fnArg, errArg, infoArg, ClassName(), Error(),
                      Db()->getAliasName().c_str() );
}

/*
;;===========================================================
;;    CLASS TQueryNoAnswer  --  Query without Results
;;===========================================================
;;
;;virtual
;;TQueryNoAnswer::~TQueryNoAnswer()
;;
;;  Destructor for TQueryNoAnswer
;;
*/
TQueryNoAnswer::~TQueryNoAnswer()
{
}

/*
;;Retcode
;;TQueryNoAnswer::Run()
;;
;;    -Throws TTdbfExc exception if query returns an Answer cursor
;;
;;    -Will also throw TTdbfExc on other query error, which the App should
;;        catch. The original Query string is returned in TTdbfExec::errInfo.
;;
*/
Retcode
TQueryNoAnswer::Run()
{
  //
  QueryString(); //Ensure strstream term. and buf assigned
  //
  //   Run the Query.
  Retcode rtnVal;
  try
  {
    rtnVal = Db()->queryNoAnswer( buf.c_str(), Language() );
  }
  catch ( TTdbfExc bad )   // Query itself tosses exception
  {
    SetError( bad.lastError );
    string txt( bad.GetFnName() + string(": ") + bad.GetErrMsg() );
    ThrowExc ( "Run()", txt.c_str(), buf.c_str() );
  }

  if (rtnVal != DBIERR_NONE)  //  Query failed w/o exception
  {
    SetError( Db()->lastError );
    ThrowExc ( "Run()", NULL, buf.c_str() );
  }

  return ( rtnVal ); //Should always be DBIERR_NONE
}

/*
;;===========================================================
;;   CLASS TQueryAnswerRO     --  Query returning Result Table
;;===========================================================
;;TQueryAnswerRO::TQueryAnswerRO( TBDatabase& ptDatabaseArg )
;;
;;  Constructor for TQueryAnswerRO
;;
*/
TQueryAnswerRO::TQueryAnswerRO( TBDatabase& theDatabase )
  : TQueryBase( theDatabase ),
    resultDB(NULL),
    resultTbl(""),
    cursor(new TBCursor)
{
}

/*
;;
;;virtual
;;TQueryAnswerRO::~TQueryAnswerRO()
;;
;;  Destructor for TQueryAnswerRO
;;
*/
TQueryAnswerRO::~TQueryAnswerRO()
{
}

/*
;;Retcode
;;TQueryAnswerRO::Run()
;;
;;    -Throws TTdbfExc exception if query returns an Answer cursor
;;
;;    -Will also throw TTdbfExc on other query error, which the App should
;;        catch. The original Query string is returned in TTdbfExec::errInfo.
;;
*/
Retcode
TQueryAnswerRO::Run()
{
  //
  QueryString(); //Ensure strstream term. and buf assigned
  //   Run the Query.
  Retcode rtnVal;
  cursor->closeIfOpen(); //Ensure cursor from any prev. query is closed
  try
  {
    rtnVal = Db()->query( buf.c_str(), *cursor, Language() );
  }
  catch ( TTdbfExc bad )   // Query itself tosses exception
  {
    SetError( bad.lastError );
    string txt( bad.GetFnName() + string(": ") + bad.GetErrMsg() );
    ThrowExc ( "Run()", txt.c_str(), buf.c_str() );
  }
  if ( ! ( rtnVal == DBIERR_NONE ) ) //9990 (inv. handle) now thrown by query()
  {
    //
    //  Query failed w/o exception
    SetError( Db()->lastError );
    ThrowExc ( "Run()", NULL, buf.c_str() );
  }
  //
  //
  //  If the user has defined a result table, create it here. Otherwise, don't.
  if ( ! resultDB )
  {
    return ( DBIERR_NONE );  //Should always be DBIERR_NONE
  }
  else
  {
    try
    {
      resultDB->deleteTableIfExists( resultTbl.c_str() );
    }
    catch ( const TTdbfExc& exc )
    {
      SetError( exc.LastError() );
      ThrowExc( "Run()", exc.ErrInfo().c_str(), "deleteTableIfExists()" );
    }

    try
    {
      Db()->appendTable( *cursor, resultTbl.c_str(),
                       szPARADOX, *resultDB, batCOPY );
    }
    catch ( const TTdbfExc& exc )
    {
      SetError( Db()->lastError );
      ThrowExc( "Run()", NULL, "appendTable()" );
    }
    return ( Db()->lastError );  //Should always be DBIERR_NONE
  }
}

/*
;;TQueryAnswerRW::TQueryAnswerRW( TBDatabase& theDatabase,
;;                                TBDatabase& dbPriv,
;;                                const string& nameQueriedTable )
;;
;;  Constructor for TQueryAnswerRW
;;
;;  This class encapsulates a writable answer cursor and associated
;;  Update() function to write any changes back to the queried table.
;;
;;  This class should be adapted or expanded to work with updateable cursors
;;  when IDAPI supports those.
;;
;;  nameQueriedTable is used in two places:
;;    1) To specifiy the table for Update() to update
;;    2) To specify the dbPriv table name to use (if necessary--see below).
;;
;;  nameQueriedTable is not necessarily created in the specified dbPriv
;;  (is created there only if isRemote() ).
;;
;;  Use TQueryAnswerRO with setResultTbl() to guarantee
;;  a returned result table in dbPriv.
;;
*/
TQueryAnswerRW::TQueryAnswerRW( TBDatabase& theDatabase,
                                TBDatabase& dbPriv,
                                const string& nameTempTbl )
  : TQueryBase( theDatabase ),
    resultDB( &dbPriv ),
    resultTbl( nameTempTbl ),
    cursor(new TBCursor)
{
  if ( resultTbl.length() == 0 )
  {
    ThrowExc( "TQueryAnswerRW()", "req'd nameTempTbl is blank" );
  }
}

/*
;;
;;virtual
;;TQueryAnswerRW::~TQueryAnswerRW()
;;
;;  Destructor for TQueryAnswerRW
;;
*/
TQueryAnswerRW::~TQueryAnswerRW()
{
  DELETE_SAFE( cursor );
}

/*
;;Retcode
;;TQueryAnswerRW::Run()
;;
;;    -Throws TTdbfExc exception if query returns an Answer cursor
;;
;;    -Will also throw TTdbfExc on other query error, which the App should
;;        catch. The original Query string is returned in TTdbfExec::errInfo.
;;
*/
Retcode
TQueryAnswerRW::Run()
{
  //
  QueryString(); //Ensure strstream term. and buf assigned
  //   Run the Query.
  Retcode rtnVal;
  cursor->closeIfOpen(); //Ensure cursor from any prev. query is closed
  try
  {
    rtnVal = Db()->query( buf.c_str(), *cursor, Language() );
  }
  catch ( TTdbfExc bad )   // Query itself tosses exception
  {
    SetError( bad.lastError );
    string txt( bad.GetFnName() + string(": ") + bad.GetErrMsg() );
    ThrowExc ( "Run()", txt.c_str(), buf.c_str() );
  }
  if ( ! ( rtnVal == DBIERR_NONE ) ) //9990 (inv. handle) now thrown by query()
  {
    //
    //  Query failed w/o exception
    SetError( Db()->lastError );
    ThrowExc ( "Run()", NULL, buf.c_str() );
  }
  //
  //
  if ( Db()->isRemote() )
  {
    try
    {
      resultDB->deleteTableIfExists( resultTbl.c_str() );
    }
    catch ( const TTdbfExc& exc )
    {
      SetError( resultDB->lastError );
      ThrowExc( "Run()", exc.ErrInfo().c_str(), "deleteTableIfExists()" );
    }
    try
    {
      Db()->appendTable( *cursor, resultTbl.c_str(),
                       szPARADOX, *resultDB, batCOPY );
    }
    catch ( const TTdbfExc& exc )
    {
      SetError( Db()->lastError );
      ThrowExc( "Run()", NULL, "appendTable()" );
    }
    DELETE_SAFE( cursor );
    cursor = new TBCursor( *resultDB, resultTbl.c_str() );
        //::Csr() is open on local table, is writable
  }
  else//Local (PDOX)
  {
    ;//Returned query cursor is writable
  }
  return ( Error() );
}


/*
;;void
;;TQueryAnswerRW::Update()
;;
;;  Update the result table specified in the constructor with
;;    the query result.
;;
;;  Note that all fields in the source cursor MUST match the first
;;  n fields in the destination table--so if anything but select * was
;;  used, the fields specified in the select clause MUST be
;;  in physical table order and without gaps.
;;
*/
void
TQueryAnswerRW::Update()
{
  //Set the source db appropriately
  TBDatabase* dbLoc = Db();
  if ( Db()->isRemote() )
  {
    dbLoc = resultDB;
  }
  dbLoc->appendTable( *cursor, resultTbl.c_str(), NULL, *Db(), batUPDATE );
}

