/*
;;blob.cpp
;;
    AUTHOR: Chris Smith, ccas@mcs.com


    OVERVIEW
    ========
    Class definitions for TBlob objects. These are a simplification of access
        to BLOB fields in a record.
*/

#pragma hdrstop
#include <owl\owlpch.h>

#include "blob.h"

//=============================================================
//   CLASS TBlob  --   Helper Class for BLOB fields
//=============================================================

/*
;;TBlob::TBlob(const string & field_name, const string & init_data)
;;
*/
//-------------------------------------------------------------
//   constuctor for class TBlob
//-------------------------------------------------------------
TBlob::TBlob(const string & field_name, const string & init_data)
    :  field( field_name ), data(init_data)
{
}

/*
;;TBlob::TBlob( const TBlob & other )
;;
//-------------------------------------------------------------
//   constuctor for class TBlob  -- copy ctor
//-------------------------------------------------------------
*/
TBlob::TBlob( const TBlob & other )
    :  field(other.field), data(other.data)
{
}

/*
;;TBlob::~TBlob()
;;
*/
//-------------------------------------------------------------
//   destuctor for class TBlob
//-------------------------------------------------------------
TBlob::~TBlob()
{
}

/*
;;TBlob&
;;TBlob::operator = ( const TBlob & other )
;;
*/
//-------------------------------------------------------------
//   TBlob::operator = ()   --  Assignment operator
//-------------------------------------------------------------
TBlob & TBlob::operator = ( const TBlob & other )
{
    if ( this == &other)     // Self assignment, do nothing
        return *this;
    //
    //  NOTE that field name is NOT changed...just the
    //     data is copied on assignment.
    data  = other.data;
    return *this;
}

/*
;;void
;;TBlob::Read( const TBRecord & rawRec )
;;
*/
//-------------------------------------------------------------
//   TBlob::Read()   --  Load object from Record data
//-------------------------------------------------------------
void
TBlob::Read( const TBRecord & rawRec )
{
    //
    //   Info field is a Blob. Read it here
    rawRec.openBlobRead(field.c_str());
    rawRec.getBlob(field.c_str(), 0L, data );
    rawRec.closeBlob(field.c_str(), FALSE);

}

/*
;;void
;;TBlob::Write( TBRecord & rawRec ) const
;;
*/
//-------------------------------------------------------------
//   TBlob::Write()   --  Load Record with object data
//-------------------------------------------------------------
void
TBlob::Write( TBRecord & rawRec ) const
{
    rawRec.openBlobWrite(field.c_str(), 0, FALSE);
    rawRec.putBlob(field.c_str(), 0L, data );
    rawRec.closeBlob(field.c_str());

}

/*
;;TBlobRO::TBlobRO( const string& field_name,
;;                      const TBRecord& rawRec )
;;
;;  TBlobRO constructor--open blob for reading a segment at a time
;;
;;  Entry:
;;    rawRec must be attached to an open TBCursor
;;
*/
TBlobRO::TBlobRO( const string& field_name, const TBRecord& rawRec )
  : data( NULL ),
    field( field_name ),
    maxSegsize( 0x7FFF ),
    offset( 0 ),
    rec( &rawRec )
{
  rec->openBlobRead( field.c_str() );
  sizeBlob = rec->getBlobSize( field.c_str() );
  data = new unsigned char[ maxSegsize ]; //Dynamic alloc last
}


/*
;;TBlobRO::TBlobRO( const TBlobRO& other )
;;
;;  TBlobRO copy constructor--open blob for reading a segment at a time
;;
;;  **Not tested (or fully defined) as of 950705
;;
*/
TBlobRO::TBlobRO( const TBlobRO& other )
  : data( NULL ),
    field( other.field ),
    maxSegsize( 0x7FFF ),
    offset( other.offset ),
    sizeBlob( other.sizeBlob ),
    rec( other.rec )
{
  rec->openBlobRead( field.c_str() );
  data = new unsigned char[ maxSegsize ]; //Dynamic alloc last
  memmove( const_cast<TBlobRO*>(this)->data, other.data, maxSegsize );
}

/*
;;TBlobRO::~TBlobRO()
;;
*/
TBlobRO::~TBlobRO()
{
  DELETE_A_SAFE( data );
  rec->closeBlob( field.c_str(), FALSE );
}

/*
;;UINT16
;;TBlobRO::GetString( string& str,
;;                    UINT16 maxLen = 0 ) const
;;
;;  **Not implemented as of 950706
;;
;;  Buffered read from blob
;;
;;  Get up to maxLen chrs into string (not incl. c_str() term-null)
;;
;;
*/
UINT16
TBlobRO::GetString( string& , UINT16 ) const
{
  return ( 0 );
}

/*
;;UINT16
;;TBlobRO::Read() const
;;
;;  Read up to maxSegsize or bytes remaining in BLOB, whichever is
;;  lesser, reading from where last Read() left off
;;
;;  Return:
;;    Number of bytes read, 0 if EOB
;;  Exit:
;;    Data() is pointer to what is read
;;
*/
UINT16
TBlobRO::Read() const
{
  long diff = sizeBlob - offset;
  UINT16 size;
  if ( diff > maxSegsize )
  {
    size = maxSegsize;
  }
  else
  {
    size = (UINT16)( sizeBlob - offset ); //cast-away compiler warning
  }
  if ( size > 0 )
  {
    rec->getBlob( field.c_str(), size, offset, data );
    const_cast<TBlobRO*>(this)->offset += size;
  }
  return ( size );
}

/*
;;TBlobRW::TBlobRW( const string& field_name,
;;                        TBRecord& rawRec )
;;
;;  TBlobRW constructor--open blob for writing a segment at a time
;;
;;  Entry:
;;    rawRec must be attached to an open TBCursor
;;
*/
TBlobRW::TBlobRW( const string& field_name, TBRecord& rawRec )
  : data( NULL ),
    field( field_name ),
    maxSegsize( 0x7FFF ),
    offset( 0 ),
    rec( &rawRec )
{
  //rec->setNull( field.c_str() ); //Ensure target BLOB is null to start with
  rec->openBlobWrite( field.c_str() );
  sizeBlob = rec->getBlobSize( field.c_str() );
  data = new unsigned char[ maxSegsize ]; //Dynamic alloc last
}

/*
;;TBlobRW::TBlobRW( const TBlobRW& other )
;;
;;  TBlobRW copy constructor--open blob for writing a segment at a time
;;
;;  **Not tested (or fully defined) as of 950705
;;
*/
TBlobRW::TBlobRW( const TBlobRW& other )
  : data( NULL ),
    field( other.field ),
    maxSegsize( 0x7FFF ),
    offset( other.offset ),
    sizeBlob( other.sizeBlob ),
    rec( other.rec )
{
  rec->openBlobWrite( field.c_str() );
  data = new unsigned char[ maxSegsize ]; //Dynamic alloc last
  memmove( data, other.data, maxSegsize );
}

/*
;;TBlobRW::~TBlobRW()
;;
*/
TBlobRW::~TBlobRW()
{
  DELETE_A_SAFE( data );
  rec->closeBlob( field.c_str() );
}

/*
;;UINT16
;;TBlobRW::GetString( string& str,
;;                    UINT16 maxLen = 0 ) const
;;
;;  **Not implemented as of 950706
;;
;;  Buffered read from blob
;;
;;  Get up to maxLen chrs into string (not incl. c_str() term-null)
;;
;;  Return:
;;    Number of bytes read (not incl. term-null)
;;
*/
UINT16
TBlobRW::GetString( string& , UINT16 ) const
{
  return ( 0 );
}

/*
;;void
;;TBlobRW::PutString( const string& str )
;;
;;  **Not implemented as of 950706
;;
;;  Buffered write to blob
;;
;;  Put contents of string to blob
;;
;;  Return:
;;    none
;;
*/
void
TBlobRW::PutString( const string& )
{
}

/*
;;void
;;TBlobRW::Write( unsigned char* str,
;;                UINT16 len )
;;
;;  Write len bytes to BLOB, appending to previous Write() calls
;;
*/
void
TBlobRW::Write( unsigned char* str, UINT16 len )
{
  if ( len > 0 ) //Only if length is > 0
  {
    if ( offset == 0 )
    {
      //Null out target if offset is 0 (at beginning)
      rec->setNull( field.c_str() ); //Ensure target BLOB is null to start with
    }
    rec->putBlob( field.c_str(), len, offset, str );
    offset += len;
    if ( offset > sizeBlob ) //? Just appended to blob
    {
      sizeBlob = offset;
    }
  }
}

