/*********************************************************************
**
**                    BENGINE.CPP
**
** Implement functions of the BEngine class.
**
*********************************************************************/

// DBF - (C) Copyright 1994 by Borland International

#include "kdbf.h"
#pragma hdrstop

// Constructor. Makes a C++ BEngine object without opening IDAPI

BEngine::BEngine()
{
    isOpen = FALSE;
    engineType = pxLocal;
    lastError = DBIERR_NONE;
    engdef *eo = 0;

    try
    {
        eo = new engdef;           // See the INTSTRCT.H file.

        eo->engenv.engineType   = pxLocal;
        eo->engenv.tabCrtMode   = px50Fmt;
        eo->engenv.dosShare     = pxLocalShare;
        eo->engenv.tabLckMode   = px40Lck;
        eo->engenv.tblMaxSize   = 2048;
        eo->handleCnt           = 0;
        eo->compIdxCnt          = 0;
        engobj = (void *) eo;
    }
    catch(xalloc)
    {
        lastError = DBIERR_NOMEMORY;
        return;
    }
}

//  This constructor makes a BEngine object and actually opens the
//  Paradox engine in the requested mode.

BEngine::BEngine(BEngineType eType)
{
    isOpen = FALSE;
    engdef *eo = 0;

    try
    {
        eo = new engdef;          // See the INTSTRCT.H file.
    }
    catch(xalloc)
    {
        lastError = DBIERR_NOMEMORY;
        return;
    }
    eo->engenv.engineType       = eType;
    eo->engenv.tabCrtMode       = px50Fmt;
    eo->engenv.dosShare         = pxLocalShare;
    eo->engenv.tabLckMode       = px40Lck;
    eo->engenv.winShare         = pxShared;
    eo->engenv.tblMaxSize       = 2048;
    eo->engenv.netNamePath[0]   = eo->engenv.userName[0]
                                = eo->engenv.clientName[0]
                                = 0;
    eo->handleCnt               = 0;
    eo->compIdxCnt              = 0;
    engobj = (void *) eo;
    switch (eType)
    {
        case pxWin:
#ifdef WINDOWS
            engineType = pxWin;
            if ((lastError = DbiInit(NULL)) != DBIERR_NONE)
            {
                return;
            }
            break;
#else
            lastError = PXERR_INVENGINETYPE;
            return;
#endif
        case pxNet:
#ifdef WINDOWS
            lastError = PXERR_INVENGINETYPE;
            return;
#else
            lastError = DBIERR_NOTSUPPORTED;
            return;
#endif
        case pxLocal:
#ifdef WINDOWS
            lastError = PXERR_INVENGINETYPE;
            return;
#else
            lastError = DBIERR_NOTSUPPORTED;
            return;
        #endif
    default:
      lastError = PXERR_INVENGINETYPE;
      return;
  }
  isOpen = TRUE;
  lastError = DbiDebugLayerOptions(DEBUGON | OUTPUTTOFILE, "DBF.DBG");
}

// Make an engine object and open it based on the specified 
// environment.  
BEngine::BEngine(const BEnv& env)
{
    lastError = DBIERR_NONE;
    isOpen = FALSE;
    engdef *eo = 0;

    try
    {
        eo = new engdef;          // See the INTSTRCT.H file.
    }
    catch(xalloc)
    {
        lastError = DBIERR_NOMEMORY;
        return;
    }
    eo->handleCnt = 0;
    eo->compIdxCnt = 0;
    engobj = (void *) eo;
    engineType = env.engineType;
    setDefaults(env);
    open();
}

// Destructor; Closes IDAPI if it's open.

BEngine::~BEngine()
{
    close();
    engdef *eo  = (engdef *)engobj;
    if (eo)
    {
        // Reset pointers in any dependent database objects.
        for (int i=0; i < eo->handleCnt; i++)
        {
            if (eo->dbList[i])
            {
                ((dbdef *)eo->dbList[i]->dbobj)->engH = 0;
            }
        }
        if (eo->handleCnt)
        {
            delete [] eo->dbList;
            eo->dbList = 0;
        }

        delete eo;
        eo = 0;
    }
}

// Set default values for the Engine's resource parameters.

Retcode BEngine::setDefaults(const BEnv &env)
{
    BEnv &senv = ((engdef *) engobj)->engenv;

    senv = env;

    lastError = DBIERR_NONE;

    return lastError;
}

// Open the engine based on a previously set environment.
// The engine is opened in Local, Network or Windows mode
// based on the value of engineType.

Retcode BEngine::open(void)
{
    BEnv &senv = ((engdef *) engobj)->engenv;

    if (isOpen)
    {
        return (lastError = PXERR_ENGINEOPEN);
    }

    switch (senv.engineType)
    {
        case pxWin:
#ifdef WINDOWS
            if ((lastError = DbiInit(NULL)) != DBIERR_NONE)
            {
                return lastError;
            }
            break;
#else
            return (lastError = PXERR_INVENGINETYPE);
#endif
        case pxNet:
#ifdef WINDOWS
            return (lastError = PXERR_INVENGINETYPE);
#else
            lastError = DBIERR_NOTSUPPORTED;
            return lastError;
#endif
        case pxLocal:
#ifdef WINDOWS
            return (lastError = PXERR_INVENGINETYPE);
#else
            if ((lastError = DbiInit(NULL)) != DBIERR_NONE)
            {
                return lastError;
            }
            break;
#endif
        default:
            return (lastError = PXERR_INVENGINETYPE);
    }

    isOpen = TRUE;
    DbiDebugLayerOptions(DEBUGON | OUTPUTTOFILE, "DBF.DBG");
    setTblCreateMode(senv.tabCrtMode);
    return (lastError = DBIERR_NONE);
}

// Shut down the currently open engine and close all databases.

Retcode BEngine::close(void)
{
    // Close all open databases.

    engdef *eo = (engdef *) engobj;
    for (int i=0; i < eo->handleCnt; i++)
    {
        if (eo->dbList[i]  && eo->dbList[i]->isOpen)
        {
            if ((lastError = eo->dbList[i]->close())   // Close database.
                != DBIERR_NONE)
            {
                return (lastError);
            }
        }
    }
    for (i = 0; i < eo->compIdxCnt; i++) // Delete compound key
    {                          // map information.
        delete eo->keymap[i].tableName;
        eo->keymap[i].tableName = 0;
        delete [] eo->keymap[i].fieldArray;
        eo->keymap[i].fieldArray = 0;
    }
    if (eo->compIdxCnt)
    {
        delete [] eo->keymap;
        eo->keymap = 0;
    }
    eo->compIdxCnt = 0;
    if (isOpen)
    {
        DbiDebugLayerOptions(0, NULL);
        if ((lastError = DbiExit()) != DBIERR_NONE)
        {
            return lastError;
        }
    }
    else
    {
        lastError = PXERR_ENGINENOTOPEN;
    }
    isOpen = FALSE;
    return lastError;
}

// Get the current values for various engine parameters.

void BEngine::getDefaults(BEnv &env) const
{
    env = ((engdef *)engobj)->engenv;
}

// Enable or Disable the internal hardware error handler.

#pragma argsused
Retcode BEngine::setHWHandler(BOOL ehEnable)
{
    lastError = DBIERR_NOTSUPPORTED;
    return lastError;
}

// Add a password

Retcode BEngine::addPassword(char *password)
{
    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }
    lastError = DbiAddPassword(password);
    return lastError;
}

// Set the maximum size of tables you can create.

Retcode BEngine::setMaxTblSize(int maxSize)
{
    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }
    ((engdef *)engobj)->engenv.tblMaxSize = maxSize;
    lastError = DBIERR_NONE;
    return lastError;
}

// Choose the table format
Retcode BEngine::setTblCreateMode(PXTabCrtMode crtMode)
{
    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }

    ((engdef *)engobj)->engenv.tabCrtMode = crtMode;
    lastError = DBIERR_NONE;
    return lastError;
}  

// Remove a previously entered password.
Retcode BEngine::deletePassword(char *password)
{
    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }
    lastError = DbiDropPassword(password);
    return lastError;
}

// Get the error message corresponding to an error.  

char far *BEngine::getErrorMessage(int errnbr)
{
    return (PXOopErrMsg(errnbr));
}

Retcode BEngine::getErrorContext(INT16 context, char far *message)
{
    if (!isOpen)
    {
        lastError = PXERR_ENGINENOTOPEN;
        return 0;
    }
    lastError = DbiGetErrorContext(context, message);
    
    return lastError;
}

Retcode BEngine::getErrorEntry(UINT16 entry, UINT32 &nativeCode,
                               char far *message)
{
    if (!isOpen)
    {
        lastError = PXERR_ENGINENOTOPEN;
        return 0;
    }
    lastError = DbiGetErrorEntry(entry, &nativeCode, message);
    
    return lastError;
}

Retcode BEngine::GetErrorInfo(DBIErrInfo &errorInfo)
{
    if (!isOpen)
    {
        lastError = PXERR_ENGINENOTOPEN;
        return 0;
    }

    lastError = DbiGetErrorInfo(TRUE, &errorInfo);

    return lastError;
}


// Return the name of the user in a shared environment.
char *BEngine::getUsername(void)
{
    if (!isOpen)
    {
        lastError = PXERR_ENGINENOTOPEN;
        return NULL;
    }
    engdef *eo = (engdef *) engobj;
    if ((lastError = DbiGetNetUserName(eo->engenv.userName)) == DBIERR_NONE)
    {
        return eo->engenv.userName;
    }
    else
    {
        return 0;
    }
}

// Note - sessions are not fully integrated into the KDBF. Make certain to
//        explicitly close any child objects associated with the session
//        before closing a given session.

hDBISes BEngine::getHandle(void)
{
    hDBISes temp;

    getCurrSession(temp);

    return temp;
}

Retcode BEngine::startSession(hDBISes &sessionHandle, const char *netFile)
{
    if (!isOpen)
    {
        lastError = PXERR_ENGINENOTOPEN;
        return lastError;
    }

    lastError = DbiStartSession(NULL, &sessionHandle, (pCHAR)netFile);

    return lastError;
}

Retcode BEngine::getCurrSession(hDBISes &sessionHandle)
{
    if (!isOpen)
    {
        lastError = PXERR_ENGINENOTOPEN;
        return lastError;
    }

    lastError = DbiGetCurrSession(&sessionHandle);

    return lastError;
}

Retcode BEngine::setCurrSession(hDBISes &sessionHandle)
{
    if (!isOpen)
    {
        lastError = PXERR_ENGINENOTOPEN;
        return lastError;
    }

    lastError = DbiSetCurrSession(sessionHandle);

    return lastError;
}

Retcode BEngine::closeSession(hDBISes &sessionHandle)
{
    if (!isOpen)
    {
        lastError = PXERR_ENGINENOTOPEN;
        return lastError;
    }

    lastError = DbiCloseSession(sessionHandle);

    return lastError;
}

Retcode BEngine::getSessionInfo(SESInfo &sessionInfo)
{
    if (!isOpen)
    {
        lastError = PXERR_ENGINENOTOPEN;
        return lastError;
    }

    lastError = DbiGetSesInfo(&sessionInfo);

    return lastError;
}

Retcode BEngine::setPrivateDir(const char *Directory)
{
    if (!isOpen)
    {
        lastError = PXERR_ENGINENOTOPEN;
        return lastError;
    }

    lastError = DbiSetPrivateDir((pCHAR)Directory);

    return lastError;
}

// Get the private directory

Retcode BEngine::getPrivateDir(char *Directory)
{
    SESInfo sesInfo;
    
    if (!isOpen)
    {
        lastError = PXERR_ENGINENOTOPEN;
        return lastError;
    }

    lastError = DbiGetSesInfo(&sesInfo);

    if (lastError == DBIERR_NONE)
    {
        strcpy(Directory, sesInfo.szPrivDir);
    }

    return lastError;
}

Retcode BEngine::getSysVersion(UINT16 &version, UINT16 &interfaceLevel,
//BR BEG
#if 0//950515
                               DATE &versionDate, TIME &versionTime)
#else
                               DBIDATE &versionDate, TIME &versionTime)
#endif//950515
//BR END
{
    SYSVersion sysVersion;

    if (!isOpen)
    {
        lastError = PXERR_ENGINENOTOPEN;
        return lastError;
    }

    lastError = DbiGetSysVersion(&sysVersion);

    version = sysVersion.iVersion;
    interfaceLevel = sysVersion.iIntfLevel;
    versionDate = sysVersion.dateVer;
    versionTime = sysVersion.timeVer;

    return lastError;
}

Retcode BEngine::getSysConfig(BOOL   &localShare,
                              UINT16 &netProtocol,
                              BOOL   &netShare,
                              CHAR   *netType,
                              CHAR   *userName,
                              CHAR   *iniFile,
                              CHAR   *langDriver)
{
    SYSConfig configInfo;

    if (!isOpen)
    {
        lastError = PXERR_ENGINENOTOPEN;
        return lastError;
    }

    lastError = DbiGetSysConfig(&configInfo);

    localShare  = configInfo.bLocalShare;
    netProtocol = configInfo.iNetProtocol;
    netShare    = configInfo.bNetShare;
    strcpy(netType, configInfo.szNetType);
    strcpy(userName, configInfo.szUserName);
    strcpy(iniFile, configInfo.szIniFile);
    strcpy(langDriver, configInfo.szLangDriver);

    return lastError;
}

Retcode BEngine::getClientInfo(CHAR   *name,
                               UINT16 &sessions,
                               CHAR   *workingDirectory,
                               CHAR   *language)
{
    CLIENTInfo clientInfo;

    if (!isOpen)
    {
        lastError = PXERR_ENGINENOTOPEN;
        return lastError;
    }

    lastError = DbiGetClientInfo(&clientInfo);

    strcpy(name, clientInfo.szName);
    sessions = clientInfo.iSessions;
    strcpy(workingDirectory, clientInfo.szWorkDir);
    strcpy(language, clientInfo.szLang);

    return lastError;
}

Retcode BEngine::getSysInfo(UINT16 &bufferSpace,
                            UINT16 &heapSpace,
                            UINT16 &drivers,
                            UINT16 &clients,
                            UINT16 &sessions,
                            UINT16 &databases,
                            UINT16 &cursors)
{
    SYSInfo sysInfo;

    if (!isOpen)
    {
        lastError = PXERR_ENGINENOTOPEN;
        return lastError;
    }

    lastError = DbiGetSysInfo(&sysInfo);

    bufferSpace = sysInfo.iBufferSpace;
    heapSpace   = sysInfo.iHeapSpace;
    drivers     = sysInfo.iDrivers;
    clients     = sysInfo.iClients;
    sessions    = sysInfo.iSessions;
    databases   = sysInfo.iDatabases;
    cursors     = sysInfo.iCursors;

    return lastError;
}

Retcode BEngine::openDriverList(BListCursor<DRVType> &driverList)
{
    TABLEHANDLE tabH;

    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }

    lastError = DbiOpenDriverList(&tabH);
    if (lastError != DBIERR_NONE)
    {
        return lastError;
    }

    lastError = driverList.attach(tabH);

    return lastError;
}

Retcode BEngine::openDatabaseList(BListCursor<DBDesc> &databaseList)
{
    TABLEHANDLE tabH;

    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }

    lastError = DbiOpenDatabaseList(&tabH);
    if (lastError != DBIERR_NONE)
    {
        return lastError;
    }

    lastError = databaseList.attach(tabH);

    return lastError;
}

Retcode BEngine::openTableTypesList(BListCursor<TBLType> &tableTypesList,
                                    const char *driver)
{
    TABLEHANDLE tabH;

    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }

    lastError = DbiOpenTableTypesList((pCHAR)driver, &tabH);
    if (lastError != DBIERR_NONE)
    {
        return lastError;
    }

    lastError = tableTypesList.attach(tabH);

    return lastError;
}

Retcode BEngine::openFieldTypesList(BListCursor<FLDType> &fieldTypesList,
                                    const char *driver,
                                    const char *tableType)
{
    TABLEHANDLE tabH;

    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }

    lastError = DbiOpenFieldTypesList((pCHAR)driver, (pCHAR)tableType, &tabH);
    if (lastError != DBIERR_NONE)
    {
        return lastError;
    }

    lastError = fieldTypesList.attach(tabH);

    return lastError;
}

Retcode BEngine::openIndexTypesList(BListCursor<IDXType> &indexTypesList,
                                    const char *driver)
{
    TABLEHANDLE tabH;

    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }

    lastError = DbiOpenIndexTypesList((pCHAR)driver, &tabH);
    if (lastError != DBIERR_NONE)
    {
        return lastError;
    }

    lastError = indexTypesList.attach(tabH);

    return lastError;
}

Retcode BEngine::openLanguageDriverList(BListCursor<LDDesc> &languageDriverList)
{
    TABLEHANDLE tabH;

    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }

    lastError = DbiOpenLdList(&tabH);
    if (lastError != DBIERR_NONE)
    {
        return lastError;
    }

    lastError = languageDriverList.attach(tabH);

    return lastError;
}

Retcode BEngine::openUserList(BListCursor<USERDesc> &userList)
{
    TABLEHANDLE tabH;

    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }

    lastError = DbiOpenUserList(&tabH);
    if (lastError != DBIERR_NONE)
    {
        return lastError;
    }

    lastError = userList.attach(tabH);

    return lastError;
}

Retcode BEngine::openCfgInfoList(BListCursor<CFGDesc> &cfgList,
                                 const char *cfgPath,
                                 hDBICfg cfgH,
                                 DBIOpenMode openMode,
                                 CFGMode cfgMode)
{
    TABLEHANDLE tabH;

    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }

    lastError = DbiOpenCfgInfoList(cfgH, openMode, cfgMode, (pCHAR)cfgPath,
                                   &tabH);
    if (lastError != DBIERR_NONE)
    {
        return lastError;
    }

    lastError = cfgList.attach(tabH);

    return lastError;
}

Retcode BEngine::useIdleTime(void)
{
    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }

    lastError = DbiUseIdleTime();

    return lastError;
}

Retcode BEngine::registerCallBack(CBType callBackType, UINT32 clientData,
                                  UINT16 callBackBufferLength,
                                  pVOID callBackBuffer,
                                  pfDBICallBack callBackFunction)
{
    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }

    lastError = DbiRegisterCallBack(NULL, callBackType, clientData,
                                    callBackBufferLength, callBackBuffer,
                                    callBackFunction);

    return lastError;
}

Retcode BEngine::getCallBack(CBType callBackType, UINT32 &clientData,
                             UINT16 &callBackBufferLength,
                             ppVOID callBackBuffer,
                             ppfDBICallBack callBackFunction)
{
    if (!isOpen)
    {
        return (lastError = PXERR_ENGINENOTOPEN);
    }

    lastError = DbiGetCallBack(NULL, callBackType, &clientData,
                               &callBackBufferLength, callBackBuffer,
                               callBackFunction);

    return lastError;
}

char * BEngine::nameOf() const
{
  return ("BEngine");
}

void BEngine::printOn(ostream& os)
{
  os << nameOf() << "{ Engine Type = " << (int) engineType
     << ", Open Status = " << (int) isOpen << "}\n";
}

static void
makeEngInst()
{
    BListCursor<USERDesc> bListUserDesc;
    BListCursor<LDDesc> bListLdDesc;
    BListCursor<IDXType> bListIdxType;
    BListCursor<FLDType> bListFldType;
    BListCursor<TBLType> bListTblType;
    BListCursor<DBDesc> bListDbDesc;
    BListCursor<DRVType> bListDrvType;
    BListCursor<CFGDesc> bListCfgDesc;
}

