/*********************************************************************
**
**                          BCURSOR.H
**
** This file defines the interface to the BCursor class. This class
** encapsulates the notion of a table cursor, an active connection
** to a Paradox table that allows iteration through its records.
** More than one cursor can be open on a table at the same time.              
**
*********************************************************************/

// DBF - (C) Copyright 1994 by Borland International

#ifndef BCURSOR_H 
#define BCURSOR_H

#include "envdef.h"
#include "bdbobjec.h"

class BCursor : public BDbObject {

    friend class BDatabase;
    friend class BRecord;

public:

    // Generic Record for this cursor used internally for some custom record
    // transfer operations. Application programs are free to use this object
    // for the cursor's current record operations.

    BRecord *genericRec;
    Retcode lastError;
    BOOL    isOpen;

public:

    // Constructor. Make a BCursor object without opening any Paradox tables.

    BCursor();

    // Constructor; make a BCursor object and open the specified Paradox table.

    BCursor(BDatabase *db,
            const char *tableName,
            int indexID = 0,
            BOOL saveEveryChange = FALSE,
            DBIOpenMode openMode = dbiREADWRITE,
            DBIShareMode shareMode = dbiOPENSHARED,
            XLTMode xltMode = xltFIELD);

    // Constructor; make a BCursor object and open the specified table of the
    // specified type

    BCursor(BDatabase *db,
            const char *tableName,
            const char *tableType,
            int indexID,
            DBIOpenMode openMode = dbiREADWRITE,
            DBIShareMode shareMode = dbiOPENSHARED,
            XLTMode xltMode = xltFIELD);

    // Constructor; make a BCursor object and open the specified table of the
    // specified type. Use the indexName to identify the index. Note that for
    // dBASE tables, the indexName is used as the indexTagName and the
    // <tablename>.MDX is used as the indexname (idnetifying the production
    // index). 

    BCursor(BDatabase *db,
            const char *tableName,
            const char *indexName,
            DBIOpenMode openMode = dbiREADWRITE,
            DBIShareMode shareMode = dbiOPENSHARED,
            XLTMode xltMode = xltFIELD);

    // Constructor; make a BCursor object and open the specified table of the
    // specified type. Use the indexName and indexTagName to identify the
    // index. Needed for dBASE tables

    BCursor(BDatabase *db,
            const char *tableName,
            const char *indexName,
            const char *indexTagName,
            DBIOpenMode openMode = dbiREADWRITE,
            DBIShareMode shareMode = dbiOPENSHARED,
            XLTMode xltMode = xltFIELD);

//BR BEG
#if 0//950524
#else
    // Copy constructor: uses DBICloneCursor to duplicate cursor
    BCursor(const BCursor &copyCur);

    // Assignment operator: Assign a new cursor to this one, duplicating it in
    //   the process.
    BCursor& operator=(const BCursor &copyCur);
#endif//950524
//BR END

    // Destructor; closes the engine if it's open.

    virtual ~BCursor();

    // Open a cursor on the specified table. indexID specifies the
    // index to use.

    virtual Retcode open(BDatabase *db,
                         const char *tableName,
                         int indexID = 0,
                         BOOL saveEveryChange = FALSE,
                         DBIOpenMode openMode = dbiREADWRITE,
                         DBIShareMode shareMode = dbiOPENSHARED,
                         XLTMode xltMode = xltFIELD);
  
    // Open a cursor on the specified table. indexID specifies the 
    // index to use.
  
    virtual Retcode open(BDatabase *db,
                         const char *tableName,
                         const char *tableType,
                         int indexID,
                         DBIOpenMode openMode = dbiREADWRITE,
                         DBIShareMode shareMode = dbiOPENSHARED,
                         XLTMode xltMode = xltFIELD);

    // Open a cursor on the specified table. Use the indexName to identify
    // the index. Note that for dBASE tables, the indexName is used as the
    // indexTagName and the <tablename>.MDX is used as the indexname
    // (identifying the production index).

    virtual Retcode open(BDatabase *db,
                         const char *tableName,
                         const char *indexName,
                         DBIOpenMode openMode = dbiREADWRITE,
                         DBIShareMode shareMode = dbiOPENSHARED,
                         XLTMode xltMode = xltFIELD);

    // Open a cursor on the specified table. Use the indexName
    // and the indexTagName to specify the index
  
    virtual Retcode open(BDatabase *db,
                         const char *tableName,
                         const char *indexName,
                         const char *indexTagName,
                         DBIOpenMode openMode = dbiREADWRITE,
                         DBIShareMode shareMode = dbiOPENSHARED,
                         XLTMode xltMode = xltFIELD);
  
    // Attach the cursor to an existing TABLEHANDLE
    
    Retcode attach(BDatabase *db, TABLEHANDLE tableHandle);
               
    // Close the cursor if it's open.

    virtual Retcode close();
  
    // Append a record to the end of table. The record can be a 
    // generic or custom record.
  
    virtual Retcode appendRec(BRecord *rec);
  
    // Append a record to the end of table using raw I/O mode 
    // the Paradox Engine. 
    
    virtual Retcode appendRec(const void far *buffer, int size);

    // Insert the generic record
    
    virtual Retcode insertRec();

    // Insert a record at the current cursor position. The record
    // can be a generic or custom record.
  
    virtual Retcode insertRec(BRecord *rec);
  
    // Insert a record at the current cursor position using raw I/O. 
  
    virtual Retcode insertRec(const void far *buffer, int size);
  
    // Delete the current record of the cursor.
  
    virtual Retcode deleteRec();

    // Update the current record using the generic record

    virtual Retcode updateRec();

    // Update the current record of the cursor. The record can be
    // a generic or custom record.
  
    virtual Retcode updateRec(BRecord *rec);

    // Update the current of the cursor using the Paradox
    // Engine's raw I/O mode.

    virtual Retcode updateRec(const void far *buffer, int size);
  
    virtual Retcode clearRec(BRecord* rec);
    virtual Retcode clearRec();

    // Position the cursor before the first record of the table.
  
    Retcode gotoBegin() const;

    // Position past the last record of the table.

    Retcode gotoEnd() const;

    // Go to a specific record number in the table.
  
    Retcode gotoRec(RECORDNUMBER recNum) const;
  
    // Go to the next record of the table.
  
    Retcode gotoNext() const;
  
    // Go to the previous record of the table.
  
    Retcode gotoPrev() const;
  
    // Get the record number of the current record. 
  
    RECORDNUMBER getCurRecNum();

    // Get the current record of the cursor, which can be a generic
    // record or a custom record.

    virtual Retcode getRecord(BRecord *rec) const;

    // Get the current record of the cursor in its genericRec record variable.

    virtual Retcode getRecord() const;

    // Get the next record of the cursor, which can be a generic
    // record or a custom record.

    virtual Retcode getNextRecord(BRecord *rec) const;

    // Get the next record of the cursor in its genericRec record variable.

    virtual Retcode getNextRecord() const;

    // Get the next record of the cursor, which can be a generic
    // record or a custom record.

    virtual Retcode getPreviousRecord(BRecord *rec) const;

    // Get the next record of the cursor in its genericRec record variable.

    virtual Retcode getPreviousRecord() const;

    // Get the current record of the cursor using raw I/O.
  
    virtual Retcode getRecord(void far *buffer, int size);
  
    // Lock the table in a shared environment in the requested lock mode.  
  
    Retcode lockTable(PXLockMode lockMode);

    // Release a previously acquired lock on a table.
  
    Retcode unlockTable(PXLockMode lockMode);
  
    // Get a lock for the current record of the cursor.
  
    LOCKHANDLE lockRecord();
  
    // Release the previously acquired lock on a record.
  
    Retcode unlockRecord(LOCKHANDLE lckH);

    // Release the lock on the current record.
    
    Retcode unlockRecord();

    // Determine whether the current record is locked by another user.
  
    BOOL isLocked();
  
    // Determine if the table of the cursor was changed by another
    // user on the network since the last refresh of the buffers.
  
    BOOL hasChanged();
  
    // Switch the current record which the table is sorted on to the
    // specified index
    
    Retcode switchToIndex(const char *indexName,
                          const char *tagName = NULL);
  
    // Switch the current record which the table is sorted on to the
    // specified index
  
    Retcode switchToIndex(int indexID);
  
    // Refresh the cursor's buffer to reflect changes that
    // may have occurred.
  
    Retcode refresh();
  
    // Find the number of records in the cursor (or table).
  
    RECORDNUMBER getRecCount();

    //  Clone a cursor with its associated properties and return 
    //  the new cursor.
  
    virtual BCursor *clone(BOOL stayCurrent = TRUE);
  
    // Position the cursor on the same record as the current record
    // of another cursor on the same table.
  
    Retcode setToCursor(BCursor *otherCursor);

    // Search the table for the record that matches key values in keyRec. 
    // The currently open index determines the order of records for the search. 
  
    Retcode searchIndex(const BRecord *keyRec,
                        PXSearchMode mode, int fldCnt = 1);

    // Search the table for a record that matches key values supplied as 
    // parameters in the argument list. 
  
    Retcode searchIndex(PXSearchMode mode, int fldCnt, ...);

    // Limit the result set of the table using a range

    Retcode setRange(BRecord &highRecord, BRecord &lowRecord,
                     BOOL incHigh = TRUE, BOOL incLow = TRUE);
  
    // Remove any ranges that are set on the Cursor
    
    Retcode resetRange();
  
    // Limit the result set of the table using a filter
  
    Retcode addFilter(pCANExpr canExpr, hDBIFilter &filterHandle);
  
    // Activate the specified filter
  
    Retcode activateFilter(hDBIFilter filterHandle);
  
    // Deactivate the specified filter.

    Retcode deactivateFilter(hDBIFilter filterHandle);
  
    // Drop the specified filter

    Retcode dropFilter(hDBIFilter filterHandle);
  
    // Link the cursor to another cursor
    
    Retcode linkToCursor(BCursor &masterCursor, pUINT16 masterFields,
                         pUINT16 detailFields);
  
    // Unlink the cursor from the master cursor
    
    Retcode unlinkFromCursor();
    
    // Set a property of the Cursor
  
    Retcode setProp(UINT32 prop, UINT32 value);
  
    // Get a property of the Cursor

    Retcode getProp(UINT32 prop, pVOID value, UINT16 maxLen,
                    UINT16 &retLen);

    // Get the properties of the table

    Retcode getProperties(CURProps &curProps);
    
    // Store the current location in the table
  
    Retcode getBookMark(BOOKMARK& bookMark);
  
    // Set the cursor to a stored location in the table
  
    Retcode setToBookMark(BOOKMARK bookMark);

    // Compare two bookMarks

    Retcode compareBookMarks(BOOKMARK bookMark1, BOOKMARK bookMark2,
                             CMPBkMkRslt &rslt);

    // Free memory allocated for the bookMark

    Retcode freeBookMark(BOOKMARK bookMark);

    // Limit the number of fields that are displayed
    // NOTE: Need to changed this function to take a FieldMap as the
    // parameter.

    Retcode setFieldMap(UINT16 numberOfFields, FLDDesc *fieldDesc);

    // Remove any Field Mappings

    Retcode dropFieldMap();

    // dBASE only. If the current record is marked as Deleted,
    //  will undelete that record.

    Retcode undeleteRecord();

    // dBASE only. Remove undeleted records. Regenerate
    //  maintained indexes.

    Retcode BCursor::packTable(BOOL regenIndexes = TRUE);

    // Get the handle to the Cursor. WARNING: Be carefull when using
    // the handle outside of the class.

    TABLEHANDLE getHandle();

    // Get the status of the curosr

    virtual PXCursorStatus getCurStatus();

    // Function called whenever the underlying cursor has changed.
    //   For example, if the current record has changed.

    virtual Retcode curChange();

    // Redefine pure virtuals from the BDbObject class.

    virtual char *nameOf() const;
    virtual void printOn( ostream& os);

protected:
    char              *tabname;
    char              tabtype[DBIMAXNAMELEN+1];
    TABLEHANDLE       tabH;              // Table handle for cursor.
    TABLEHANDLE       *masterTabH;       // Handle used for Linked Cursors
    PXCursorStatus    curStatus;         // At Begin, End, Crack, or
                                         // on a record.
  
protected:

    // Don't allow the copy constructor to be called unless specifically
    //   defined
    BCursor(BCursor &copyCur);
    // Don't allow the assignment opperator to be called unless specifically
    //   defined
    BCursor& operator=(BCursor &copyCur);

    // Get the IDAPI handle to a Table - only called from BCursor methods
    virtual Retcode openTable(BDatabase *db,
                              const char *tableName,
                              const char *tblType,
                              int indexID,
                              DBIOpenMode openMode = dbiREADWRITE,
                              DBIShareMode shareMode = dbiOPENSHARED,
                              XLTMode xltMode = xltFIELD);

private:
//BR BEG
#if 0//950524
#else
    //
    //  CloneInternals:  used by copy ctor & assignment, duplicates internal
    //     state of a cursor into the current object
    void            CloneInternals( const BCursor & other );
#endif//950524
//BR END
    void            *curobj;             // Variable used to keep track of
                                         // cursor's objects.
    TIMESTAMP       lastChange;
};

#endif

