/*\t*******************************************************************
   Filename  ...........  SPLASH.CPP
   Project .............  Windows support
   Author  .............  Matthew J. W. Ratcliff
   Language  ...........  C++
   Operating System  ...  Windows
   Function:    Display a logo bitmap "splash screen" and then
        start another application with WinExec.

  (c) Ratware Softworks 1995 - Matthew J. W. Ratcliff - Free to use in your
  applications if this copyright notice remains in the source code.
 **\t******************************************************************/

/*\r********************** Revision History ****************************
   Date    By           Change Description
dd-mmm-yy  nnn          text
---------  ----         -----------------------------------------------
21-Jan-94  MJWR         Generic SPLASH SCREEN application launcher.
                        User provides RC file that defines a bitmap
                        file of his choosing. The name must be SPLASH
                        in the resource file.

**\r*/

/*\i******************** System Include Files *************************/
#include <windows.h>
#include <mmsystem.h>
#include <string.h>
#include <stdio.h>

/********************** Constant Include Files ************************/
#include "portable.h"

/***************** External Procedure Include Files *******************/
#include "splash.h"

/*\i*/

/*\m********************** Module Constants ***************************/
#define SPM_TIMER_ID 1
#define SPM_STRLEN 240

/************************* Module Variables ***************************/
static HINSTANCE       spmHinst;
static HINSTANCE       spmPrevHinstance;
static LPSTR           spmLpCmdLine;
static int             spmNcmdShow;

// Handle to the LoadResource() result
// for the "SPLASH" bitmap in the resource
// file.
static HGLOBAL         spmHglblBmp;
static int             spmXwidth;
static int             spmYheight;
static int             spmXpos;
static int             spmYpos;
static int             spmCrtWidth;
static int             spmCrtHeight;
static HPALETTE        spmHpalette = NULL;

static BOOL            spmStretch;
static UINT            spmTimer;
static char           *spmExecStr;

static char           *spmPath;
static char           *spmFile;
static char            spmAppName[SPM_STRLEN];
static char            spmClassName[SPM_STRLEN];
static HANDLE          spmHspeak;


#define SPM_DEBUG_BMP   0 // 0=normal, 1=show bitmap details instead of
                          // bitmap image.

#define SPM_DEBUG_WINEX 0 // 0=normal, 1=disable WinExec calls, Set to 1
                          // when working on the Bitmap code and you
                          // don't actually want to execute anything.
/************************* Module Procedures **************************/
long FAR PASCAL WndProc(HWND, UINT, UINT, LONG);

/*\m*/
HPALETTE    splashGetPalette
( LPBITMAPINFO pasBmi )
{
/*********** Local Constant & Variable Declarations *******************/
LOGPALETTE      *splLogPal;
int              splPalSize;
PALETTEENTRY    *splPalPtr;
RGBQUAD         *splRgbPtr;
WORD             splInx;
HPALETTE         splHpal;
/************************* Procedure Body *****************************/
if (!spmHpalette)
  {
  // Create a palette
  splPalSize               = sizeof(LOGPALETTE) + 
                               pasBmi->bmiHeader.biClrUsed * 
                                 sizeof(PALETTEENTRY);
  splLogPal                = (LOGPALETTE *)new char[splPalSize];
  splPalPtr                = &splLogPal->palPalEntry[0];
  splLogPal->palVersion    = 0x300;
  splLogPal->palNumEntries = (WORD)pasBmi->bmiHeader.biClrUsed;
  splRgbPtr                = (RGBQUAD *)((LPSTR)pasBmi + 
                                          pasBmi->bmiHeader.biSize);
  /* copy colors from the color table to the LogPalette structure */
  for (splInx=0; splInx < pasBmi->bmiHeader.biClrUsed; splInx++)
    {
    splPalPtr[splInx].peRed   = splRgbPtr->rgbRed;
    splPalPtr[splInx].peGreen = splRgbPtr->rgbGreen;
    splPalPtr[splInx].peBlue  = splRgbPtr->rgbBlue;
    splPalPtr[splInx].peFlags = 0;
    splRgbPtr++;
    }
  // Create the palette
  splHpal = CreatePalette(splLogPal);
  // After creation, we can release our copy of it
  LIMDELA(splLogPal);
  if (splHpal)
    {
    spmHpalette = splHpal;
    }
  else
    {
    splHpal     = (HPALETTE)GetStockObject(DEFAULT_PALETTE);
    }
  }
else
  {
  splHpal = spmHpalette;
  }
return(splHpal);
}
/*\p********************************************************************
**                                                                    **
    NAME:  splashScreen

    PURPOSE:  to display the splash bitmap.  If the bitmap handle
      is null, then simply print some text to the window
      indicating that we're starting some application.

**Matthew J. W. Ratcliff                                              **
**\p*******************************************************************/
void splashScreen( HWND pasHwnd )
{
/*********** Local Constant & Variable Declarations *******************/
char            splInfo[SPM_STRLEN];
LPBITMAPINFO    splBmi;
HDC             splHdc;
PAINTSTRUCT     splPs;
RECT            splRect;
HPALETTE        splHpal;
int             splNumColors;
WORD            splOffBits;
HPALETTE        splHoldPal;
/************************* Procedure Body *****************************/
splHdc         = BeginPaint(pasHwnd,&splPs);
splRect.left   = 0;
splRect.top    = 0;
splRect.right  = spmXwidth;
splRect.bottom = spmYheight;

if (spmHglblBmp)
  {
  splBmi = (LPBITMAPINFO)LockResource(spmHglblBmp);
#if SPM_DEBUG_BMP
  sprintf(splInfo,
"Width: %d\r\n\
Height: %d\r\n\
Colors: %d\r\n\
Planes: %d\r\n\
BitsPerPix: %d",
  (int)splBmi->bmiHeader.biWidth,
  (int)splBmi->bmiHeader.biHeight,
  (int)splBmi->bmiHeader.biClrUsed,
  (int)splBmi->bmiHeader.biPlanes,
  (int)splBmi->bmiHeader.biBitCount );
  DrawTextEx(splHdc,splInfo,strlen(splInfo),&splRect,DT_WORDBREAK,NULL);
#else // draw a bitmap baby
  // Create or fetch palette handle
  splHpal    = splashGetPalette( splBmi );
  // Use the palette
  splHoldPal = SelectPalette(splHdc, splHpal, FALSE);
  // Debug: Look at splNumColors to see how many colors were realized for us
  splNumColors = RealizePalette(splHdc);
  // Now draw it
  splOffBits = (WORD)splBmi->bmiHeader.biSize + 
                 (WORD)splBmi->bmiHeader.biClrUsed * sizeof(RGBQUAD);
  SetDIBitsToDevice( splHdc,0,0,
                    (WORD)splBmi->bmiHeader.biWidth,
                    (WORD)splBmi->bmiHeader.biHeight,
                    0,0,0,
                    (WORD)splBmi->bmiHeader.biHeight,
                    (LPSTR)splBmi + splOffBits,
                    splBmi, DIB_RGB_COLORS );
  SelectPalette(splHdc,splHoldPal,FALSE);
#endif

#ifndef WIN32
  // UnlockResource if !WIN32
  UnlockResource(spmHglblBmp);
#endif
  }
else
  {
  sprintf(splInfo,"Start Application: %s", spmAppName );
  DrawText(splHdc,splInfo,strlen(splInfo),&splRect,DT_WORDBREAK);
  }
EndPaint(pasHwnd,&splPs);
} /* splashScreen end */

/*\p********************************************************************
**                                                                    **
    NAME:  Splash::Splash 

    PURPOSE:  to construct the Splash class.  Get setup to show a
      splash screen when the user calls the Launch member
    function.

** By Matthew J. W. Ratcliff                                          **
**\p*******************************************************************/

Splash::Splash ( HINSTANCE       pasHinst,
                 HINSTANCE       pasPrevHinstance,
                 LPSTR           pasLpCmdLine,
                 int             pasNcmdShow )
{
/*********** Local Constant & Variable Declarations *******************/

int          splCrtWidth;
int          splCrtHeight;
char        *splCptr;

LPBITMAPINFO splBmi;
HRSRC        splHrsrcBmp;

/************************* Procedure Body *****************************/
spmHspeak          = NULL;
GetModuleFileName( pasHinst, spmAppName, SPM_STRLEN );
spmHinst           = pasHinst;
spmPrevHinstance   = pasPrevHinstance;
spmLpCmdLine       = pasLpCmdLine;
spmNcmdShow        = pasNcmdShow;

/***********************************************
* Now load the bitmap. If not found, tell
* the user about the faux paux
*/
splHrsrcBmp  = FindResource(spmHinst,"SPLASH",RT_BITMAP);
if (splHrsrcBmp)
  {
  spmHglblBmp = LoadResource(spmHinst,splHrsrcBmp);
  }
if (spmHglblBmp == NULL)
  {
  MessageBeep(0);
  MessageBox( NULL,
"The bitmap 'SPLASH' was not found.\r\n\
The entry: SPLASH BITMAP \"mylogo.bmp\"\r\n\
MUST appear in your resource (.RC) file\r\n\
for this program.",
  spmAppName,
  MB_OK | MB_TASKMODAL | MB_ICONHAND );
  spmXwidth    = 320;
  spmYheight   = 200;
  }
else
  {
  splBmi = (LPBITMAPINFO)LockResource(spmHglblBmp);
  if (splBmi)
    {
    // Width and Height of bitmap
    spmXwidth    = splBmi->bmiHeader.biWidth;
    spmYheight   = splBmi->bmiHeader.biHeight;
#ifndef WIN32
    UnlockResource(spmHglblBmp);
#endif
    }
  else
    { // arbitrary default values
    spmXwidth  = 320;
    spmYheight = 200;
    }
  }
// Add to size of bitmap the size of the window's borders
spmXwidth   += (2*GetSystemMetrics(SM_CXBORDER));
spmYheight  += (2*GetSystemMetrics(SM_CYBORDER));
// Get CRT limits
splCrtWidth  = GetSystemMetrics(SM_CXSCREEN);
splCrtHeight = GetSystemMetrics(SM_CYSCREEN);
// Compute window position that will center
// the bitmap
spmXpos      = (splCrtWidth  - spmXwidth)/2;
spmYpos      = (splCrtHeight - spmYheight)/2;
if (spmXpos    < 0)            spmXpos    = 0;
if (spmYpos    < 0)            spmYpos    = 0;
if (spmXwidth  > splCrtWidth)  spmXwidth  = spmCrtWidth;
if (spmYheight > splCrtHeight) spmYheight = spmCrtHeight;

spmPath = new char[strlen(spmAppName)+1];
spmFile = new char[strlen(spmAppName)+1];
strcpy(spmPath,spmAppName);
splCptr = &spmPath[strlen(spmPath)-1];
// Look for last ocurrence of '\\' character,
// path name separator.
while ((splCptr > spmPath) && (*splCptr != '\\'))
  {
  splCptr--;
  }
if (*splCptr == '\\')
  {
  splCptr++;
  *splCptr = '\0';
  }
else
  {
  splCptr = strchr(spmPath,':'); // No '\\', look for drive separator
  if (splCptr)
    {
    splCptr++;
    *splCptr = '\0';
    }
  else
    {
    *spmPath = '\0'; // No drive/path spec at all
    }
  }
// spmPath contains path to this program
// spmFile will receive the rest of the app name string
splCptr = &spmAppName[strlen(spmPath)];
strcpy(spmFile,splCptr);

} /* Splash::Splash  end */

/*\p********************************************************************
**                                                                    **
    NAME:  Splash::~Splash 

    PURPOSE:  to delete storage allocated for the Splash class

** By Matthew J. W. Ratcliff                                          **
**\p*******************************************************************/

Splash::~Splash ()
{
/************************* Procedure Body *****************************/
LIMDELA(spmPath);
LIMDELA(spmFile);
if (spmHglblBmp)
  {
#ifndef WIN32
  UnlockResource(spmHglblBmp);
#endif
  FreeResource(spmHglblBmp);
  }
if (spmHspeak)
  {
#ifndef WIN32
  UnlockResource(spmHspeak);
#endif
  FreeResource(spmHspeak);
  }

if (spmHpalette)
  {
  DeleteObject( (HGDIOBJ)spmHpalette );
  spmHpalette = 0;
  }


} /* Splash::~Splash  end */

/*\p********************************************************************
**                                                                    **
    NAME:  Splash::Launch

    PURPOSE:  to display the bitmap logo window, launch the
      specified application, wait for our time to expire,
      and then terminate.

** By Matthew J. W. Ratcliff                                          **
**\p*******************************************************************/

int Splash::Launch( int             pasShowTime,// Show time in milliseconds
                    const char     *pasAppName,// Application name to run
                    BOOL            pasExtractAppPath)
{
/*********** Local Constant & Variable Declarations *******************/
int           splLen;
WNDCLASS      splWclass;
HWND          splHwnd;
MSG           splMsg;
char         *splExPtr;

/************************* Procedure Body *****************************/

strcpy(spmClassName,spmFile);
strupr(spmClassName);
splExPtr = strstr(spmClassName,".EXE");
if (splExPtr)
  {
  *splExPtr = '\0';
  }
splLen =strlen(pasAppName)+strlen(spmPath)+strlen(spmLpCmdLine)+20;
spmExecStr = new char[splLen];
if (pasExtractAppPath)
  {
  strcpy(spmExecStr,spmPath);
  strcat(spmExecStr,pasAppName);
  }
else
  {
  strcpy(spmExecStr,pasAppName);
  }
if (spmLpCmdLine)
  {
  if (*spmLpCmdLine)
    {
    strcat(spmExecStr," ");
    strcat(spmExecStr,spmLpCmdLine);
    }
  }
if (!spmPrevHinstance)
  {
  splWclass.style         = CS_HREDRAW | CS_VREDRAW;
  splWclass.lpfnWndProc   = WndProc;
  splWclass.cbClsExtra    = 0;
  splWclass.cbWndExtra    = 0;
  splWclass.hInstance     = spmHinst;
  splWclass.hIcon         = LoadIcon(spmHinst,"SPLASHICON");
  splWclass.hCursor       = LoadCursor(NULL,IDC_ARROW);
  splWclass.hbrBackground = GetStockObject( WHITE_BRUSH );
  splWclass.lpszMenuName  = NULL;
  splWclass.lpszClassName = (LPSTR)spmClassName;

  if (!RegisterClass( &splWclass ))
    {               
    MessageBox( NULL, "Failed to register window class",
                spmClassName,
                MB_OK | MB_ICONASTERISK );
    }
  }
splHwnd = CreateWindow( spmClassName,
                        NULL, //spmClassName,
                        WS_POPUP | WS_BORDER,
                        spmXpos,
                        spmYpos,
                        spmXwidth,
                        spmYheight,
                        NULL,
                        NULL,
                        spmHinst,
                        NULL );
if (!splHwnd)
  {
  MessageBox( NULL, "Failed to create window",
              spmClassName,
              MB_OK | MB_ICONASTERISK );
#if !SPM_DEBUG_WINEX
  WinExec( spmExecStr, SW_SHOW );
#endif
  splMsg.wParam = 0;
  }
else
  {
  if (pasShowTime <= 0)
    {
    pasShowTime = 250; // 1/4 second minimum
    }
  spmTimer = SetTimer(splHwnd, SPM_TIMER_ID, pasShowTime, NULL);
  if (!spmTimer)
    {
    MessageBox(splHwnd, "Too many clocks or timers!",
               spmClassName, MB_ICONEXCLAMATION | MB_OK );
#if !SPM_DEBUG_WINEX
    WinExec( spmExecStr, SW_SHOWNORMAL );
#endif
    PostMessage(splHwnd,WM_DESTROY,0,0);
    }
  ShowWindow( splHwnd, SW_SHOW );
  UpdateWindow( splHwnd );

  while (GetMessage( &splMsg, NULL, 0, 0))
    {
    TranslateMessage( &splMsg );
    DispatchMessage ( &splMsg );
    }
  }
LIMDELA(spmExecStr);
return splMsg.wParam;
} /* Splash::Launch end */

/*\p********************************************************************
**                                                                    **
    NAME:  WndProc

    PURPOSE:  to handle window processing for the Splash program.
        On WM_PAINT paint the logo.  On WM_TIMER timeout and
        shut down this application.

**   Return Val         type/expected/description                     **
**   ------------       --------------------------------------------  **
**   splResult          long, return value for WndProc function       **
**\p*******************************************************************/

long FAR PASCAL WndProc
( HWND  pasHwnd,
  UINT  pasMsg,
  UINT  pasWparam,
  LONG  pasLparam )
{
/*********** Local Constant & Variable Declarations *******************/
long          splResult;
static        splPainted;
LPSTR         splCspeak;

/************************* Procedure Body *****************************/
splResult = 0L;
switch(pasMsg)
  {
  case WM_CREATE:
#if !SPM_DEBUG_BMP
    SetWindowPos( pasHwnd, HWND_TOPMOST, 0, 0, 0, 0, SWP_NOMOVE );
#endif
    splPainted = FALSE;
    break;
  case WM_TIMER:
    if (spmHspeak) sndPlaySound(NULL,0);
    KillTimer( pasHwnd, spmTimer );
    DestroyWindow(pasHwnd);
    break;
  case WM_PAINT:
    splashScreen( pasHwnd );
    if (!splPainted)
      {
      splPainted = TRUE;
#if !SPM_DEBUG_WINEX
      WinExec( spmExecStr, SW_SHOWNORMAL );
#endif
      // SPLASHSPEAK WAVE "MYINTRO.WAV" is defined in
      // the user's resource file if he wishes the
      // Splash program to say something on startup.
      spmHspeak = FindResource( spmHinst, "SPLASHSPEAK", "WAVE" );
      if (spmHspeak)
        {
        spmHspeak = LoadResource( spmHinst, spmHspeak );
        if (spmHspeak)
          {
          splCspeak = (char *)LockResource( spmHspeak );
          if (splCspeak)
            {
            sndPlaySound( splCspeak, SND_ASYNC | SND_MEMORY );
            }
          }
        }
      }
    break;
  case WM_CLOSE:
    if (spmHspeak) sndPlaySound(NULL,0);
    if (spmTimer) KillTimer( pasHwnd, spmTimer );
    DestroyWindow(pasHwnd);
    break;
  case WM_COMMAND:
    switch (pasWparam)
      {
      case IDCANCEL:
      case IDOK:
        if (spmTimer) KillTimer( pasHwnd, spmTimer );
        DestroyWindow(pasHwnd);
        break;
      }
    break;
  case WM_SIZE:
    SetWindowPos( pasHwnd, NULL,
                  spmXpos,
                  spmYpos,
                  spmXwidth,
                  spmYheight,
                  SWP_NOZORDER );
    break;
  case WM_DESTROY:
    if (spmHspeak) sndPlaySound(NULL,0);
    if (spmTimer) KillTimer( pasHwnd, spmTimer );
    PostQuitMessage(0);
    break;
  default:
    splResult = DefWindowProc( pasHwnd, pasMsg, pasWparam, pasLparam );
    break;
  }
return (splResult);
} /* WndProc end */
