/***************************************************************

	sfs_vi.c        Visual Simulation Routines for SFS

			Copyright (c) 1991, Ted A. Campbell

			Bywater Software
			P. O. Box 4023 
			Duke Station 
			Durham, NC  27706

			email: tcamp@hercules.acpub.duke.edu

	Copyright and Permissions Information:

	All U.S. and international copyrights are claimed by the
	author. The author grants permission to use this code
	and software based on it under the following conditions:
	(a) in general, the code and software based upon it may be 
	used by individuals and by non-profit organizations; (b) it
	may also be utilized by governmental agencies in any country,
	with the exception of military agencies; (c) the code and/or
	software based upon it may not be sold for a profit without
	an explicit and specific permission from the author, except
	that a minimal fee may be charged for media on which it is
	copied, and for copying and handling; (d) the code must be 
	distributed in the form in which it has been released by the
	author; and (e) the code and software based upon it may not 
	be used for illegal activities. 

***************************************************************/

#include "stdio.h"
#include "ctype.h"
#include "math.h"
#include "bw.h"
#include "gr.h"
#include "kb.h"
#include "ui.h"
#include "as.h"
#include "sfs.h"

/* #define NOT_HIDDEN */

/***************************************************************

	vi_draw()       Draw visual simulation screen

***************************************************************/

vi_draw( uiwind, focus, elements, g_start, g_end, s_start, s_end,
      o_start, o_end, radius, distance, sorbit_array, orbit_n )
   struct uiwindow *uiwind;     /* ui window to draw in */
   struct as_focus *focus;      /* orbital focus to display */
   unsigned int elements;       /* elements to draw */
   struct spj_pt *g_start;      /* start array of points for grid */
   struct spj_pt *g_end;        /* end array of points for grid */
   struct spj_pt *s_start;      /* start array of points for surface */
   struct spj_pt *s_end;        /* end array of points for surface */
   struct spj_pt *o_start;      /* start array for orb data */
   struct spj_pt *o_end;        /* end array for orb data */
   double radius;               /* radius for size/horizon calculation */
   double distance;             /* distance for size/horizon calculation */
   struct sfs_orbit **sorbit_array;
   int orbit_n;
   {
   int l, c;                            /* line, column for title display */
   int x, y;                            /* coordinates to locate orb */
   static int clip_status = FALSE;      /* is clipping allowed? */
   int x_screen;

#ifdef  DEBUG
   if ( distance == 0.0 )
      {
      sprintf( bw_ebuf, "[pr: vi_draw() received distance = %.2lf",
	 distance );
      bw_error( bw_ebuf );
      return;
      }
   if ( radius == 0.0 )
      {
      sprintf( bw_ebuf, "[pr: vi_draw() received radius = %.2lf",
	 radius );
      bw_error( bw_ebuf );
      return;
      }
#endif

   /***  inform the viewer */

   bw_message( VI_DRAW );

   /***  initial screen setup */

#ifdef  NOT_HIDDEN
   x_screen = GR_PRIMARY;
#else
   if ( gr_screens > 1 )
      {
      x_screen = GR_HIDDEN;
      }
   else
      {
      x_screen = GR_PRIMARY;
      }
   ui_setscreen( x_screen );
#endif

#ifdef  OLD_DEBUG
   bw_message( "DEBUG: Blank window..." );
#endif

   /***  Blank the user area of the window */

   ui_fbox( uiwind->u_x1, uiwind->u_y1,
       uiwind->u_x2, uiwind->u_y2, BLACK, SOLID );

#ifdef  OLD_DEBUG
   bw_message( "DEBUG: Set font..." );
#endif

   /***  Set appropriate font */

   gr_font( x_screen, F_DEFAULT, ui_grwind->ymax / 40 );

   /***  Element 1: Blank the orb */

   /***  Element 2: Draw the orb */

   if ( ( elements & VI_ORBOUTLINE ) > 0 )
      {
#ifdef  OLD_DEBUG
      bw_debug( "ready to draw orb..." );
#endif
      vi_showseq( o_start, o_end, uiwind, HORDEGS, radius, distance,
	 cl_grid, SOLID, AL_HORIZON, SPJ_ALLSIDES, x_screen );
      }

   /***  Element 3: Draw front of the grid */

   if ( ( elements & VI_GRIDFRONT ) > 0 )
      {
      vi_showseq( g_start, g_end, uiwind, HORDEGS, radius, distance,
	 cl_grid, GRID, AL_HORIZON, SPJ_NEARSIDE, x_screen );
      }

   /***  Element 4: Draw reverse of the grid */

   if ( ( elements & VI_GRIDBACK ) > 0 )
      {
      vi_showseq( g_start, g_end, uiwind, HORDEGS, radius, distance,
	 cl_grid, GRID, AL_HORIZON, SPJ_FARSIDE, x_screen );
      }

   /***  Element 5: Draw reverse of the surface features */

   if ( ( elements & VI_SURBACK ) > 0 )
      {
      vi_showseq( s_start, s_end, uiwind, HORDEGS, radius, distance,
	 cl_surface, GRID, AL_HORIZON, SPJ_FARSIDE, x_screen );
      }

   /***  Element 6: Draw front of the surface features */

   if ( ( elements & VI_SURFRONT ) > 0 )
      {
      vi_showseq( s_start, s_end, uiwind, HORDEGS, radius, distance,
	 cl_surface, SOLID, AL_HORIZON, SPJ_NEARSIDE, x_screen );
      }

   /***  Element 7: Show the points */

#ifdef  USEPOINTS
   if ( ( elements & VI_POINTS ) > 0 )
      {
      vi_showseq( p_start, p_end, uiwind, HORDEGS, radius, distance,
	 cl_marker, GRID, AL_HORIZON, SPJ_NEARSIDE, x_screen );
      }
#endif

   /***  Element 8: Show titles for point data */

#ifdef  USEPOINTS
   if ( ( elements & VI_PTITLES ) > 0 )
      {
      vi_titles( p_start, p_end, uiwind, HORDEGS, radius, distance,
	 cl_mback, cl_mfore, AL_HORIZON, SPJ_NEARSIDE, x_screen );
      }
#endif

   /***  Element 9: Show crosshairs */

   if ( ( elements & VI_CROSSH ) > 0 )
      {
      pe_crosshair( uiwind, HORDEGS, x_screen );
      }

   /***  Element 10: Show screen title */

   if ( ( elements & VI_STITLE ) > 0 )
      {
      sprintf( bw_ebuf, "%s:   %s %ld   %s %.2lf   %s %.2lf   %s %.0lf km",
	 VI_POS, VI_ORBIT, sorbit_array[ orbit_n ]->ssp_n,
	 VI_LAT, sorbit_array[ orbit_n ]->ssp_lat,
	 VI_LON, sorbit_array[ orbit_n ]->ssp_lon,
	 VI_ALT, sorbit_array[ orbit_n ]->ssp_r - focus->radius );
      gr_rectangle( x_screen, uiwind->u_x1 + 1, uiwind->u_y1 + 1,
	 uiwind->u_x2 - 1, uiwind->u_y1 + 1 + ui_grwind->fysize + 1,
	 cl_marker, SOLID );
      gr_text( x_screen, uiwind->u_x1 + 1, uiwind->u_y1 + 1,
	 bw_ebuf, cl_mfore, cl_marker );
      }

   /***  Restore font */

   sfsx_font( 0 );

   /* Show toggle status */

   el_show( x_screen, uiwind, elements );

   /***  blit and return screen to original state */

#ifndef NOT_HIDDEN
   if ( gr_screens > 1 )
      {
      gr_blit( GR_HIDDEN, GR_PRIMARY, uiwind->u_x1, uiwind->u_y1,
	 uiwind->u_x2, uiwind->u_y2 );
      }
   ui_setscreen( GR_PRIMARY );
#endif

   /* Title the screen */

   sprintf( bw_ebuf, VI_WTITLE,
      focus->name,
      sorbit_array[ orbit_n ]->name );
   ui_wtitle( uiwind, bw_ebuf );

   }

/***************************************************************

	vi_titles()     Show point titles on visual display

***************************************************************/

#ifdef  USEPOINTS

vi_titles( start, end, uiwind, hor_deg, radius, distance,
   background, foreground, alignment, spjmode, screen )
   struct spj_pt *start, *end;
   struct uiwindow *uiwind;
   double hor_deg, radius, distance;
   int background, foreground;
   int alignment, spjmode, screen;
   {
   struct spj_pt *current;
   static struct uiwindow *oldwind;     /* save window pointer to avoid recalculation */
   static int x_midx, x_midy;           /* x, y mid points of window */
   static double x_xfactor, x_yfactor;  /* factors for x, y expansion in window */
   static double old_hdeg;              /* save horizontal degrees to avoid recalculation */
   static double vert_degrees;          /* calculate vertical degrees */
   static int old_align;                /* save alignment to avoid recalculation */
   int v;                               /* is the point visible? */

   if ( ( uiwind != oldwind ) || ( alignment != old_align )
			      || ( hor_deg != old_hdeg ))
      {
      x_xfactor = ( uiwind->u_x2 - uiwind->u_x1 ) / hor_deg;
      vert_degrees = hor_deg * ( ( ( uiwind->u_y2 - (double) uiwind->u_y1 )
	 * ( gr_pysize / (double) gr_pxsize ) )
	 / ( uiwind->u_x2 - (double) uiwind->u_x1 ) );
      x_yfactor = ( uiwind->u_y2 - uiwind->u_y1 ) / vert_degrees;
      old_hdeg = hor_deg;
      x_midx = uiwind->u_x1 + (( uiwind->u_x2 - uiwind->u_x1 ) / 2 );
      if ( alignment == AL_HORIZON )
	 {
	 x_midy = ( ( ( uiwind->u_y2 - uiwind->u_y1 ) / 3 ) * 2 )
	    - ( x_yfactor * spj_angrad( distance, radius ));
	 }
      else
	 {
	 x_midy = uiwind->u_y1 + (( uiwind->u_y2 - uiwind->u_y1 ) / 2 );
	 }
      old_align = alignment;
      oldwind = uiwind;
      }

   current = start;
   while( current != end )
      {

      /* Check to see if the coordinates are within the bounds
	 of the whole screen */

      v = TRUE;
      if ( current->name == NULL )
	 {
	 v = FALSE;
	 }
      else if ( ( current->x < 0 ) || ( current->x > ui_grwind->xmax ) )
	 {
	 v = FALSE;
	 }
      else if ( ( current->y < 0 ) || ( current->y > ui_grwind->ymax ) )
	 {
	 v = FALSE;
	 }
      else if ( current->side != spjmode )
	 {
	 v = FALSE;
	 }

      if ( v == TRUE )
	 {
	 gr_text( screen,
	    (int) ( current->x * x_xfactor ) + 3 + x_midx,
	    (int) ( current->y * x_yfactor ) - ( ui_grwind->fysize / 2 )
	       + x_midy,
	    current->name, foreground, background );
	 }

      current = current->next;
      }
   }

#endif

/***************************************************************

	vi_showseq()    Show a linked sequence of spherical
			projection data points in a ui window

***************************************************************/

vi_showseq( start, end, uiwind, hor_deg, radius, distance,
   color, linestyle, alignment, spjmode, screen )
   struct spj_pt *start, *end;          /* start, end structures for
					   spherical projection data */
   struct uiwindow *uiwind;             /* ui window for display */
   double hor_deg;                      /* degrees of horizontal axis */
   double radius;                       /* radius for horizon position calculation */
   double distance;                     /* distance for horizon position calculation */
   int color;                           /* gr color for lines, points */
   int linestyle;                       /* gr line style for lines */
   int alignment;                       /* horizon or centered alignment */
   int spjmode;                         /* spj mode (sides) for display */
   int screen;                          /* screen for display */
   {
   struct spj_pt *current;              /* current point structure */
   static struct uiwindow *oldwind;     /* save window pointer to avoid recalculation */
   static int x_midx, x_midy;           /* x, y mid points of window */
   int x, y, side;                      /* current x, y, side values */
   int prevcode, prevx, prevy, prevside;   /* save previous values */
   static double x_xfactor, x_yfactor;  /* factors for x, y expansion in window */
   static double old_hdeg;              /* save horizontal degrees to avoid recalculation */
   static double vert_degrees;          /* calculate vertical degrees */
   static double old_distance;		/* save distance to avoid recalc */
   static int old_align;                /* save alignment to avoid recalculation */
#ifdef	DEBUG
   int Dc;
#endif

#ifdef  DEBUG
   if ( distance == 0.0 )
      {
      sprintf( bw_ebuf, "[pr: vi_showseq() received distance = %.2lf",
	 distance );
      bw_error( bw_ebuf );
      return;
      }
   if ( radius == 0.0 )
      {
      sprintf( bw_ebuf, "[pr: vi_showseq() received radius = %.2lf",
	 radius );
      bw_error( bw_ebuf );
      return;
      }
#endif

   if ( ( uiwind != oldwind ) || ( alignment != old_align )
        || ( distance != old_distance )  || ( hor_deg != old_hdeg ))
      {

#ifdef  DEBUG
      bw_message( "DEBUG: recalculating for showseq()..." );
#endif

      x_xfactor = ( uiwind->u_x2 - (double) uiwind->u_x1 ) / hor_deg;
      vert_degrees = hor_deg * (double)
	 ( ( ( uiwind->u_y2 - (double) uiwind->u_y1 )
	 * ( gr_pysize / (double) gr_pxsize ) )
	 / ( uiwind->u_x2 - (double) uiwind->u_x1 ) );
      x_yfactor = ( uiwind->u_y2 - (double) uiwind->u_y1 ) / vert_degrees;
      old_hdeg = hor_deg;
      x_midx = uiwind->u_x1 + (( uiwind->u_x2 - (double) uiwind->u_x1 ) / 2.0 );
      if ( alignment == AL_HORIZON )
	 {
	 x_midy = uiwind->y1 +
	    ( ( (double) uiwind->u_y2 - (double) uiwind->u_y1 ) * 0.7 )
	    - ( x_yfactor * spj_angrad( distance, radius ));
	 }
      else
	 {
	 x_midy = uiwind->u_y1 + (( uiwind->u_y2 - (double) uiwind->u_y1 ) / 2 );
	 }
      old_align = alignment;
      oldwind = uiwind;
      old_distance = distance;
#ifdef  OLD_DEBUG
      sprintf( bw_ebuf, "vert_degrees = %.2lf  xfactor = %.2lf, yfactor = %.2lf",
         vert_degrees, x_xfactor, x_yfactor );
      bw_debug( bw_ebuf );
#endif

      }

#ifdef  OLD_DEBUG
   Dc = 0;
   sprintf( bw_ebuf, "ready to draw points, mode %d", spjmode );
   bw_debug( bw_ebuf );
#endif

   current = start->next;
   prevside = 255;

   while( current != end )
      {
      x = current->x * x_xfactor;
      y = current->y * x_yfactor;
      side = current->side;
#ifdef	DEBUG
      ++Dc;
#endif

      switch ( spjmode )
	 {
	 case SPJ_ALLSIDES:
#ifdef  OLD_DEBUG
            sprintf( bw_ebuf, "DEBUG: allsides point %d: x %d y %d", Dc, x, y );
            bw_message( bw_ebuf );
#endif
	    if ( current->code < 1000 )
	       {
	       sfs_line( screen,
		  x_midx + x,
		  x_midy + y,
		  x_midx + prevx,
		  x_midy + prevy, color, linestyle );
	       }
	    else
	       {
	       sfs_pixel( screen, x_midx + x,
		  x_midy + y, color );
	       }
	    break;
	 case SPJ_FARSIDE:
#ifdef  OLD_DEBUG
            sprintf( bw_ebuf, "DEBUG: farside point %d: x %d y %d", Dc, x, y );
            bw_message( bw_ebuf );
#endif
	    if ( ( prevside == SPJ_FARSIDE ) && ( side == SPJ_FARSIDE ))
	       {
	       if ( current->code < 1000 )
		  {
		  sfs_line( screen,
		     x_midx + x,
		     x_midy + y,
		     x_midx + prevx,
		     x_midy + prevy, color, linestyle );
		  }
	       else
		  {
		  sfs_pixel( screen, x_midx + x,
		  x_midy + y, color );
		  }
	       }
	    break;
	 default:
#ifdef  OLD_DEBUG
            sprintf( bw_ebuf, "DEBUG: nearside point %d: x %d y %d", Dc, x, y );
            bw_message( bw_ebuf );
#endif
	    if ( ( prevside == SPJ_NEARSIDE ) && ( side == SPJ_NEARSIDE ))
	       {
	       if ( current->code < 1000 )
		  {
		  sfs_line( screen,
		     x_midx + x,
		     x_midy + y,
		     x_midx + prevx,
		     x_midy + prevy, color, linestyle );
		  }
	       else
		  {
		  sfs_pixel( screen, x_midx + x,
		     x_midy + y, color );
		  }
	       }
	    break;
	 }

      prevx = x;
      prevy = y;
      prevside = side;
      prevcode = current->code;
      current = current->next;
      }
   }

/***************************************************************

	vi_rotate()     Calculate rotation based on 2 positions

***************************************************************/

double
vi_rotate( current_x, current_y, next_x, next_y )
   double current_x;           /* current x coordinate */
   double current_y;           /* current y coordinate */
   double next_x;              /* next x coordinate */
   double next_y;              /* next y coordinate */
   {
   double x, y, r;

   x = next_x - current_x;
   y = next_y - current_y;

   if ( x == 0.0 )
      {
      if ( y == 0.0 )
	 {
	 return 270.0;
	 }
      else if ( y > 0.0 )
	 {
	 return 0.0;
	 }
      else if ( y < 0.0 )
	 {
	 return 180.0;
	 }
      }

   if ( y == 0.0 )
      {
      if ( x > 0.0 )
	 {
	 return 270.0;
	 }
      else if ( x < 0.0 )
	 {
	 return 90.0;
	 }
      }

   r = atan2( y, x ) * RAD_DEG;

#ifdef  OLD_DEBUG
   sprintf( bw_ebuf, "exit vi_rotate(), current_x %.2lf  next_x %.2lf x %.2lf  y %.2lf  r %.2lf",
      current_x, next_x, x, y, r );
   bw_debug( bw_ebuf );
#endif

   return spj_degfix( 270.0 + r );
   }

sfs_line( screen, x1, y1, x2, y2, color, style )
   int screen, x1, y1, x2, y2, color, style;
   {

   if ( ( x1 < 0 ) || ( x1 > ui_grwind->xmax ))
      {
      return FALSE;
      }
   if ( ( x2 < 0 ) || ( x2 > ui_grwind->xmax ))
      {
      return FALSE;
      }
   if ( ( y1 < 0 ) || ( y1 > ui_grwind->ymax ))
      {
      return FALSE;
      }
   if ( ( y2 < 0 ) || ( y2 > ui_grwind->ymax ))
      {
      return FALSE;
      }

   gr_line( screen, x1, y1, x2, y2, color, style );

   }

sfs_pixel( screen, x, y, color )
   int screen, x, y, color;
   {

   if ( ( x < 0 ) || ( x > ui_grwind->xmax ))
      {
      return FALSE;
      }
   if ( ( y < 0 ) || ( y > ui_grwind->ymax ))
      {
      return FALSE;
      }

   gr_pixel( screen, x, y, color );

   }

