///////////////////////////////////////////////////////////
// SCROLL.CPP: Shows how to handle scroll bars in a frame
//             window.
///////////////////////////////////////////////////////////

#include <owl\applicat.h>
#include <owl\framewin.h>
#include <owl\scroller.h>
#include <owl\dialog.h>
#include <owl\edit.h>
#include <owl\dc.h>
#include <owl\printer.h>
#include <stdio.h>
#include "scroll.rc"

#define PRINTBUFFERSIZE 256

static char szPageNumber[] = "Page %i";
static char szBuffer[PRINTBUFFERSIZE+1];

// Transfer buffer.
struct
{
	char unitsEdit[3];
	char lineEdit[3];
	char hRangeEdit[4];
	char vRangeEdit[4];
} transBuf;

// Structure for holding image data.
struct rec
{
	int color;
	int x;
	int y;
};

// The application class.
class TApp : public TApplication
{
public:
	TApp() : TApplication() {}
	void InitMainWindow();
};

// The main window class.
class TWndw : public TFrameWindow
{
private:
	TPrinter* printer;

protected:
	rec recArray[100];

public:
	TWndw(TWindow *parent, const char far *title);
	~TWndw();

protected:
	void Paint(TDC &dc, BOOL, TRect&);
	void CmSetScroll();
	void CmPrint();
	void SetScroller();

	DECLARE_RESPONSE_TABLE(TWndw);
};

DEFINE_RESPONSE_TABLE1(TWndw, TFrameWindow)
	EV_COMMAND(CM_SETSCROLL, CmSetScroll),
	EV_COMMAND(CM_PRINT, CmPrint),
END_RESPONSE_TABLE;

// The Scroll-Values dialog class.
class TDlg : public TDialog
{
public:
	TDlg(TWindow *parent, TResId resId);
};

// The Printout class
class TYourPrintout: public TPrintout{
public:
	TYourPrintout(TWindow* window, const char* title, TPrinter* printer,
					  rec Array[100]);
	~TYourPrintout();
	void BeginPrinting();
	void EndPrinting();
	void EnableBanding() {Banding = TRUE;}

protected:
	void PrintPage(int page, TRect& rect, unsigned flags);
	void GetDialogInfo(int& minPage, int& maxPage, int& selFromPage,
							 int& selToPage);
	void SetPrintParams(TPrintDC* dc, TSize pageSize);
	BOOL HasPage(int pageNumber);

private:
	int Scale;					  // scaling factor
	int maxPages;				  // total number of pages that can be printed
	int xPages, yPages;		  // the number of column and row pages
	int maxX, maxY;			  // the maximum coordinates of rectangle to print
	int xFactor, yFactor;	  // the factor to multiply the next origin by
	TWindow* parentWindow;	  // reference to the parent window
	TPoint nextOrigin;		  // the next origin in the window to begin printing
	int cgPageNumber;
	TPrintDialog::TData& Data;// reference to dialog data
	rec coordArray[100];
	int prevMode;
	TSize oldVExt, oldWExt;
	TPoint oldVOrg, oldWOrg;
};

///////////////////////////////////////////////////////////
// The TWndw class's implementation.
///////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////
// TWndw::TWndw()
//
// This is the main window's constructor.
///////////////////////////////////////////////////////////
TWndw::TWndw(TWindow *parent, const char far *title) :
	TFrameWindow(parent, title)
{
	// Add vertical and horizontal scroll bars
	// to the main window.
	Attr.Style |= WS_VSCROLL | WS_HSCROLL;

	// Position and size the window.
	Attr.X = 50;
	Attr.Y = 50;
	Attr.W = 400;
	Attr.H = 300;

	// Create the window's OWL scroller object.
	Scroller = new TScroller(this, 1, 1, 2560, 1920);

	// Add the menu to the main window.
	AssignMenu(MENU_1);

	// Printer initialization
	printer = new TPrinter;

	// Initialize the image array with the data
	// needed to display 500 colored rectangles.
	randomize();
	for (int i=0; i<100; ++i)
	{
		recArray[i].color = random(16);
		recArray[i].x = random(919);    // added 300
		recArray[i].y = random(759);	  // added 300
	}
}


// The main window's destructor
TWndw::~TWndw()
{
	delete printer;
}


///////////////////////////////////////////////////////////
// TWndw::Paint()
//
// This function, which overrides TWindow's Paint() member
// function, responds to WM_PAINT messages, which Windows
// sends to a window when it must be repainted.
///////////////////////////////////////////////////////////
void TWndw::Paint(TDC& dc, BOOL, TRect&)
{
	// Draw the 500 rectangles.
	for (int i=0; i<100; ++i)
	{
		// Create the proper color brush for
		// the current rectangle.
		TBrush *brush = new TBrush(recArray[i].color);

		// Select the new brush into the device context.
		dc.SelectObject(*brush);

		// Draw the rectangle.
		dc.Rectangle(recArray[i].x, recArray[i].y,
			recArray[i].x + 20, recArray[i].y +20);

		// Get rid of the new brush.
		delete brush;
	}
}

///////////////////////////////////////////////////////////
// TWndw::CmSetScroll()
//
// This function responds to the CM_SETSCROLL message,
// which is sent to the window when the user selects the
// Scroller menu's Set Scroller command.
///////////////////////////////////////////////////////////
void TWndw::CmSetScroll()
{
	// Create the Scroll-Values dialog box.
	TDialog *dialog = new TDlg(this, SCROLLDLG);

	// Copy the current scroller settings into the
	// Scroll-Values dialog box's transfer buffer.
	char s[10];
	itoa(Scroller->XUnit, s, 10);
	strcpy(transBuf.unitsEdit, s);
	itoa(Scroller->XLine, s, 10);
	strcpy(transBuf.lineEdit, s);
	ltoa(Scroller->XRange, s, 10);
	strcpy(transBuf.hRangeEdit, s);
	ltoa(Scroller->YRange, s, 10);
	strcpy(transBuf.vRangeEdit, s);

	// Execute the dialog box.
	int result = dialog->Execute();

	// If the user exits the dialog box via the
	// OK button, reset the scroller to the new values.
	if (result == IDOK)
		SetScroller();
}

///////////////////////////////////////////////////////////
// TWndw::SetScroller()
//
// This function sets the TScroller object to the new
// values the user entered into the Scroll-Values dialog
// box.
///////////////////////////////////////////////////////////
void TWndw::SetScroller()
{
	// Set new XUnit and YUnit values.
	int units = atoi(transBuf.unitsEdit);
	if (units > 0)
		Scroller->SetUnits(units, units);

	// Set new XLine and YLine values.
	int line = atoi(transBuf.lineEdit);
	if (line > 0)
	{
		Scroller->XLine = line;
		Scroller->YLine = line;
	}

	// Get the new scroller ranges.
	long hRange = atol(transBuf.hRangeEdit);
	long vRange = atol(transBuf.vRangeEdit);

	// Set the new scroller ranges.
	Scroller->SetRange(hRange, vRange);
}


void TWndw::CmPrint()
{
	if (printer) {
		TYourPrintout printout(this, "Test", printer, recArray);
		printout.EnableBanding();
		printer->Print(this, printout, TRUE);
		}
}

///////////////////////////////////////////////////////////
// The TDlg class's implementation.
///////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////
// TDlg::TDlg()
//
// This is the Scroll-Values dialog box's constructor.
///////////////////////////////////////////////////////////
TDlg::TDlg(TWindow *parent, TResId resId) :
	TDialog(parent, resId)
{
	// Create OWL TEdit objects for each edit
	// control in the dialog box.
	new TEdit(this, ID_UNITS,
		sizeof(transBuf.unitsEdit));
	new TEdit(this, ID_LINE,
		sizeof(transBuf.lineEdit));
	new TEdit(this, ID_HRANGE,
		sizeof(transBuf.hRangeEdit));
	new TEdit(this, ID_VRANGE,
		sizeof(transBuf.vRangeEdit));

	// Set the transfer buffer address.
	TransferBuffer = &transBuf;
}


///////////////////////////////////////////////////////////
// The TPrintout class's implementation
///////////////////////////////////////////////////////////
TYourPrintout::TYourPrintout(TWindow* window, const char* title,
									  TPrinter* printer, rec Array[100])
	: TPrintout(title),
	  Data(printer->GetSetup())
{
	for(int i = 0; i <= 100; ++i) {
		coordArray[i].color = Array[i].color;
		coordArray[i].x = Array[i].x;
		coordArray[i].y = Array[i].y;
		}
	parentWindow = window;
	maxX = maxY = 0;
	xPages = xPages = 0;
	xFactor = yFactor = 0;
	nextOrigin.x = nextOrigin.y = 0;
}


TYourPrintout::~TYourPrintout()
{
	//delete nextOrigin;
	//if(nextPage != NULL)
	  //delete nextPage;
}

void TYourPrintout::BeginPrinting()
{
	prevMode = DC->SetMapMode(MM_ISOTROPIC);
	TSize windowSize(0,300);
	DC->SetWindowExt(windowSize, &oldWExt);
	DC->SetViewportExt(PageSize, &oldVExt);
}

void TYourPrintout::EndPrinting()
{
	DC->SetWindowExt(oldWExt);
	DC->SetViewportExt(oldVExt);
	DC->SetMapMode(prevMode);
}
	
BOOL TYourPrintout::HasPage(int pageNumber)
{
	if (pageNumber <= maxPages) {
		cgPageNumber = pageNumber;
		return TRUE;
		}
	else
		return FALSE;
}

void TYourPrintout::PrintPage(int page, TRect& rect, unsigned /*flags*/)
{
	if (page != 1) {
		if (xFactor >= (xPages - 1)) {
			yFactor += 1;
			xFactor = 0;
			}
		else {
			xFactor += 1;
			}
		nextOrigin.x = xFactor * 250;
		nextOrigin.y = yFactor * 300;	
	}

	DC->SetWindowOrg(nextOrigin,&oldWOrg);
   TRect psrect(0,0,PageSize.cx,PageSize.cy);
	DC->IntersectClipRect(psrect);
	DC->DPtoLP(rect, 2);

	sprintf(szBuffer, "Page %i", cgPageNumber);
	DC->TextOut(nextOrigin,szBuffer);
	parentWindow->Paint(*DC, FALSE, rect);

	/*
	scale = 4;
	windowExt.cx = windowExt.cx / scale;
	windowExt.cy = windowExt.cy / scale;
	*/
}

void TYourPrintout::SetPrintParams(TPrintDC* dc, TSize pageSize)
{
	TPrintout::SetPrintParams(dc, pageSize);

	for (int i = 0; i < 100; ++i) {
		if (coordArray[i].x > maxX) {
			maxX = coordArray[i].x;
			}
		if (coordArray[i].y > maxY) {
			maxY = coordArray[i].y;
			}
		}
	xPages = (maxX / 250) + 1;
	yPages = (maxY / 300) + 1;

	maxPages = xPages * yPages;
}

void TYourPrintout::GetDialogInfo(int& minPage, int& maxPage,
											 int& selFromPage, int& selToPage)
{
	minPage = 1;
	//maxPage = pagesToPrint;
	maxPage = 99;
	selFromPage = minPage;
	selToPage = maxPage;

	Data.FromPage = minPage;
	Data.ToPage = maxPage;
}

///////////////////////////////////////////////////////////
// The TApp class's implementation.
///////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////
// TApp::InitMainWindow()
//
// This function creates the application's main window.
///////////////////////////////////////////////////////////
void TApp::InitMainWindow()
{
	TFrameWindow *wndw = new TWndw(0, "Scroll App");
	SetMainWindow(wndw);
}

///////////////////////////////////////////////////////////
// OwlMain()
///////////////////////////////////////////////////////////
int OwlMain(int, char*[])
{
  return TApp().Run();
}

